/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */  

package org.apache.ftpserver.command;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.ftpserver.ftplet.FtpException;
import org.apache.ftpserver.ftplet.FtpReply;
import org.apache.ftpserver.ftplet.FtpRequest;
import org.apache.ftpserver.ftplet.Ftplet;
import org.apache.ftpserver.ftplet.FtpletEnum;
import org.apache.ftpserver.interfaces.Command;
import org.apache.ftpserver.interfaces.CommandFactory;
import org.apache.ftpserver.interfaces.FtpIoSession;
import org.apache.ftpserver.interfaces.FtpServerContext;
import org.apache.ftpserver.util.FtpReplyUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handle SITE command.
 */
public class SITE extends AbstractCommand implements CommandFactory {

    private final Logger LOG = LoggerFactory.getLogger(SITE.class);

    // default command set
    private static final HashMap<String, Command> DEFAULT_COMMAND_MAP = new HashMap<String, Command>(5);

    // user-defined command set
    private HashMap<String, Command> commandMap;
    
    /**
     * Execute command.
     */
    public void execute(FtpIoSession session,
                        FtpServerContext context,
                        FtpRequest request) throws IOException, FtpException {
        
        // call Ftplet.onSite method
        Ftplet ftpletContainer = context.getFtpletContainer();
        FtpletEnum ftpletRet;
        try {
            ftpletRet = ftpletContainer.onSite(session.getFtpletSession(), request);
        } catch(Exception e) {
            LOG.debug("Ftplet container threw exception", e);
            ftpletRet = FtpletEnum.RET_DISCONNECT;
        }
        if(ftpletRet == FtpletEnum.RET_SKIP) {
            return;
        }
        else if(ftpletRet == FtpletEnum.RET_DISCONNECT) {
            session.closeOnFlush().awaitUninterruptibly(10000);
            return;
        }
        
        // get request name
        String argument = request.getArgument();
        if(argument != null) {
            int spaceIndex = argument.indexOf(' ');
            if(spaceIndex != -1) {
                argument = argument.substring(0, spaceIndex);
            }
            argument = argument.toUpperCase();
        }
        
        // no params
        if(argument == null) {
            session.resetState();
            session.write(FtpReplyUtil.translate(session, request, context, FtpReply.REPLY_200_COMMAND_OKAY, "SITE", null));
            return;
        }
        
        // call appropriate command method
        String siteRequest = "SITE_" + argument; 
        Command command = (Command) this.getCommand(siteRequest);
        try {
            if(command != null) {
                command.execute(session, context, request);
            }
            else {
                session.resetState();
                session.write(FtpReplyUtil.translate(session, request, context, FtpReply.REPLY_502_COMMAND_NOT_IMPLEMENTED, "SITE", argument));
            }
        }
        catch(Exception ex) {
            LOG.warn("SITE.execute()", ex);
            session.resetState();
            session.write(FtpReplyUtil.translate(session, request, context, FtpReply.REPLY_500_SYNTAX_ERROR_COMMAND_UNRECOGNIZED, "SITE", null));
        }

    }

    /**
     * Get the installed commands
     * @return The installed commands
     */
    public Map<String, Command> getCommandMap() {
        return commandMap;
    }

    /**
     * Get command. Returns null if not found.
     */
    public Command getCommand(String cmdName) {

        // command null or empty? no-service
        if (cmdName == null || cmdName.equals("")) {
            return null;
        }

        // normalize to upper-case
        cmdName = cmdName.toUpperCase();

        // get the command from the default command map
        Command command = DEFAULT_COMMAND_MAP.get(cmdName);

        // if it was not in the default map, try the user-one
        if (command == null) {
            command = commandMap.get(cmdName);
        }

        // returns command class or null
        return command;
    }

    /**
     *  Adds a command to the SITE command set
     *  
     *  @param cmdName  - String holding the the SITE_xxx name. Where
     *                    xxx = the name. Example: CHMOD
     *  @param cmdClass - class name of the command.
     *                    Example: org.company.command.CHMOD
     * 
     *  note: Should a boolean be returned to indicate success
     *        or failure?
     */
    public void setCommand(String cmdName, String cmdClass) {

        // some quick checking on parameters
        if (cmdName == null || cmdClass == null ||
                cmdName.equals("") || cmdClass.equals("")) {
            return;
        }

        // initialize the user-command-map  if it is null
        if (commandMap == null) {
            commandMap = new HashMap<String, Command>();
        }

        // pre-pend the SITE_
        StringBuffer sb = new StringBuffer();
        
        sb.append("SITE_");
        sb.append( cmdName.toUpperCase() );

        // attempt to load the commmand class
        try {

            commandMap.put(sb.toString(), (Command) Class.forName( cmdClass ).newInstance());

            LOG.info("SITE.setCommand() " + cmdName + " was inserted into command map as: " + sb.toString() + " using class: " + cmdClass);

        } catch (InstantiationException ex) {
        // ignore, just return
            ex.printStackTrace();
        } catch (IllegalAccessException ex) {
        // ignore, just return 
            ex.printStackTrace();
        } catch (ClassNotFoundException ex) {
        // ignore, just return       
            ex.printStackTrace();
        }
    }

    /**
     *  Adds a command to the SITE command set
     *  
     *  @param cmdName  - String holding the the SITE_xxx name. Where
     *                    xxx = the name. Example: CHMOD
     *  @param cmd      - command class object.                         
     */
    public void setCommand(String cmdName, Command cmd) {

        // some quick checking on parameters
        if (cmdName == null || cmdName.equals("")) {
            return;
        }

        // initialize the user-command-map  if it is null
        if (commandMap == null) {
            commandMap = new HashMap<String, Command>();
        }

        // not best use of string, but it works
        cmdName = cmdName.toUpperCase();

        // add the command
        commandMap.put(cmdName, cmd);
    }
    
    // initialize all the SITE command handlers
    static {
        DEFAULT_COMMAND_MAP.put("SITE_DESCUSER", new org.apache.ftpserver.command.SITE_DESCUSER());
        DEFAULT_COMMAND_MAP.put("SITE_HELP",     new org.apache.ftpserver.command.SITE_HELP());
        DEFAULT_COMMAND_MAP.put("SITE_STAT",     new org.apache.ftpserver.command.SITE_STAT());
        DEFAULT_COMMAND_MAP.put("SITE_WHO",      new org.apache.ftpserver.command.SITE_WHO());
        DEFAULT_COMMAND_MAP.put("SITE_ZONE",     new org.apache.ftpserver.command.SITE_ZONE());
    }
}
