package org.apache.ftptest.command;

import org.apache.ftptest.ftplet.LocalFileObject;

import java.io.IOException;

import org.apache.ftpserver.DefaultFtpReply;
import org.apache.ftpserver.command.AbstractCommand;
import org.apache.ftpserver.ftplet.FtpException;
import org.apache.ftpserver.ftplet.FtpReply;
import org.apache.ftpserver.ftplet.FtpRequest;
import org.apache.ftpserver.interfaces.FtpIoSession;
import org.apache.ftpserver.interfaces.FtpServerContext;
import org.apache.ftpserver.util.FtpReplyUtil;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Changes the mode on a user-owned file.
 */
public class SITE_CHMOD extends AbstractCommand {

    private final Logger LOG = LoggerFactory.getLogger(SITE_CHMOD.class);

    /**
     * Execute command.
     */
    public void execute(FtpIoSession session,
            FtpServerContext context,
            FtpRequest request) throws IOException, FtpException {

        LOG.info("SITE_CHMOD called");

        // reset state variables
        session.resetState();

        // should check the string length to make sure client is not
        // trying to ask for some large string aka attempting
        // a buffer overflow-of-sorts, yes this is Java, however ...
        String result = request.getRequestLine().trim();

        LOG.info("SITE_CHMOD arguments: " + result);

        // really interested in the last two arguments, 
        // but need to get all four
        String[] args = result.split(" ", 4);

        // We do not know if the FileObject implementation checks for
        // conditions, best to check here for some minimum requirements:
        // arg[3] = filename, must not be null, must be at least one character
        // arg[2] = mode, must be at least three characters, maximum of four        
        if (args.length != 4 || args[3] == null || args[3].length() < 1) {

            // what in the wide-wide-world-of-sports-is-a-goin'-on-here?
            session.write(FtpReplyUtil.translate(session, request, context,
                    FtpReply.REPLY_501_SYNTAX_ERROR_IN_PARAMETERS_OR_ARGUMENTS,
                    "SITE CHMOD", null));
            return;
        }

        // get the mode command string
        String mode = args[2];

        // todo: out of bounds?  wrong characters? complain        
        if (mode == null) {
            LOG.info("SITE_CHMOD mode value invalid: ");
            session.write(FtpReplyUtil.translate(session, request, context,
                    FtpReply.REPLY_501_SYNTAX_ERROR_IN_PARAMETERS_OR_ARGUMENTS,
                    "SITE CHMOD MODE", null));
            return;
        }

        // get file object
        LocalFileObject file = null;

        try {
            file = (LocalFileObject) session.getFileSystemView().getFileObject(args[3]);
        } catch (Exception ex) {
            LOG.debug("Exception getting file object", ex);
        }

        // complain if file name is missing        
        if (file == null) {
            LOG.info("SITE_CHMOD missing file name: " + args[3]);
            session.write(FtpReplyUtil.translate(session, request, context,
                    FtpReply.REPLY_550_REQUESTED_ACTION_NOT_TAKEN,
                    "SITE CHMOD.missing", args[3]));
            return;
        }

        // get the full file name
        String fileName = file.getFullName();

        // see if the file exists
        if (!file.doesExist()) {
            LOG.info("SITE_CHMOD file does not exist: " + fileName);
            session.write(FtpReplyUtil.translate(session, request, context,
                    FtpReply.REPLY_550_REQUESTED_ACTION_NOT_TAKEN,
                    "SITE CHMOD.missing", fileName));
        }

        // get permission
        if (!file.hasWritePermission()) {
            LOG.info("SITE_CHMOD write permission denied for: " + fileName);
            session.write(FtpReplyUtil.translate(session, request, context,
                    FtpReply.REPLY_550_REQUESTED_ACTION_NOT_TAKEN,
                    "SITE CHMOD.permission", fileName));
            return;
        }

        // try to set the mode
        try {

            int ret = file.chmod(mode);

            // see if the result was successful
            if (ret == 0) {
                session.write(new DefaultFtpReply(
                        FtpReply.REPLY_200_COMMAND_OKAY, result));
            } else {                
                LOG.info("SITE_CHMOD FileObject operation failed for: " + fileName + ", chmod return code: " + ret);
                session.write(FtpReplyUtil.translate(session, request, context,
                        FtpReply.REPLY_501_SYNTAX_ERROR_IN_PARAMETERS_OR_ARGUMENTS,
                        "SITE CHMOD.invalid", args[2]));
            }
        } catch (Exception ex) {
            LOG.info("SITE_CHMOD FileObject operation failed for: " + fileName);
            session.write(FtpReplyUtil.translate(session, request, context,
                    FtpReply.REPLY_501_SYNTAX_ERROR_IN_PARAMETERS_OR_ARGUMENTS,
                    "SITE CHMOD.invalid", args[2]));
        }
    }
}
