/*
 * Common code
 */

package org.apache.ftptest.filesystem;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

/**
 * Code copied/originally from:
 * 
 * http://www.javaworld.com/javaworld/jw-12-2000/jw-1229-traps.html
 * 
 * When Runtime.exec() won't: Navigate yourself around pitfalls related 
 * to the Runtime.exec() method
 * 
 * @author Michael C. Daconta <original author>
 * @author Andy Thomson

 */
public abstract class  OSCommand {
    
    class StreamGobbler extends Thread {

        InputStream is;
        String type;

        StreamGobbler(InputStream is, String type) {
            this.is = is;
            this.type = type;
        }

        @Override
        public void run() {
            InputStreamReader isr = null;
            try {
                isr = new InputStreamReader(is);
                BufferedReader br = new BufferedReader(isr);
                String line = null;
                while ((line = br.readLine()) != null) {
                    System.out.println(type + ">" + line);
                }
            } catch (IOException ex) {
                ex.printStackTrace();
            } finally {
                try {
                    isr.close();
                } catch (IOException ex) {
                    ex.printStackTrace();
                }
            }
        }
    }

    /**
     * Run's an OS command that returns a integer result
     * 
     * @param cmdline
     * @return int, result from the OS command
     */
    public int runCommand(String cmdline) {

        BufferedReader input = null;
        int exitVal = -1;

        try {
            String osName = System.getProperty("os.name");
            String[] cmd = new String[3];

            if (osName.equals("Windows 95")) {
                cmd[0] = "command.com";
                cmd[1] = "/C";
                cmd[2] = cmdline;
            } else if (osName.contains("Windows")) {
                cmd[0] = "cmd.exe";
                cmd[1] = "/C";
                cmd[2] = cmdline;
            } else if (osName.equals("Linux")) {
                cmd[0] = "sh";
                cmd[1] = "-c";
                cmd[2] = cmdline;
            }

            Runtime rt = Runtime.getRuntime();

            // run the command
            Process proc = rt.exec(cmd);

            // handle error messages
            StreamGobbler errorGobbler = new StreamGobbler(proc.getErrorStream(), "ERROR");

            // run the handlers
            errorGobbler.start();

            // get the output
            input = new BufferedReader(new InputStreamReader(proc.getInputStream()));

            // wait until complete
            exitVal = proc.waitFor();

            input.close();

        } catch (Throwable t) {
            t.printStackTrace();
        } finally {
            if (input != null) {
                try {
                    input.close();
                } catch (Exception ex) {
                    ex.printStackTrace();
                }
            }
        }

        return exitVal;
    }
    
    /**
     * Run's an OS command that returns a string result
     * 
     * @param cmdline
     * @param Cnt, int representing the desired result line from the 
     *             OS command.  Example 'df -b512 /' will return two
     *             lines:
     * 
     * Filesystem  512B-blocks  Used      Available  Use% Mounted on
     * /           942098912    118319424 775923480  14% /
     * 
     *             Cnt =2 would result in only the second line being
     *             returned.
     * 
     *             Cnt =0 returns all the lines
     *
     * @return String, string result from the OS command, multiple lines
     *             are concatenated together using a ':' as the line 
     *             separator.
     */
    public String runCommand(String cmdline, int Cnt) {

        StringBuffer resultString = new StringBuffer(80);
        BufferedReader input = null;
        String line = null;
        int lineCnt;

        try {
            String osName = System.getProperty("os.name");
            String[] cmd = new String[3];

            if (osName.equals("Windows 95")) {
                cmd[0] = "command.com";
                cmd[1] = "/C";
                cmd[2] = cmdline;
            } else if (osName.contains("Windows")) {
                cmd[0] = "cmd.exe";
                cmd[1] = "/C";
                cmd[2] = cmdline;
            } else if (osName.equals("Linux")) {
                cmd[0] = "sh";
                cmd[1] = "-c";
                cmd[2] = cmdline;
            }

            Runtime rt = Runtime.getRuntime();

            // run the command
            Process proc = rt.exec(cmd);

            // handle error messages
            StreamGobbler errorGobbler = new StreamGobbler(proc.getErrorStream(), "ERROR");

            // run the handlers
            errorGobbler.start();

            // get the output
            input = new BufferedReader(new InputStreamReader(proc.getInputStream()));

            // wait until complete
            int exitVal = proc.waitFor();

            if (exitVal == 0) {
                lineCnt = 0;

                // get the results
                while ((line = input.readLine()) != null) {

                    if (Cnt == 0 && lineCnt > 0) {
                        resultString.append(":");
                    }

                    lineCnt++;

                    if (Cnt > 0 && lineCnt == Cnt) {
                        resultString.append(line);
                    } else if (Cnt == 0) {
                        resultString.append(line);
                    }
                }
            }

            input.close();

        } catch (Throwable t) {
            t.printStackTrace();
        } finally {
            if (input != null) {
                try {
                    input.close();
                } catch (Exception ex) {
                    ex.printStackTrace();
                }
            }
        }

        return resultString.toString();
    }   
}
