# --
# Kernel/Modules/Report.pm - Reporting tool
# Copyright (C) 2001-2004 Martin Edenhofer <martin+code@otrs.org>
# --
# $Id: $
# --
# This software comes with ABSOLUTELY NO WARRANTY. For details, see
# the enclosed file COPYING for license information (GPL). If you
# did not receive this file, see http://www.gnu.org/licenses/gpl.txt.
# --

package Kernel::Modules::Report;

use strict;
use Kernel::System::CustomerUser;
use Kernel::System::Priority;
use Kernel::System::State;

use Date::Pcalc qw(Today_and_Now Days_in_Month Day_of_Week Day_of_Week_Abbreviation);

use Date::Manip;
# use Statistics::Descriptive;

BEGIN {
  &Date_Init("WorkDayBeg=8:00","WorkDayEnd=16:00");
}

use vars qw($VERSION);
$VERSION = '$Revision: 1.59 $';
$VERSION =~ s/^\$.*:\W(.*)\W.+?$/$1/;

# --
sub new {
    my $Type = shift;
    my %Param = @_;

    # allocate new hash for object
    my $Self = {};
    bless ($Self, $Type);

    foreach (keys %Param) {
        $Self->{$_} = $Param{$_};
    }

    # check needed Opjects
    foreach (qw(ParamObject DBObject TicketObject LayoutObject LogObject ConfigObject)) {
        die "Got no $_!" if (!$Self->{$_});
    }
    $Self->{CustomerUserObject} = Kernel::System::CustomerUser->new(%Param);
    $Self->{PriorityObject} = Kernel::System::Priority->new(%Param);
    $Self->{StateObject} = Kernel::System::State->new(%Param);

    # if we need to do a report query on an external mirror database
    # (using code from AgentUtils)
    if ($Self->{ConfigObject}->Get('AgentUtil::DB::DSN')) {
        my $ExtraDatabaseObject = Kernel::System::DB->new(
            LogObject => $Param{LogObject},
            ConfigObject => $Param{ConfigObject},
            DatabaseDSN => $Self->{ConfigObject}->Get('AgentUtil::DB::DSN'),
            DatabaseUser => $Self->{ConfigObject}->Get('AgentUtil::DB::User'),
            DatabasePw => $Self->{ConfigObject}->Get('AgentUtil::DB::Password'),
        ) || die $DBI::errstr;
        $Self->{TicketObjectReport} = Kernel::System::Ticket->new(
            %Param,
            DBObject => $ExtraDatabaseObject,
        );
    }
    else {
        $Self->{TicketObjectReport} = $Self->{TicketObject};
    }

    return $Self;
}
# --
sub Run {
    my $Self = shift;
    my %Param = @_;
    my $Output;
    # get confid data
    $Self->{StartHit} = $Self->{ParamObject}->GetParam(Param => 'StartHit') || 1;
    $Self->{QueryLimit} = $Self->{ConfigObject}->Get('QueryLimit') || 200;
    $Self->{QueryPageShown} = $Self->{ConfigObject}->Get('QueryPageShown') || 40;
    $Self->{SortBy} = $Self->{ParamObject}->GetParam(Param => 'SortBy') || 'Age';
    $Self->{Order} = $Self->{ParamObject}->GetParam(Param => 'Order') || 'Down';
    $Self->{Profile} = $Self->{ParamObject}->GetParam(Param => 'Profile') || '';
    $Self->{SaveProfile} = $Self->{ParamObject}->GetParam(Param => 'SaveProfile') || '';
    $Self->{TakeLastQuery} = $Self->{ParamObject}->GetParam(Param => 'TakeLastQuery') || '';
    $Self->{SelectTemplate} = $Self->{ParamObject}->GetParam(Param => 'SelectTemplate') || '';
    $Self->{EraseTemplate} = $Self->{ParamObject}->GetParam(Param => 'EraseTemplate') || '';
    # check request
    if ($Self->{ParamObject}->GetParam(Param => 'QueryTemplate') && $Self->{Profile}) {
        return $Self->{LayoutObject}->Redirect(OP => "Action=Report&Subaction=Query&TakeLastQuery=1&SaveProfile=1&Profile=$Self->{Profile}");
    }
    # get signle params
    my %GetParam = ();
    foreach (qw(TicketNumber From To Cc Subject Body CustomerID CustomerUserLogin
      Agent ResultForm TimeQueryType
      TicketCreateTimePointFormat TicketCreateTimePoint
      TicketCreateTimePointStart
      TicketCreateTimeStart TicketCreateTimeStartDay TicketCreateTimeStartMonth
      TicketCreateTimeStartYear
      TicketCreateTimeStop TicketCreateTimeStopDay TicketCreateTimeStopMonth
      TicketCreateTimeStopYear
    )) {
        # load profiles string params (press load profile)
        if (($Self->{Subaction} eq 'LoadProfile' && $Self->{Profile}) || $Self->{TakeLastQuery}) {
            my $SQL = "SELECT profile_value FROM report_profile".
              " WHERE ".
              " profile_name = '".$Self->{DBObject}->Quote($Self->{Profile})."' AND ".
              " profile_key = '$_' AND ".
              " login = '".$Self->{DBObject}->Quote($Self->{UserLogin})."'";
            $Self->{DBObject}->Prepare(SQL => $SQL);
            while (my @Row = $Self->{DBObject}->FetchrowArray()) {
                $GetParam{$_} = $Row[0];
            }
        }
        # get query string params (get submitted params)
        else {
            $GetParam{$_} = $Self->{ParamObject}->GetParam(Param => $_);
            # remove white space on the start and end
            if ($GetParam{$_}) {
                $GetParam{$_} =~ s/\s+$//g;
                $GetParam{$_} =~ s/^\s+//g;
            }
        }
    }
    # get array params
    foreach (qw(StateIDs StateTypeIDs QueueIDs PriorityIDs UserIDs
      TicketFreeKey1 TicketFreeText1 TicketFreeKey2 TicketFreeText2
      TicketFreeKey3 TicketFreeText3 TicketFreeKey4 TicketFreeText4
      TicketFreeKey5 TicketFreeText5 TicketFreeKey6 TicketFreeText6
      TicketFreeKey7 TicketFreeText7 TicketFreeKey8 TicketFreeText8)) {
        # load profile array params (press load profile)
        if (($Self->{Subaction} eq 'LoadProfile' && $Self->{Profile}) || $Self->{TakeLastQuery}) {
            my $SQL = "SELECT profile_value FROM report_profile".
              " WHERE ".
              " profile_name = '".$Self->{DBObject}->Quote($Self->{Profile})."' AND ".
              " profile_key = '$_' AND ".
              " login = '".$Self->{DBObject}->Quote($Self->{UserLogin})."'";
            $Self->{DBObject}->Prepare(SQL => $SQL);
            my @Array = ();
            while (my @Row = $Self->{DBObject}->FetchrowArray()) {
                push(@{$GetParam{$_}}, $Row[0]);
            }
        }
        # get query array params (get submitted params)
        else {
            my @Array = $Self->{ParamObject}->GetArray(Param => $_);
            if (@Array) {
                $GetParam{$_} = \@Array;
            }
        }
    }
    # get time option
    if (!$GetParam{TimeQueryType}) {
        $GetParam{'TimeQueryType::None'} = 'checked';
    }
    elsif ($GetParam{TimeQueryType} eq 'TimePoint') {
        $GetParam{'TimeQueryType::TimePoint'} = 'checked';
    }
    elsif ($GetParam{TimeQueryType} eq 'TimeSlot') {
        $GetParam{'TimeQueryType::TimeSlot'} = 'checked';
    }
    # set result form env
    if (!$GetParam{ResultForm}) {
        $GetParam{ResultForm} = '';
    }
    if ($GetParam{ResultForm} eq 'Print' || $GetParam{ResultForm} eq 'CSV') {
        $Self->{QueryPageShown} = $Self->{QueryLimit};
    }
    # show result site
    if ($Self->{Subaction} eq 'Query' && !$Self->{EraseTemplate}) {
        # fill up profile name (e.g. with last-query)
        if (!$Self->{Profile} || !$Self->{SaveProfile}) {
            $Self->{Profile} = 'last-query';
        }
        # store last queue screen
        my $URL = "Action=Report&Subaction=Query&Profile=$Self->{Profile}&SortBy=$Self->{SortBy}&Order=$Self->{Order}&TakeLastQuery=1&StartHit=$Self->{StartHit}";
        $Self->{SessionObject}->UpdateSessionID(
            SessionID => $Self->{SessionID},
            Key => 'LastScreenQueue',
            Value => $URL,
        );
        $Self->{SessionObject}->UpdateSessionID(
            SessionID => $Self->{SessionID},
            Key => 'LastScreen',
            Value => $URL,
        );
        # save query profile (under last-query or real profile name)
        $Self->{SaveProfile} = 1;
        # remember last query values
        if ($Self->{SaveProfile} && $Self->{Profile}) {
            # remove old profile stuff
            my $SQL = "DELETE FROM report_profile WHERE ".
                  "profile_name = '".$Self->{DBObject}->Quote($Self->{Profile}).
                  "' AND login = '".$Self->{DBObject}->Quote($Self->{UserLogin})."'";
            $Self->{DBObject}->Do(SQL => $SQL);
            # insert new profile params
            foreach my $Key (keys %GetParam) {
              if ($GetParam{$Key}) {
                if (ref($GetParam{$Key}) eq 'ARRAY') {
                    foreach (@{$GetParam{$Key}}) {
                      my $SQL = "INSERT INTO report_profile (login, profile_name, ".
                        "profile_key, profile_value) VALUES ".
                        " ('".$Self->{DBObject}->Quote($Self->{UserLogin})."', '".
                        $Self->{DBObject}->Quote($Self->{Profile})."', '$Key', '".
                        $Self->{DBObject}->Quote($_)."')";
                      $Self->{DBObject}->Do(SQL => $SQL);
                    }
                }
                else {
                    my $SQL = "INSERT INTO report_profile (login, profile_name, ".
                      "profile_key, profile_value) VALUES ".
                      " ('".$Self->{DBObject}->Quote($Self->{UserLogin})."', '".
                        $Self->{DBObject}->Quote($Self->{Profile})."', '$Key', '".
                        $Self->{DBObject}->Quote($GetParam{$Key})."')";
                    $Self->{DBObject}->Do(SQL => $SQL);
                }
              }
            }
        }

#        foreach (qw(email-notification-int email-notification-ext)) {

        # get time settings
        if (!$GetParam{TimeQueryType}) {
            # do noting ont time stuff
        }
        elsif ($GetParam{TimeQueryType} eq 'TimeSlot') {
          foreach (qw(Month Day)) {
              if ($GetParam{"TicketCreateTimeStart$_"} <= 9) {
                  $GetParam{"TicketCreateTimeStart$_"} = '0'.$GetParam{"TicketCreateTimeStart$_"};
              }
          }
          foreach (qw(Month Day)) {
              if ($GetParam{"TicketCreateTimeStop$_"} <= 9) {
                  $GetParam{"TicketCreateTimeStop$_"} = '0'.$GetParam{"TicketCreateTimeStop$_"};
              }
          }
          if ($GetParam{TicketCreateTimeStartDay} && $GetParam{TicketCreateTimeStartMonth} && $GetParam{TicketCreateTimeStartYear}) {
              $GetParam{TicketCreateTimeNewerDate} = $GetParam{TicketCreateTimeStartYear}.
                '-'.$GetParam{TicketCreateTimeStartMonth}.
                '-'.$GetParam{TicketCreateTimeStartDay}.
                ' 00:00:01';
          }
          if ($GetParam{TicketCreateTimeStopDay} && $GetParam{TicketCreateTimeStopMonth} && $GetParam{TicketCreateTimeStopYear}) {
              $GetParam{TicketCreateTimeOlderDate} = $GetParam{TicketCreateTimeStopYear}.
                '-'.$GetParam{TicketCreateTimeStopMonth}.
                '-'.$GetParam{TicketCreateTimeStopDay}.
                ' 23:59:59';
          }
        }
        elsif ($GetParam{TimeQueryType} eq 'TimePoint') {
          if ($GetParam{TicketCreateTimePoint} && $GetParam{TicketCreateTimePointStart} && $GetParam{TicketCreateTimePointFormat}) {
            my $Time = 0;
            if ($GetParam{TicketCreateTimePointFormat} eq 'day') {
                $Time = $GetParam{TicketCreateTimePoint} * 60 * 24;
            }
            elsif ($GetParam{TicketCreateTimePointFormat} eq 'week') {
                $Time = $GetParam{TicketCreateTimePoint} * 60 * 24 * 7;
            }
            elsif ($GetParam{TicketCreateTimePointFormat} eq 'month') {
                $Time = $GetParam{TicketCreateTimePoint} * 60 * 24 * 30;
            }
            elsif ($GetParam{TicketCreateTimePointFormat} eq 'year') {
                $Time = $GetParam{TicketCreateTimePoint} * 60 * 24 * 356;
            }
            if ($GetParam{TicketCreateTimePointStart} eq 'Before') {
                $GetParam{TicketCreateTimeOlderMinutes} = $Time;
            }
            else {
                $GetParam{TicketCreateTimeNewerMinutes} = $Time;
            }
          }
        }
        # focus of "From To Cc Subject Body"
        foreach (qw(From To Cc Subject Body)) {
            if (defined($GetParam{$_}) && $GetParam{$_} ne '') {
                $GetParam{$_} = "*$GetParam{$_}*";
            }
        }
        # perform ticket query
        my $Counter = 0;
        my @ViewableIDs = $Self->{TicketObjectReport}->TicketSearch(
            Result => 'ARRAY',
            SortBy => $Self->{SortBy},
            OrderBy => $Self->{Order},
            Limit => $Self->{QueryLimit},
            UserID => $Self->{UserID},
            %GetParam,
        );

	# prepare interesting history types
	my (%HistoryTypes, @HistoryTypes);
	if (@ViewableIDs) {
	  %HistoryTypes = $Self->GetHistoryTypes(
	    qw(NewTicket FollowUp FollowUpAgent SendAnswer
	       EmailAgent EmailCustomer PhoneCallAgent PhoneCallCustomer
	       Move Remove StateUpdate WebRequestCustomer));
	  @HistoryTypes = values(%HistoryTypes);
	}

	# DEBUG
	unlink ("/opt/otrs/var/tmp/xreport.log");
	
        my @CSVHead = ();
        my @CSVData = ();
        foreach my $TicketID (@ViewableIDs) {
          $Counter++;
          # build query result
          if ($Counter >= $Self->{StartHit} && $Counter < ($Self->{QueryPageShown}+$Self->{StartHit}) ) {
            # get first article data
            my %Data = $Self->{TicketObjectReport}->ArticleFirstArticle(TicketID => $TicketID);
            foreach (qw(From To Cc Subject)) {
                $Data{$_} = $Self->{LayoutObject}->{LanguageObject}->CharsetConvert(
                    Text => $Data{$_},
                    From => $Data{ContentCharset},
                ) || '-';
            }
            # customer info
            my %CustomerData = ();
            if ($Data{CustomerUserID}) {
                %CustomerData = $Self->{CustomerUserObject}->CustomerUserDataGet(
                    User => $Data{CustomerUserID},
                );
            }
            elsif ($Data{CustomerID}) {
                %CustomerData = $Self->{CustomerUserObject}->CustomerUserDataGet(
                    CustomerID => $Data{CustomerID},
                );
            }
            # customer info (customer name)
            if ($CustomerData{UserLogin}) {
                $Data{CustomerName} = $Self->{CustomerUserObject}->CustomerName(
                    UserLogin => $CustomerData{UserLogin},
                );
            }
            # user info
            my %UserInfo = $Self->{UserObject}->GetUserData(
                User => $Data{Owner},
                Cached => 1
            );
            # get age
            $Data{Age} = $Self->{LayoutObject}->CustomerAge(Age => $Data{Age}, Space => ' ');
            # customer info string
            $UserInfo{CustomerName} = '('.$UserInfo{CustomerName}.')' if ($UserInfo{CustomerName});

	    # get history data
	    $GetParam{"TicketHistory"} =
	      $Self->GetHistoryRecords(
		 TicketID => $TicketID,
		 HistoryTypes => \@HistoryTypes,
	      );

	    if (my $TimingStats = TicketTimes($GetParam{"TicketHistory"})) {
	      foreach (keys(%$TimingStats)) {
		$Data{$_} = $TimingStats->{$_};
	      }
	    }

	    if (1) { # debug logging
	      require Data::Dumper;
	      import Data::Dumper;
	      if (open (T, ">>/opt/otrs/var/tmp/xreport.log")) {
		print T "GetParam:\n",Dumper(\%GetParam),
		  "\nData:\n",Dumper(\%Data),
		    "\nUserInfo:\n",Dumper(\%UserInfo),
		      "\nCustomerData:\n",Dumper(\%CustomerData),
			"\n";
		close(T);
	      }
	    }
	    
            # generate ticket result
            if ($GetParam{ResultForm} eq 'Preview') {
                # check if just a only html email
                if (my $MimeTypeText = $Self->{LayoutObject}->CheckMimeType(
                    %Data,
                    Action => 'AgentZoom',
                )) {
                    $Data{TextNote} = $MimeTypeText;
                    $Data{Body} = '';
                }
                else {
                    # charset convert
                    $Data{Body} = $Self->{LayoutObject}->{LanguageObject}->CharsetConvert(
                        Text => $Data{Body},
                        From => $Data{ContentCharset},
                    );
                    # do some text quoting
                    $Data{Body} = $Self->{LayoutObject}->Ascii2Html(
                        NewLine => $Self->{ConfigObject}->Get('ViewableTicketNewLine') || 85,
                        Text => $Data{Body},
                        VMax => $Self->{ConfigObject}->Get('ViewableTicketLinesByQuery') || 15,
                        StripEmptyLines => 1,
                        HTMLResultMode => 1,
                    );
                    # do charset check
                    if (my $CharsetText = $Self->{LayoutObject}->CheckCharset(
                        Action => 'AgentZoom',
                        ContentCharset => $Data{ContentCharset},
                        TicketID => $Data{TicketID},
                        ArticleID => $Data{ArticleID} )) {
                        $Data{TextNote} = $CharsetText;
                    }
                }
                # customer info string
                $UserInfo{CustomerTable} = $Self->{LayoutObject}->AgentCustomerViewTable(
                    Data => \%CustomerData,
                    Max => $Self->{ConfigObject}->Get('ShowCustomerInfoQueueMaxSize'),
                );
                # do some html highlighting
                my $HighlightStart = '<font color="orange"><b><i>';
                my $HighlightEnd = '</i></b></font>';
                if (%GetParam) {
                    foreach (qw(Body From To Subject)) {
                        if ($GetParam{$_}) {
                            $GetParam{$_} =~ s/(\*|\%)//g;
                            my @Parts = split('%', $GetParam{$_});
                            if ($Data{$_}) {
                                foreach my $Part (@Parts) {
                                    $Data{$_} =~ s/($Part)/$HighlightStart$1$HighlightEnd/gi;
                                }
                            }
                        }
                    }
                }
                foreach (qw(From To Subject)) {
                    if (!$GetParam{$_}) {
                        $Data{$_} = $Self->{LayoutObject}->Ascii2Html(Text => $Data{$_}, Max => 80);
                    }
                }
                # add ticket block
                $Self->{LayoutObject}->Block(
                    Name => 'Record',
                    Data => {
                        %Data,
                        %UserInfo,
                    },
                );
            }
            elsif ($GetParam{ResultForm} eq 'Print') {
                # add table block
                $Self->{LayoutObject}->Block(
                    Name => 'Record',
                    Data => {
                        %Data,
                        %UserInfo,
                    },
                );
            }
            elsif ($GetParam{ResultForm} eq 'CSV') {
                # merge row data
                my %Info = (
                    %Data,
                    %UserInfo,
                    AccountedTime => $Self->{TicketObjectReport}->TicketAccountedTimeGet(
		       TicketID => $TicketID
		    ) || 0,
                );
                # csv quote
                if (!@CSVHead) {
                    @CSVHead = @{$Self->{ConfigObject}->Get('ReportCSVData')};
                }
                my @Data = ();
                foreach (@CSVHead) {
		    $Info{$_} = "'".$Info{$_} if /TicketNumber/; # apostrophe for Excel
                    push (@Data, $Info{$_});
                }
                push (@CSVData, \@Data);
            }
            else {
                # Condense down the subject
                my $TicketHook = $Self->{ConfigObject}->Get('TicketHook');
                my $Subject = $Data{Subject};
                $Subject =~ s/^..: ?//i;
                $Subject =~ s/\[${TicketHook}:\s*\d+\]//;
                $Subject =~ s/^..: ?//i;
                # add table block
                if (!$Data{Answered}) {
                  $Self->{LayoutObject}->Block(
                    Name => 'Record',
                    Data => {
                        StartFont => '<font color="red">',
                        StopFont => '</font>',
                        %Data,
                        Subject => $Subject,
                        %UserInfo,
                    },
                  );
                }
                else {
                  $Self->{LayoutObject}->Block(
                    Name => 'Record',
                    Data => {
                        %Data,
                        Subject => $Subject,
                        %UserInfo,
                    },
                  );
                }
            }
          }
        }
        # start html page
        my $Output = $Self->{LayoutObject}->Header(Area => 'Agent', Title => 'Query');
        my %LockedData = $Self->{TicketObject}->GetLockedCount(UserID => $Self->{UserID});
        $Output .= $Self->{LayoutObject}->NavigationBar(LockData => \%LockedData);

        # build query navigation bar
        my %PageNav = $Self->{LayoutObject}->PageNavBar(
            Limit => $Self->{QueryLimit},
            StartHit => $Self->{StartHit},
            PageShown => $Self->{QueryPageShown},
            AllHits => $Counter,
            Action => "Action=Report&Subaction=Query",
            Link => "Profile=$Self->{Profile}&SortBy=$Self->{SortBy}&Order=$Self->{Order}&TakeLastQuery=1&",
        );
        # build shown ticket
        if ($GetParam{ResultForm} eq 'Preview') {
            $Output .= $Self->{LayoutObject}->Output(
                TemplateFile => 'ReportQueryResult',
                Data => { %Param, %PageNav, Profile => $Self->{Profile}, },
            );
        }
        elsif ($GetParam{ResultForm} eq 'Print') {
            $Output = $Self->{LayoutObject}->PrintHeader(Area => 'Agent', Title => 'Result', Width => 800);
            if (@ViewableIDs == $Self->{QueryLimit}) {
                $Param{Warning} = '$Text{"Reached max. count of %s query hits!", "'.$Self->{QueryLimit}.'"}';
            }
            $Output .= $Self->{LayoutObject}->Output(
                TemplateFile => 'ReportQueryResultPrint',
                Data => \%Param,
            );
            # add footer
            $Output .= $Self->{LayoutObject}->PrintFooter();
            # return output
            return $Output;
        }
        elsif ($GetParam{ResultForm} eq 'CSV') {
            my $CSV = $Self->{LayoutObject}->OutputCSV(
                Head => \@CSVHead,
                Data => \@CSVData,
            );
            # return csv to download
            my $CSVFile = 'report';
            my ($s,$m,$h, $D,$M,$Y, $wd,$yd,$dst) = localtime(time);
            $Y = $Y+1900;
            $M++;
            $M = sprintf("%02d", $M);
            $D = sprintf("%02d", $D);
            $h = sprintf("%02d", $h);
            $m = sprintf("%02d", $m);
            return $Self->{LayoutObject}->Attachment(
                Filename => $CSVFile."_"."$Y-$M-$D"."_"."$h-$m.csv",
                ContentType => "text/csv",
                Content => $CSV,
            );
        }
        else {
            $Output .= $Self->{LayoutObject}->Output(
                TemplateFile => 'ReportQueryResultShort',
                Data => { %Param, %PageNav, Profile => $Self->{Profile}, },
            );
        }
        # build footer
        $Output .= $Self->{LayoutObject}->Footer();
        return $Output;
    }
    # empty query site
    else {
        # delete profile
        if ($Self->{EraseTemplate} && $Self->{Profile}) {
            $Self->{DBObject}->Do(
                SQL => "DELETE FROM report_profile WHERE ".
                  "profile_name = '".$Self->{DBObject}->Quote($Self->{Profile}).
                  "' AND login = '".$Self->{DBObject}->Quote($Self->{UserLogin})."'",
            );
            %GetParam = ();
            $Self->{Profile} = '';
        }
        # set profile to zero
        elsif (!$Self->{SelectTemplate}) {
#            $Self->{Profile} = '';
        }
        # generate query mask
        my $Output = $Self->{LayoutObject}->Header(Area => 'Agent', Title => 'Query');
        my %LockedData = $Self->{TicketObject}->GetLockedCount(UserID => $Self->{UserID});
        # get free text config options
        my %TicketFreeText = ();
        foreach (1..8) {
            $TicketFreeText{"TicketFreeKey$_"} = $Self->{TicketObject}->TicketFreeTextGet(
                Type => "TicketFreeKey$_",
                Action => $Self->{Action},
                UserID => $Self->{UserID},
            );
            $TicketFreeText{"TicketFreeText$_"} = $Self->{TicketObject}->TicketFreeTextGet(
                Type => "TicketFreeText$_",
                Action => $Self->{Action},
                UserID => $Self->{UserID},
            );
        }
        my %TicketFreeTextHTML = $Self->{LayoutObject}->AgentFreeText(
            NullOption => 1,
            Ticket => \%GetParam,
            Config => \%TicketFreeText,
        );
        $Output .= $Self->{LayoutObject}->NavigationBar(LockData => \%LockedData);
        $Output .= $Self->MaskForm(
            %GetParam,
            %TicketFreeTextHTML,
            Profile => $Self->{Profile},
        );
        $Output .= $Self->{LayoutObject}->Footer();
        return $Output;
    }
}
# --
sub MaskForm {
    my $Self = shift;
    my %Param = @_;
    # --
    # get user of own groups
    # --
    my %ShownUsers = $Self->{UserObject}->UserList(
        Type => 'Long',
        Valid => 1,
    );
    $Param{'UserStrg'} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data => \%ShownUsers,
        Name => 'UserIDs',
        Multiple => 1,
        Size => 5,
        SelectedIDRefArray => $Param{UserIDs},
    );
    $Param{'ResultFormStrg'} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data => {
            Preview => 'Preview',
            Normal => 'Normal',
            Print => 'Print',
            CSV => 'CSV',
        },
        Name => 'ResultForm',
        SelectedID => $Param{ResultForm} || 'Normal',
    );
    $Param{'ProfilesStrg'} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data => { '', '-', $Self->{DBObject}->GetTableData(
                      What => 'profile_name, profile_name',
                      Table => 'report_profile',
                      Where => "login = '$Self->{UserLogin}'",
                    ) },
        Name => 'Profile',
        SelectedID => $Param{Profile},
    );
    $Param{'StatesStrg'} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data => { $Self->{StateObject}->StateList(
             UserID => $Self->{UserID},
             Action => $Self->{Action},
             )
        },
        Name => 'StateIDs',
        Multiple => 1,
        Size => 5,
        SelectedIDRefArray => $Param{StateIDs},
    );
    $Param{'QueuesStrg'} = $Self->{LayoutObject}->AgentQueueListOption(
        Data => { $Self->{QueueObject}->GetAllQueues(
            UserID => $Self->{UserID},
            Type => 'ro',
          ),
        },
        Size => 5,
        Multiple => 1,
        Name => 'QueueIDs',
        SelectedIDRefArray => $Param{QueueIDs},
        OnChangeSubmit => 0,
    );
    $Param{'PriotitiesStrg'} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data => { $Self->{PriorityObject}->PriorityList(
            UserID => $Self->{UserID},
            Action => $Self->{Action},
            ),
        },
        Name => 'PriorityIDs',
        Multiple => 1,
        Size => 5,
        SelectedIDRefArray => $Param{PriorityIDs},
    );
    $Param{'TicketCreateTimePoint'} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data => {
            1 => 1,
            2 => 2,
            3 => 3,
            4 => 4,
            5 => 5,
            6 => 6,
            7 => 7,
            8 => 8,
            9 => 9,
        },
        Name => 'TicketCreateTimePoint',
        SelectedID => $Param{TicketCreateTimePoint},
    );
    $Param{'TicketCreateTimePointStart'} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data => {
            'Last' => 'last',
            'Before' => 'before',
        },
        Name => 'TicketCreateTimePointStart',
        SelectedID => $Param{TicketCreateTimePointStart} || 'Last',
    );
    $Param{'TicketCreateTimePointFormat'} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data => {
            day => 'day(s)',
            week => 'week(s)',
            month => 'month(s)',
            year => 'year(s)',
        },
        Name => 'TicketCreateTimePointFormat',
        SelectedID => $Param{TicketCreateTimePointFormat},
    );
    $Param{TicketCreateTimeStart} = $Self->{LayoutObject}->BuildDateSelection(
        %Param,
        Prefix => 'TicketCreateTimeStart',
        Format => 'DateInputFormat',
        DiffTime => -((60*60*24)*30),
    );
    $Param{TicketCreateTimeStop} = $Self->{LayoutObject}->BuildDateSelection(
        %Param,
        Prefix => 'TicketCreateTimeStop',
        Format => 'DateInputFormat',
    );
    # html query mask output
    my $Output = $Self->{LayoutObject}->Output(
        TemplateFile => 'ReportQuery',
        Data => \%Param,
    );
    return $Output;
}
# --
sub GetHistoryTypes {
  my $Self = shift;
  my @HistoryNames = @_;

  if (!@HistoryNames) {
    $Self->{LogObject}->Log(Priority => 'error', Message => "Need array of history type names!");
    return;
  }
  my $names = join(',', map { "'$_'" } @HistoryNames);
  
  my $SQL = "SELECT name, id FROM ticket_history_type WHERE name IN ($names)";

  $Self->{DBObject}->Prepare(SQL => $SQL);
  my %HistoryTypes = ();
  my @Row;
  while (my @Row = $Self->{DBObject}->FetchrowArray()) {
    $HistoryTypes{$Row[0]} = $Row[1];
  }
  return %HistoryTypes;
}
# --
sub GetHistoryRecords {
  my $Self = shift;
  my %Param = @_;

  # check needed stuff
  foreach (qw(TicketID)) { # HistoryType HistoryComment
    if (!$Param{$_}) {
      $Self->{LogObject}->Log(Priority => 'error', Message => "Need $_!");
        return;
    }
  }
  
  my $SQL = "SELECT ticket_history.id, history_type_id, ".
    " ticket_history.create_time, ticket_history.create_by, ".
    " ticket_history.name ".
    " FROM ticket, ticket_history ".
    " WHERE ticket.id = ticket_history.ticket_id ".
    " AND ticket.id = $Param{TicketID} ".
    " AND 1>0 ".
    " AND ticket_history.valid_id = 1 ".
    " ORDER BY ticket_history.id ASC";

  if ($Param{HistoryTypes} and ref($Param{HistoryTypes}) eq 'ARRAY') {
    my ($types) = join(',', @{$Param{HistoryTypes}});
    $SQL =~ s/ AND 1>0 / AND history_type_id IN ($types) /;
  } else {
    $SQL =~ s/ AND 1>0 //;
  }
  
  $Self->{DBObject}->Prepare(SQL => $SQL);
  my @Rows = ();
  while (my @Row = $Self->{DBObject}->FetchrowArray()) {
    my %Row =
      ( HistoryID      => $Row[0],
        HistoryType    => $Row[1],
	CreateTime     => $Row[2],
	CreateUserID   => $Row[3],
	HistoryComment => $Row[4],
      );
    push(@Rows, \%Row);
  }
  return \@Rows;
}

# ticket timings
sub TicketTimes {
  my ($th) = @_; # history array

  return -1 unless ($th and ref($th) eq 'ARRAY');

  my ($c, @tm, $err, %Times);
  
  $c = scalar(@$th); # count of history records

  # parse dates for aritmetic use
  for (my $i = 0; $i < $c; $i ++) {
    if ($th->[$i]->{CreateTime}) {
      $tm[$i] = ParseDate($th->[$i]->{CreateTime});
    }
  }

  my ($tm_total_wtm, $tm_total_biz);
  # total time
  $tm_total_wtm = interval_to_min($tm[0], $tm[$c-1], 0);
  $tm_total_biz = interval_to_min($tm[0], $tm[$c-1], 2);
  
  $Times{TotalTime_RM} = $tm_total_wtm;
  $Times{TotalTime_BM} = $tm_total_biz;
  
  return \%Times unless $tm_total_wtm;
  
  my ($state_cur, $state_new, $answered) = (0, 0, 0);
  # 0: message unanswered
  # 1: message answered by agent
  my ($cust_time, $agent_time);
  # waiting period is what we calculate
  for (my $i = 0; $i < $c; $i ++) {
    if ($th->[$i]->{HistoryType}     ==  1 or  # New by web
	$th->[$i]->{HistoryType}     == 29 or  # WebRequest
	$th->[$i]->{HistoryType}     == 12 or  # EmailCustomer
	($th->[$i]->{HistoryType}    ==  5 and # SendAutoFollowUp
	 $th->[$i]->{HistoryComment} !~ /\@srce\.hr/) or
	($th->[$i]->{HistoryType}    ==  2 and # FollowUp (customer)
	 $th->[$i]->{HistoryComment} =~ /^%/)  # - new OTRS
       ) {
      $state_new = 0;
      $cust_time ||= $tm[$i];
    }
    elsif ($th->[$i]->{HistoryType}  == 32 or  # FollowUpAgent
	   $th->[$i]->{HistoryType}  ==  8 or  # SendAnswer
	   ($th->[$i]->{HistoryType} ==  5 and # SendAutoFollowUp
	    $th->[$i]->{HistoryComment} =~ /\@srce\.hr/) or
	   ($th->[$i]->{HistoryType} == 27 and # StateUpdate
	    ($th->[$i]->{HistoryComment} eq '%%new%%open'))
	  ) {
      $state_new = 1;
      $agent_time ||= $tm[$i];
    }
    if ($state_cur == 0 and $state_new == 1) {
      # ticket has been answered
      my ($dur_wtm, $dur_biz);
      $dur_wtm = interval_to_min($cust_time, $agent_time, 0);
      $dur_biz = interval_to_min($cust_time, $agent_time, 2);
      ++ $answered;
      if ($answered == 1) {
	# it's first answer
	$Times{Response_RM} = $dur_wtm;
	$Times{Response_BM} = $dur_biz;
	$Times{TotalWait_RM} = 0;
	$Times{TotalWait_BM} = 0;
      }
      $Times{TotalWait_RM} += $dur_wtm;
      $Times{TotalWait_BM} += $dur_biz;
      $cust_time = undef;
    }
    elsif ($state_cur == 1 and $state_new == 0) {
      # customer sent followup
      $agent_time = undef;
      # we do not count that statistics
    }
    $state_cur = $state_new;
  }

  # subtract the time the ball was on the clien't side
  
  return \%Times;
}

# parse interval string and return minutes
sub interval_to_min {
    my ($tm_1, $tm_2, $mode) = @_;
    my ($sec) = 30; # for rounding
    my ($err, $intv);
    $intv = DateCalc($tm_1, $tm_2, \$err, $mode);
    if ($err) {
      warn "OTRS:Report: DateCalc('$tm_1', '$tm_2') mode: $mode, error: $err\n";
      return;
    }
    my $sign = ($intv =~ /^-/) ? -1 : 1;
    my @f = reverse(split(/:/, $intv));
    $sec += $f[0];
    $sec += $f[1] * 60;
    $sec += $f[2] * 3600;
    if ($mode == 2 and ! $f[4] and $f[3] < 5) {
      # calculate business day as 8 hours
      # if there are no weeks passed
      $sec += $f[3] * 28800;
    } else {
      $sec += $f[3] * 86400;
    }
    $sec += $f[4] * 604800;   # 7 days
    $sec += $f[5] * 2629800;  # 30.4 days
    $sec += $f[6] * 31557600; # 365.25 days
    return int($sign * $sec / 60);
}

1;
