/* 
 * Copyright (c) 2004-2007 QOS.CH
 * 
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to  deal in  the Software without  restriction, including
 * without limitation  the rights to  use, copy, modify,  merge, publish,
 * distribute, and/or sell copies of  the Software, and to permit persons
 * to whom  the Software is furnished  to do so, provided  that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the  Software and  that both  the above  copyright notice(s)  and this
 * permission notice appear in supporting documentation.
 * 
 * THE  SOFTWARE IS  PROVIDED  "AS  IS", WITHOUT  WARRANTY  OF ANY  KIND,
 * EXPRESS OR  IMPLIED, INCLUDING  BUT NOT LIMITED  TO THE  WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR  A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF  THIRD PARTY  RIGHTS. IN  NO EVENT  SHALL THE  COPYRIGHT  HOLDER OR
 * HOLDERS  INCLUDED IN  THIS  NOTICE BE  LIABLE  FOR ANY  CLAIM, OR  ANY
 * SPECIAL INDIRECT  OR CONSEQUENTIAL DAMAGES, OR  ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS  OF USE, DATA OR PROFITS, WHETHER  IN AN ACTION OF
 * CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS  ACTION, ARISING OUT OF  OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * Except as  contained in  this notice, the  name of a  copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

package org.slf4j.impl;

import junit.framework.TestCase;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;
import org.slf4j.SilentPrintStream;


/**
 * Abstract test class for the the SimpleLogger. Is intended to be extended
 * by tests for specific log levels.
 * 
 * @author Erik Dick
 *
 */
public abstract class SimpleLoggerAbstractTest extends TestCase {

  
  SilentPrintStream ps;
  String logLevel;
  
  public SimpleLoggerAbstractTest (String arg0) {
    super(arg0);
    this.logLevel = initiateLogLevel();
  }

  protected void setUp() throws Exception {
    super.setUp();
    
    this.ps = new SilentPrintStream(System.err);

    System.setErr(this.ps);
  }

  protected void tearDown() throws Exception {
    super.tearDown();
    System.setErr(this.ps.getSuperPrintStream());
  }

  protected abstract String initiateLogLevel();
  
  //===== Tests for simple log message output ==================================
  
  public void test1() {

	final String method = "test1";

	final String message = "Hello world";
	final String messageTemplate = message + ".";

	final String reference = " [main] "
                             + this.logLevel
                             + " " + method
                             + " - "
                             + message
                             + ".\n";

    Logger logger = LoggerFactory.getLogger( method );
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
    	logger.trace( messageTemplate );
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
    	logger.debug( messageTemplate );
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
    	logger.info( messageTemplate );
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
    	logger.warn( messageTemplate );
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
    	logger.error( messageTemplate );
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    int idx = guineaPig.indexOf(' ');
    guineaPig = guineaPig.substring(idx);
    
    assertEquals( reference, guineaPig );
  }
  

  public void test1_R1() {

	final String method = "test1_R1";

	final String message = null;
	final String messageTemplate = message;

	final String reference = " [main] "
                             + this.logLevel
                             + " " + method
                             + " - "
                             + message
                             + "\n";

    Logger logger = LoggerFactory.getLogger( method );
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
    	logger.trace( messageTemplate );
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
    	logger.debug( messageTemplate );
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
    	logger.info( messageTemplate );
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
    	logger.warn( messageTemplate );
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
    	logger.error( messageTemplate );
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    int idx = guineaPig.indexOf(' ');
    guineaPig = guineaPig.substring(idx);
    
    assertEquals( reference, guineaPig );
  }
  

  //===== Tests for log message output with single var placeholder =============

  public void test2() {

    final String method = "test2";

    final String message = "Log message";
    final String messageTemplate = message + " {}.";
    final String argument = "world";

    final String reference = " [main] "
                             + this.logLevel
                             + " " + method
                             + " - "
                             + message
                             + " " + argument.toString()
                             + ".\n";
    

    Logger logger = LoggerFactory.getLogger( method );
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
    	logger.trace( messageTemplate, argument );
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
    	logger.debug( messageTemplate, argument );
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
    	logger.info( messageTemplate, argument );
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
    	logger.warn( messageTemplate, argument );
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
    	logger.error( messageTemplate, argument );
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    int idx = guineaPig.indexOf(' ');
    guineaPig = guineaPig.substring(idx);
    
    assertEquals( reference, guineaPig );
  }
  
  public void test2_R1() {

    final String method = "test2_R1";

    final String message = "Log message";
    final String messageTemplate = message + " {}.";
    final Integer argument = null;

    final String reference = " [main] "
                             + this.logLevel
                             + " " + method
                             + " - " + message
                             + " null"
                             + ".\n";
    

    Logger logger = LoggerFactory.getLogger( method );
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
    	logger.trace( messageTemplate, argument );
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
    	logger.debug( messageTemplate, argument );
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
    	logger.info( messageTemplate, argument );
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
    	logger.warn( messageTemplate, argument );
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
    	logger.error( messageTemplate, argument );
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    int idx = guineaPig.indexOf(' ');
    guineaPig = guineaPig.substring(idx);
    
    assertEquals( reference, guineaPig );
  }
  

  //===== Tests for log message output with two var placeholers ================

  public void test3() {

    final String method = "test3";

    final String message = "Log message";
    final String messageTemplate = message + " {} {}.";
    final String argument1 = "world";
    final String argument2 = "universe";

    final String reference = " [main] "
                               + this.logLevel
                               + " " + method
                               + " - " + message
                               + " " + argument1.toString()
                               + " " + argument2.toString()
                               + ".\n";
    

    Logger logger = LoggerFactory.getLogger( method );
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
    	logger.trace( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
    	logger.debug( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
    	logger.info( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
    	logger.warn( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
    	logger.error( messageTemplate, argument1, argument2 );
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    int idx = guineaPig.indexOf(' ');
    guineaPig = guineaPig.substring(idx);
    
    assertEquals( reference, guineaPig );
  }
  
  public void test3_R1() {

    final String method = "test3_R1";

    final String message = "Log message";
    final String messageTemplate = message + " {} {}.";
    final String argument1 = null;
    final String argument2 = "universe";

    final String reference = " [main] "
                               + this.logLevel
                               + " " + method
                               + " - " + message
                               + " null"
                               + " " + argument2.toString()
                               + ".\n";
    

    Logger logger = LoggerFactory.getLogger( method );
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
    	logger.trace( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
    	logger.debug( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
    	logger.info( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
    	logger.warn( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
    	logger.error( messageTemplate, argument1, argument2 );
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    int idx = guineaPig.indexOf(' ');
    guineaPig = guineaPig.substring(idx);
    
    assertEquals( reference, guineaPig );
  }
  
  public void test3_R2() {

    final String method = "test3_R2";

    final String message = "Log message";
    final String messageTemplate = message + " {} {}.";
    final String argument1 = "world";
    final String argument2 = null;

    final String reference = " [main] "
                               + this.logLevel
                               + " " + method
                               + " - " + message
                               + " " + argument1.toString()
                               + " null"
                               + ".\n";
    

    Logger logger = LoggerFactory.getLogger( method );
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
    	logger.trace( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
    	logger.debug( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
    	logger.info( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
    	logger.warn( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
    	logger.error( messageTemplate, argument1, argument2 );
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    int idx = guineaPig.indexOf(' ');
    guineaPig = guineaPig.substring(idx);
    
    assertEquals( reference, guineaPig );
  }
  
  public void test3_R3() {

    final String method = "test3_R3";

    final String message = "Log message";
    final String messageTemplate = message + " {} {}.";
    final Integer argument1 = null;
    final Integer argument2 = null;

    final String reference = " [main] "
                               + this.logLevel
                               + " " + method
                               + " - " + message
                               + " null"
                               + " null"
                               + ".\n";
    

    Logger logger = LoggerFactory.getLogger( method );
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
    	logger.trace( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
    	logger.debug( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
    	logger.info( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
    	logger.warn( messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
    	logger.error( messageTemplate, argument1, argument2 );
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    int idx = guineaPig.indexOf(' ');
    guineaPig = guineaPig.substring(idx);
    
    assertEquals( reference, guineaPig );
  }
  
  //===== Tests for log message output with variable var argument ==============

  public void test4() {

    final String method = "test4";

    final String message = "Log message";
    final String messageTemplate = message + " {} {} {}.";
    final Object[] arguments = new Object[]{	"world",
    											"universe",
    											new Integer(42) };

    final String reference = " [main] "
                               + this.logLevel
                               + " " + method
                               + " - " + message
                               + " " + arguments[0].toString()
                               + " " + arguments[1].toString()
                               + " " + arguments[2].toString()
                               + ".\n";
    

    Logger logger = LoggerFactory.getLogger( method );
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
    	logger.trace( messageTemplate, arguments );
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
    	logger.debug( messageTemplate, arguments );
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
    	logger.info( messageTemplate, arguments );
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
    	logger.warn( messageTemplate, arguments );
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
    	logger.error( messageTemplate, arguments );
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    int idx = guineaPig.indexOf(' ');
    guineaPig = guineaPig.substring(idx);
    
    assertEquals( reference, guineaPig );
  }
  
  public void test4_R1() {

    final String method = "test4_R1";

    final String message = null;
    final String messageTemplate = message + " {} {} {}.";
    final Object[] arguments = new Object[]{	"world",
    											"universe",
    											new Integer(42) };

    final String reference = " [main] "
                               + this.logLevel
                               + " " + method
                               + " - " + message
                               + " " + arguments[0].toString()
                               + " " + arguments[1].toString()
                               + " " + arguments[2].toString()
                               + ".\n";
    

    Logger logger = LoggerFactory.getLogger( method );
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
    	logger.trace( messageTemplate, arguments );
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
    	logger.debug( messageTemplate, arguments );
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
    	logger.info( messageTemplate, arguments );
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
    	logger.warn( messageTemplate, arguments );
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
    	logger.error( messageTemplate, arguments );
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    int idx = guineaPig.indexOf(' ');
    guineaPig = guineaPig.substring(idx);
    
    assertEquals( reference, guineaPig );
  }
  
  // http://bugzilla.slf4j.org/show_bug.cgi?id=78
  public void test4_R2() {

    final String method = "test4_R2";

    final String message = null;
    final String messageTemplate = message + " {} {} {}.";
    final Object[] arguments = null;

    final String reference = " [main] "
                               + this.logLevel
                               + " " + method
                               + " - " + message
                               + " {}"
                               + " {}"
                               + " {}"
                               + ".\n";
    

    Logger logger = LoggerFactory.getLogger( method );
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
    	logger.trace( messageTemplate, arguments );
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
    	logger.debug( messageTemplate, arguments );
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
    	logger.info( messageTemplate, arguments );
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
    	logger.warn( messageTemplate, arguments );
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
    	logger.error( messageTemplate, arguments );
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    int idx = guineaPig.indexOf(' ');
    guineaPig = guineaPig.substring(idx);
    
    assertEquals( reference, guineaPig );
  }
  
  //===== Tests for log message output with exception argument =================

  public void test5() {

    final String message = "Log message";
    final String messageTemplate = message;

    final String method = "test5";

	Exception e = new Exception("This is a test exception.");

	Logger logger = LoggerFactory.getLogger( method );
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
      logger.trace(messageTemplate, e);
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
      logger.debug(messageTemplate, e);
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
      logger.info(messageTemplate, e);
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
      logger.warn(messageTemplate, e);
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
      logger.error(messageTemplate, e);
    }

    this.ps.recordingOff();
    
    String guineaPig = this.ps.getLastRecordedLine();
    System.err.println(guineaPig);
    System.err.flush();
    
  }
  
  public void test5_R1() {

    final String message = null;
    final String messageTemplate = message;

    final String method = "test5_R1";

	Exception e = new Exception("This is a test exception.");

	Logger logger = LoggerFactory.getLogger( method );
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
      logger.trace(messageTemplate, e);
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
      logger.debug(messageTemplate, e);
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
      logger.info(messageTemplate, e);
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
      logger.warn(messageTemplate, e);
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
      logger.error(messageTemplate, e);
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    System.err.println(guineaPig);
    
  }
  
  //===== Tests for log messages with marker ===================================

  public void test6() {

	final String method = "test6";

	final String message = "Log message";
	final String messageTemplate = message + ".";

	final String reference = " [main] "
                             + this.logLevel
                             + " " + method
                             + " - "
                             + message
                             + ".\n";

    Logger logger = LoggerFactory.getLogger( method );
    Marker blue = MarkerFactory.getMarker("BLUE");
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
    	logger.trace( blue, messageTemplate );
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
    	logger.debug( blue, messageTemplate );
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
    	logger.info( blue, messageTemplate );
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
    	logger.warn( blue, messageTemplate );
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
    	logger.error( blue, messageTemplate );
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    int idx = guineaPig.indexOf(' ');
    guineaPig = guineaPig.substring(idx);
    
    assertEquals( reference, guineaPig );
  }

  public void test7() {

    final String method = "test7";

    final String message = "Log message";
    final String messageTemplate = message + " {}.";
    final String argument = "world";

    final String reference = " [main] "
                             + this.logLevel
                             + " " + method
                             + " - "
                             + message
                             + " " + argument.toString()
                             + ".\n";
    

    Logger logger = LoggerFactory.getLogger( method );
    Marker blue = MarkerFactory.getMarker("BLUE");
   
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
    	logger.trace( blue, messageTemplate, argument );
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
    	logger.debug( blue, messageTemplate, argument );
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
    	logger.info( blue, messageTemplate, argument );
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
    	logger.warn( blue, messageTemplate, argument );
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
    	logger.error( blue, messageTemplate, argument );
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    int idx = guineaPig.indexOf(' ');
    guineaPig = guineaPig.substring(idx);
    
    assertEquals( reference, guineaPig );
  }
  
  public void test8() {

    final String method = "test8";

    final String message = "Log message";
    final String messageTemplate = message + " {} {}.";
    final String argument1 = "world";
    final String argument2 = "universe";

    final String reference = " [main] "
                               + this.logLevel
                               + " " + method
                               + " - " + message
                               + " " + argument1.toString()
                               + " " + argument2.toString()
                               + ".\n";
    

    Logger logger = LoggerFactory.getLogger( method );
    Marker blue = MarkerFactory.getMarker("BLUE");
    
    this.ps.recordingOn();
    
    if(SimpleLogger.TRACE_STR.equals(this.logLevel)) {
    	logger.trace( blue, messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.DEBUG_STR.equals(this.logLevel) ){
    	logger.debug( blue, messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.INFO_STR.equals(this.logLevel) ){
    	logger.info( blue, messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.WARN_STR.equals(this.logLevel) ){
    	logger.warn( blue, messageTemplate, argument1, argument2 );
    } else if( SimpleLogger.ERROR_STR.equals(this.logLevel) ){
    	logger.error( blue, messageTemplate, argument1, argument2 );
    }

    this.ps.recordingOff();

    String guineaPig = this.ps.getLastRecordedLine();
    int idx = guineaPig.indexOf(' ');
    guineaPig = guineaPig.substring(idx);
    
    assertEquals( reference, guineaPig );
  }

}
