package com.voipfuture;

import org.apache.wicket.WicketRuntimeException;
import org.apache.wicket.core.util.resource.UrlResourceStream;
import org.apache.wicket.core.util.resource.locator.IResourceStreamLocator;
import org.apache.wicket.util.file.IResourceFinder;
import org.apache.wicket.util.io.IOUtils;
import org.apache.wicket.util.resource.AbstractResourceStream;
import org.apache.wicket.util.resource.IResourceStream;
import org.apache.wicket.util.resource.ResourceStreamNotFoundException;

import javax.servlet.ServletContext;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;

/**
 * {@link IResourceStreamLocator} implementation that locates CSS , Javascripts and images
 * from the webapps top-level directory.
 *
 * @author tobias.gierke@voipfuture.com
 */
public final class CustomResourceFinder implements IResourceFinder
{
    private final ServletContext context;

    public CustomResourceFinder(ServletContext context) {
        this.context = context;
    }

    @Override
    public IResourceStream find(Class<?> clazz, String path)
    {
        final String[] components = path.split("/");
        if ( components.length < 2 ) {
            return null;
        }

        IResourceStream result = null;
        final String topLevelDir = components[ components.length - 2 ];
        final String fileName = components[ components.length -1 ];

        if ( "css".equals( topLevelDir ) ||
            "img".equals( topLevelDir ) ||
            "template".equals( topLevelDir ) ||
            "themes".equals( topLevelDir ) ||
            "script".equals( topLevelDir ) )
        {
            try
            {
                final String location ="/"+topLevelDir+"/"+fileName;

                // try to load the resource from the context root first
                final URL url = context.getResource(location);
                if (url != null)
                {
                    result = new UrlResourceStream(url);
                }

                // fall-back to using classpath directly
                if ( result == null ) {
                    if ( ClasspathResourceStream.exists( location ) ) {
                        result = new ClasspathResourceStream( location );
                    }
                }
                return result;
            }
            catch (MalformedURLException e)
            {
                throw new WicketRuntimeException(e);
            }

        }
        return null;
    }

    private static final class ClasspathResourceStream extends AbstractResourceStream {

        private final String path;

        private final Object FILE_LOCK = new Object();

        // @GuardedBy( FILE_LOCK )
        private InputStream currentStream;

        public ClasspathResourceStream(String path) {

            if ( ! exists( path ) ) {
                throw new RuntimeException("Unable to load data from classpath:"+path+"");
            }
            this.path = path;
        }

        public static boolean exists(String classpath)
        {
            final InputStream stream = openClasspathResource( classpath );
            if ( stream == null ) {
                return false;
            }

            IOUtils.closeQuietly(stream);
            return true;
        }

        private static InputStream openClasspathResource(String classpath) {
            return ClasspathResourceStream.class.getClassLoader().getResourceAsStream( classpath );
        }

        @Override
        public InputStream getInputStream() throws ResourceStreamNotFoundException
        {
            synchronized( FILE_LOCK )
            {
                if ( currentStream == null)
                {
                    currentStream = openClasspathResource( path );
                    if ( currentStream == null ) {
                        throw new ResourceStreamNotFoundException("Unable to load data from classpath:"+path+"");
                    }
                }
                return currentStream;
            }
        }

        @Override
        public void close() throws IOException
        {
            synchronized( FILE_LOCK )
            {
                if ( currentStream != null ) {
                    try {
                        currentStream.close();
                    } finally {
                        currentStream = null;
                    }
                }
            }
        }
    }
}