/*
        Generic Master Worker Application Template
*/

#ifndef GMWAT_INCLUDE_H
#define GMWAT_INCLUDE_H

typedef enum {
    ROLE_UNKNOWN=0,
    ROLE_MASTER=1,
    ROLE_WORKER=2,
    ROLE_SUBMASTER=3
    } gmwat_processor_role; 

struct DataChunk {
    int refs; // should be -1 if it is 'virtual' or, if is concrete, the number of reference copies.
    unsigned int ident; // chunk identifier, must be unique for reusable chunks or zero for non reusable.
    int size; // size in bytes
    void *data; // pointer to location in memory;
};

struct WorkData {
    int count; // number of chunks
    struct DataChunk *chunks; //array of chunks
};

#ifndef TRUE
#define TRUE -1
#define FALSE 0
#endif


/**
 * Returns a cached data chunk based on ident. This function
 * should be used by users funcions to build WorkData instances.
 * 
 * @param ident  Data chunk identification.
 * 
 * @return NULL if chunk is not found in cache, or a pointer otherwise.
 */
struct DataChunk *getCachedChunk(int ident);


/****************************************************************
 *  the followin api should be implemented by users
 */


/**
 * Configure the template execution.
 * Parameters:
 *  Input:
 *      argc, argv, env (passed througth main program)
 *  Output:
 *      wu - Number of work units
 * 
 * @param argc   argc paramenter passed by main function.
 * @param argv   argv paramenter passed by main function.
 * @param env    env paramenter passed by main function.
 * @param wu     (OUTPUT) number of work units application should process.
 */
void user_config(int argc, char **argv, char **env, int *wu, int *iter, char role);

/**
 * Returns how many small grains of size gi+1 can be obtained
 * from a grain size gi, return zero otherwise.
 * 
 * @param gi     Grain size.
 * 
 * @return Number of parts that user function may be capable of division.
 */
int user_canBreakDown(int gi);

/**
 * Build an input work of grain of size gi, using data from
 * 'top' paramenter. The index could be from 0 to the response
 * of function canBreakDown(gi-1). The 'reuse' parameter is a
 * WorkData of grain gi that may be used to store data if is an
 * input parameter or just reused. The 'top' parameter may be
 * NULL, in that scenario, the function must allocate memory.
 * 
 * @param gi      grain size that function must build.
 * @param index   if gi=0, represents the work number, otherwise represents the relative index.
 *                For example:
 *                If gi=0 and index=30, than funcion should build the 30n piece of work.
 *                If gi=1, index=5, and a grain gi can be divided in 3 grains gi+1, than funcion should build the
 *                2n part of work unit 2 (gi=0) passed as 'top' parameter.
 * @param top     Represents the upper grain (gi-1) that should be used do build smaller grains (gi).
 *                Is null if gi=0 or if template dont use composed work units.
 * @param reuse   Is a work unit of grain size gi and should be used as a placeholder to return.
 * @param isInput Specify if its an input or an output work data.
 * 
 * @return An work data filled with data if isInput is true, or a memory place holder otherwise.
 */
struct WorkData *user_buildWorkData(int gi, int index, struct WorkData *top, struct WorkData *reuse, int isInput);

/**
 * Process input work data and store results on output work
 * data.
 * 
 * @param gi     Grain size.
 * @param index  Relative index of work.
 * @param input  input data that should be used to process.
 * @param output result data placeholder.
 */
void user_processWorkData(int gi, int index, struct WorkData *input, struct WorkData *output);

/**
 * This function is called on master when a response workdata. 
 * This funcion is called on communication thread, so it should not spent time in processing or io that may cause main application slowdown.
 * 
 * @param gi     Grain size of work data.
 * @param index  Index of work data.
 * @param work   the work data of grain size gi and index position that should be written.
 */
void user_writeWorkData(int gi, int index, struct WorkData *work);

void user_readWorkData(int gi, int index, struct WorkData *work);

void user_updateWorkData(int gi, int index, struct WorkData *top, struct WorkData *current);

void user_freeWorkData(struct WorkData *work);

void user_iteration_start(int i);

void user_iteration_end(int i);

void user_finalize();

#endif // #ifndef GMWAT_INCLUDE_H
