/*
 *  drivers/net/ctx.c
 *
 * Author: Guenter Roeck <groeck@redback.com>
 *
 */

#include <linux/netdevice.h>
#include <linux/ethtool.h>
#include <linux/etherdevice.h>

#include <net/dst.h>
#include <net/xfrm.h>
#include <linux/veth.h>

#define DRV_NAME	"ctx"
#define DRV_VERSION	"1.0"

struct ctx_net_stats {
	unsigned long	rx_packets;
	unsigned long	tx_packets;
	unsigned long	rx_bytes;
	unsigned long	tx_bytes;
	unsigned long	tx_dropped;
};

struct ctx_priv {
	struct net_device *dev;
	struct ctx_net_stats *stats;
	unsigned ip_summed;
};

/*
 * ethtool interface
 */

static int ctx_get_settings(struct net_device *dev, struct ethtool_cmd *cmd)
{
	cmd->supported		= 0;
	cmd->advertising	= 0;
	cmd->speed		= SPEED_10000;
	cmd->duplex		= DUPLEX_FULL;
	cmd->port		= PORT_TP;
	cmd->phy_address	= 0;
	cmd->transceiver	= XCVR_INTERNAL;
	cmd->autoneg		= AUTONEG_DISABLE;
	cmd->maxtxpkt		= 0;
	cmd->maxrxpkt		= 0;
	return 0;
}

static void ctx_get_drvinfo(struct net_device *dev, struct ethtool_drvinfo *info)
{
	strcpy(info->driver, DRV_NAME);
	strcpy(info->version, DRV_VERSION);
	strcpy(info->fw_version, "N/A");
}

static void ctx_get_strings(struct net_device *dev, u32 stringset, u8 *buf)
{
	switch(stringset) {
	}
}

static int ctx_get_sset_count(struct net_device *dev, int sset)
{
	switch (sset) {
	default:
		return -EOPNOTSUPP;
	}
}

static void ctx_get_ethtool_stats(struct net_device *dev,
		struct ethtool_stats *stats, u64 *data)
{
}

static u32 ctx_get_rx_csum(struct net_device *dev)
{
	struct ctx_priv *priv;

	priv = netdev_priv(dev);
	return priv->ip_summed == CHECKSUM_UNNECESSARY;
}

static int ctx_set_rx_csum(struct net_device *dev, u32 data)
{
	struct ctx_priv *priv;

	priv = netdev_priv(dev);
	priv->ip_summed = data ? CHECKSUM_UNNECESSARY : CHECKSUM_NONE;
	return 0;
}

static u32 ctx_get_tx_csum(struct net_device *dev)
{
	return (dev->features & NETIF_F_NO_CSUM) != 0;
}

static int ctx_set_tx_csum(struct net_device *dev, u32 data)
{
	if (data)
		dev->features |= NETIF_F_NO_CSUM;
	else
		dev->features &= ~NETIF_F_NO_CSUM;
	return 0;
}

static struct ethtool_ops ctx_ethtool_ops = {
	.get_settings		= ctx_get_settings,
	.get_drvinfo		= ctx_get_drvinfo,
	.get_link		= ethtool_op_get_link,
	.get_rx_csum		= ctx_get_rx_csum,
	.set_rx_csum		= ctx_set_rx_csum,
	.get_tx_csum		= ctx_get_tx_csum,
	.set_tx_csum		= ctx_set_tx_csum,
	.get_sg			= ethtool_op_get_sg,
	.set_sg			= ethtool_op_set_sg,
	.get_strings		= ctx_get_strings,
	.get_sset_count		= ctx_get_sset_count,
	.get_ethtool_stats	= ctx_get_ethtool_stats,
};

/*
 * xmit
 */

static int ctx_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct ctx_priv *priv;
	struct ctx_net_stats *stats;
	int cpu;

	skb_orphan(skb);

	priv = netdev_priv(dev);

	cpu = smp_processor_id();
	stats = per_cpu_ptr(priv->stats, cpu);

	stats->tx_bytes += skb->len;
	stats->tx_packets++;

	kfree_skb(skb);
	return 0;
}

/*
 * general routines
 */

static struct net_device_stats *ctx_get_stats(struct net_device *dev)
{
	struct ctx_priv *priv;
	struct net_device_stats *dev_stats;
	int cpu;
	struct ctx_net_stats *stats;

	priv = netdev_priv(dev);
	dev_stats = &dev->stats;

	dev_stats->rx_packets = 0;
	dev_stats->tx_packets = 0;
	dev_stats->rx_bytes = 0;
	dev_stats->tx_bytes = 0;
	dev_stats->tx_dropped = 0;

	for_each_online_cpu(cpu) {
		stats = per_cpu_ptr(priv->stats, cpu);

		dev_stats->rx_packets += stats->rx_packets;
		dev_stats->tx_packets += stats->tx_packets;
		dev_stats->rx_bytes += stats->rx_bytes;
		dev_stats->tx_bytes += stats->tx_bytes;
		dev_stats->tx_dropped += stats->tx_dropped;
	}

	return dev_stats;
}

static int ctx_open(struct net_device *dev)
{
	netif_carrier_on(dev);
	return 0;
}

static int ctx_dev_init(struct net_device *dev)
{
	struct ctx_net_stats *stats;
	struct ctx_priv *priv;

	stats = alloc_percpu(struct ctx_net_stats);
	if (stats == NULL)
		return -ENOMEM;

	priv = netdev_priv(dev);
	priv->stats = stats;
	return 0;
}

static void ctx_dev_free(struct net_device *dev)
{
	struct ctx_priv *priv;

	priv = netdev_priv(dev);
	free_percpu(priv->stats);
	free_netdev(dev);
}

static void ctx_setup(struct net_device *dev)
{
	ether_setup(dev);

	dev->hard_start_xmit = ctx_xmit;
	dev->get_stats = ctx_get_stats;
	dev->open = ctx_open;
	dev->ethtool_ops = &ctx_ethtool_ops;
	dev->features |= NETIF_F_LLTX;
	dev->init = ctx_dev_init;
	dev->destructor = ctx_dev_free;
}

static void ctx_change_state(struct net_device *dev)
{
	struct ctx_priv *priv;

	priv = netdev_priv(dev);

	if (!netif_carrier_ok(dev))
		netif_carrier_on(dev);
}

static int ctx_device_event(struct notifier_block *unused,
			     unsigned long event, void *ptr)
{
	struct net_device *dev = ptr;

	if (dev->open != ctx_open)
		goto out;

	switch (event) {
	case NETDEV_CHANGE:
		ctx_change_state(dev);
		break;
	}
out:
	return NOTIFY_DONE;
}

static struct notifier_block ctx_notifier_block __read_mostly = {
	.notifier_call	= ctx_device_event,
};

/*
 * netlink interface
 */

static int ctx_validate(struct nlattr *tb[], struct nlattr *data[])
{
	if (tb[IFLA_ADDRESS]) {
		if (nla_len(tb[IFLA_ADDRESS]) != ETH_ALEN)
			return -EINVAL;
		if (!is_valid_ether_addr(nla_data(tb[IFLA_ADDRESS])))
			return -EADDRNOTAVAIL;
	}
	return 0;
}

static struct rtnl_link_ops ctx_link_ops;

static int ctx_newlink(struct net_device *dev,
			 struct nlattr *tb[], struct nlattr *data[])
{
	int err;
	struct ctx_priv *priv;

	/*
	 * register device
	 *
	 * note, that since we've registered new device the dev's name
	 * should be re-allocated
	 */

	if (tb[IFLA_ADDRESS] == NULL)
		random_ether_addr(dev->dev_addr);

	if (tb[IFLA_IFNAME])
		nla_strlcpy(dev->name, tb[IFLA_IFNAME], IFNAMSIZ);
	else
		snprintf(dev->name, IFNAMSIZ, DRV_NAME "%%d");

	if (strchr(dev->name, '%')) {
		err = dev_alloc_name(dev, dev->name);
		if (err < 0)
			return err;
	}

	err = register_netdevice(dev);
	if (err < 0)
		return err;

	netif_carrier_off(dev);

	priv = netdev_priv(dev);
	priv->dev = dev;
	return 0;
}

static void ctx_dellink(struct net_device *dev)
{
	unregister_netdevice(dev);
}

static const struct nla_policy ctx_policy[VETH_INFO_MAX + 1];

static struct rtnl_link_ops ctx_link_ops = {
	.kind		= DRV_NAME,
	.priv_size	= sizeof(struct ctx_priv),
	.setup		= ctx_setup,
	.validate	= ctx_validate,
	.newlink	= ctx_newlink,
	.dellink	= ctx_dellink,
	.policy		= ctx_policy,
	.maxtype	= VETH_INFO_MAX,
};

/*
 * init/fini
 */

static __init int ctx_init(void)
{
	register_netdevice_notifier(&ctx_notifier_block);
	return rtnl_link_register(&ctx_link_ops);
}

static __exit void ctx_exit(void)
{
	rtnl_link_unregister(&ctx_link_ops);
	unregister_netdevice_notifier(&ctx_notifier_block);
}

module_init(ctx_init);
module_exit(ctx_exit);

MODULE_DESCRIPTION("Context device");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS_RTNL_LINK(DRV_NAME);
