/*
   (c) Copyright 2000  convergence integrated media GmbH.
   All rights reserved.

   Written by Denis Oliver Kropp <dok@convergence.de> and
              Andreas Hundt <andi@convergence.de>.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#if defined(__dietlibc__) && !defined(_BSD_SOURCE)
#define _BSD_SOURCE
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <malloc.h>
#include <string.h>

#include <sys/time.h>
#include <sys/timeb.h>
#include <sys/types.h>
#include <sys/select.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <inttypes.h>
#include <linux/videodev.h>

#include <pthread.h>

#include <directfb.h>
#include <directfb_internals.h>

#include <media/idirectfbvideoprovider.h>

#include <core/core.h>
#include <core/coredefs.h>
#include <core/coretypes.h>

#include <core/state.h>
#include <core/gfxcard.h>
#include <core/layers.h>
#include <core/surfaces.h>
#include <core/surfacemanager.h>

#include <display/idirectfbsurface.h>

#include <misc/util.h>
#include <misc/mem.h>
#include <misc/memcpy.h>

static DFBResult
Probe( IDirectFBVideoProvider_ProbeContext *ctx );

static DFBResult
Construct( IDirectFBVideoProvider *thiz,
           const char             *filename );

#include <interface_implementation.h>

DFB_INTERFACE_IMPLEMENTATION( IDirectFBVideoProvider, V4L )

/*
 * private data struct of IDirectFBVideoProvider
 */
typedef struct {
     int                      ref;       /* reference counter */

     char                    *filename;
     int                      fd;
     struct video_capability  vcap;
     pthread_t                thread;
     CoreSurface             *destination;
     DVFrameCallback          callback;
     void                    *ctx;

     size_t                  mapSize;
     uint8_t		     *cbuffer0;
     uint8_t                 *cbuffer1;

     struct video_mmap	     mmap_buf0;
     struct video_mmap	     mmap_buf1;

     IDirectFBSurface	     *surface;

     int		     height;

     CoreCleanup             *cleanup;
} IDirectFBVideoProvider_V4L_data;

static const unsigned int zero = 0;
static const unsigned int one = 1;

static void* FrameThread( void *context );
static ReactionResult v4l_surface_listener( const void *msg_data, void *ctx );
static DFBResult v4l_to_surface( CoreSurface *surface, DFBRectangle *rect,
                                 IDirectFBVideoProvider_V4L_data *data );
static DFBResult v4l_stop( IDirectFBVideoProvider_V4L_data *data );
static void v4l_deinit( IDirectFBVideoProvider_V4L_data *data );
static void v4l_cleanup( void *data, int emergency );


static void IDirectFBVideoProvider_V4L_Destruct( IDirectFBVideoProvider *thiz )
{
     IDirectFBVideoProvider_V4L_data *data =
          (IDirectFBVideoProvider_V4L_data*)thiz->priv;

     v4l_deinit( data );

     if (data->cleanup)
          dfb_core_cleanup_remove( data->cleanup );

     DFBFREE( data->filename );

     DFBFREE( thiz->priv );
     thiz->priv = NULL;

#ifndef DFB_DEBUG
     DFBFREE( thiz );
#endif
}

static DFBResult IDirectFBVideoProvider_V4L_AddRef( IDirectFBVideoProvider *thiz )
{
     INTERFACE_GET_DATA (IDirectFBVideoProvider_V4L)

     data->ref++;

     return DFB_OK;
}

static DFBResult IDirectFBVideoProvider_V4L_Release( IDirectFBVideoProvider *thiz )
{
     INTERFACE_GET_DATA (IDirectFBVideoProvider_V4L)

     if (--data->ref == 0) {
          IDirectFBVideoProvider_V4L_Destruct( thiz );
     }

     return DFB_OK;
}

static DFBResult IDirectFBVideoProvider_V4L_GetCapabilities (
                                           IDirectFBVideoProvider       *thiz,
                                           DFBVideoProviderCapabilities *caps )
{
     INTERFACE_GET_DATA (IDirectFBVideoProvider_V4L)

     if (!caps)
          return DFB_INVARG;

     *caps = ( DVCAPS_BASIC      |
               DVCAPS_BRIGHTNESS |
               DVCAPS_CONTRAST   |
               DVCAPS_HUE        |
               DVCAPS_SATURATION |
               DVCAPS_INTERLACED );

     if (data->vcap.type & VID_TYPE_SCALES)
          *caps |= DVCAPS_SCALE;

     return DFB_OK;
}

static DFBResult IDirectFBVideoProvider_V4L_GetSurfaceDescription(
                                                 IDirectFBVideoProvider *thiz,
                                                 DFBSurfaceDescription  *desc )
{
     IDirectFBVideoProvider_V4L_data *data;

     if (!thiz || !desc)
          return DFB_INVARG;

     data = (IDirectFBVideoProvider_V4L_data*)thiz->priv;

     if (!data)
          return DFB_DEAD;

     desc->flags  = DSDESC_WIDTH | DSDESC_HEIGHT | DSDESC_PIXELFORMAT;
     desc->width  = 768;
     desc->height = 576;
     desc->pixelformat = dfb_primary_layer_pixelformat();

     return DFB_OK;
}

static DFBResult IDirectFBVideoProvider_V4L_PlayTo(
                                            IDirectFBVideoProvider *thiz,
                                            IDirectFBSurface       *destination,
                                            DFBRectangle           *dstrect,
                                            DVFrameCallback         callback,
                                            void                   *ctx )
{
     DFBRectangle           rect;
     IDirectFBSurface_data *dst_data;

     INTERFACE_GET_DATA (IDirectFBVideoProvider_V4L)

     if (!destination)
          return DFB_INVARG;

     dst_data = (IDirectFBSurface_data*)destination->priv;

     if (!dst_data)
          return DFB_DEAD;

     if (!dst_data->area.current.w || !dst_data->area.current.h)
          return DFB_INVAREA;

     if (dstrect) {
          if (dstrect->w < 1  ||  dstrect->h < 1)
               return DFB_INVARG;

          rect = *dstrect;

          rect.x += dst_data->area.wanted.x;
          rect.y += dst_data->area.wanted.y;
     }
     else
          rect = dst_data->area.wanted;

     if (!dfb_rectangle_intersect( &rect, &dst_data->area.current ))
          return DFB_INVAREA;


     data->surface  = destination;

     data->callback = callback;
     data->ctx      = ctx;

     return v4l_to_surface( dst_data->surface, &rect, data );
}

static DFBResult IDirectFBVideoProvider_V4L_Stop(
                                                 IDirectFBVideoProvider *thiz )
{
     DFBResult res;

     INTERFACE_GET_DATA (IDirectFBVideoProvider_V4L)

     res = v4l_stop( data );
     
     data->surface = 0;
     data->callback = 0;
     data->ctx = 0;

     return res;
}

static DFBResult IDirectFBVideoProvider_V4L_SeekTo(
                                              IDirectFBVideoProvider *thiz,
                                              double                  seconds )
{
     INTERFACE_GET_DATA (IDirectFBVideoProvider_V4L)

     return DFB_UNIMPLEMENTED;
}

static DFBResult IDirectFBVideoProvider_V4L_GetPos(
                                              IDirectFBVideoProvider *thiz,
                                              double                 *seconds )
{
     INTERFACE_GET_DATA (IDirectFBVideoProvider_V4L)

     if (!seconds)
        return DFB_INVARG;

     *seconds = 0.0;

     return DFB_UNIMPLEMENTED;
}

static DFBResult IDirectFBVideoProvider_V4L_GetLength(
                                              IDirectFBVideoProvider *thiz,
                                              double                 *seconds )
{
     INTERFACE_GET_DATA (IDirectFBVideoProvider_V4L)

     if (!seconds)
        return DFB_INVARG;

     *seconds = 0.0;

     return DFB_UNIMPLEMENTED;
}

static DFBResult IDirectFBVideoProvider_V4L_GetColorAdjustment(
                                                  IDirectFBVideoProvider *thiz,
                                                  DFBColorAdjustment     *adj )
{
     struct video_picture pic;

     INTERFACE_GET_DATA (IDirectFBVideoProvider_V4L)

     if (!adj)
        return DFB_INVARG;

     ioctl( data->fd, VIDIOCGPICT, &pic );

     adj->flags =
          DCAF_BRIGHTNESS | DCAF_CONTRAST | DCAF_HUE | DCAF_SATURATION;
     adj->brightness = pic.brightness;
     adj->contrast   = pic.contrast;
     adj->hue        = pic.hue;
     adj->saturation = pic.colour;

     return DFB_OK;
}

static DFBResult IDirectFBVideoProvider_V4L_SetColorAdjustment(
     IDirectFBVideoProvider *thiz,
     DFBColorAdjustment     *adj )
{
     struct video_picture pic;

     INTERFACE_GET_DATA (IDirectFBVideoProvider_V4L)

     if (!adj)
        return DFB_INVARG;

     if (adj->flags == DCAF_NONE)
          return DFB_OK;

     if (ioctl( data->fd, VIDIOCGPICT, &pic ) < 0) {
          DFBResult ret = errno2dfb( errno );

          PERRORMSG( "DirectFB/v4l: VIDIOCGPICT failed!\n" );

          return ret;
     }

     if (adj->flags & DCAF_BRIGHTNESS) pic.brightness = adj->brightness;
     if (adj->flags & DCAF_CONTRAST)   pic.contrast   = adj->contrast;
     if (adj->flags & DCAF_HUE)        pic.hue        = adj->hue;
     if (adj->flags & DCAF_SATURATION) pic.colour     = adj->saturation;

     if (ioctl( data->fd, VIDIOCSPICT, &pic ) < 0) {
          DFBResult ret = errno2dfb( errno );

          PERRORMSG( "DirectFB/v4l: VIDIOCSPICT failed!\n" );

          return ret;
     }

     return DFB_OK;
}


/* exported symbols */

static DFBResult
Probe( IDirectFBVideoProvider_ProbeContext *ctx )
{
     if (strncmp( ctx->filename, "/dev/video", 10 ) == 0)
          return DFB_OK;

     if (strncmp( ctx->filename, "/dev/v4l/video", 14 ) == 0)
          return DFB_OK;

     return DFB_UNSUPPORTED;
}

static DFBResult
Construct( IDirectFBVideoProvider *thiz, const char *filename )
{
     int fd;
     IDirectFBVideoProvider_V4L_data *data;

     fd = open( filename, O_RDWR );
     if (fd < 0) {
          DFBResult ret = errno2dfb( errno );

          PERRORMSG( "DirectFB/v4l: Cannot open `%s'!\n", filename );

          return ret;
     }

     data = (IDirectFBVideoProvider_V4L_data*)
          DFBCALLOC( 1, sizeof(IDirectFBVideoProvider_V4L_data) );

     thiz->priv = data;

     data->ref = 1;

     ioctl( fd, VIDIOCGCAP, &data->vcap );
     ioctl( fd, VIDIOCCAPTURE, &zero );

     data->filename = DFBSTRDUP( filename );
     data->fd = fd;
     data->thread = -1;

     thiz->AddRef    = IDirectFBVideoProvider_V4L_AddRef;
     thiz->Release   = IDirectFBVideoProvider_V4L_Release;
     thiz->GetCapabilities = IDirectFBVideoProvider_V4L_GetCapabilities;
     thiz->GetSurfaceDescription =
          IDirectFBVideoProvider_V4L_GetSurfaceDescription;
     thiz->PlayTo    = IDirectFBVideoProvider_V4L_PlayTo;
     thiz->Stop      = IDirectFBVideoProvider_V4L_Stop;
     thiz->SeekTo    = IDirectFBVideoProvider_V4L_SeekTo;
     thiz->GetPos    = IDirectFBVideoProvider_V4L_GetPos;
     thiz->GetLength = IDirectFBVideoProvider_V4L_GetLength;
     thiz->GetColorAdjustment = IDirectFBVideoProvider_V4L_GetColorAdjustment;
     thiz->SetColorAdjustment = IDirectFBVideoProvider_V4L_SetColorAdjustment;

     return DFB_OK;
}


/*****************/

static void deinterlace (int mode, int height, int pitch, uint8_t* ptr)
{
    switch (mode) {
	case 1:
	{
	    int i = 0;
	    for (; i<height; ++i) {
		memset(ptr+((i+1)*pitch),0,pitch);
		++i;
	    }
	}
	break;
	case 2:
	{
	    int i = 0;
	    for (; i<height; ++i) {
		memcpy(ptr+(i+1)*pitch, ptr+i*pitch, pitch);
		++i;
	    }
	}
	break;
	default:
	    /* do nothing */
    }
}

static void handleBufferCopy (IDirectFBVideoProvider_V4L_data *data,
			      uint8_t *buffer)
{
    int deinterlaceMode = 0;

    void *ptr = 0;
    int pitch = 0;

    data->surface->Lock(data->surface, DSLF_WRITE, &ptr, &pitch);

    deinterlace (deinterlaceMode,
		 data->height, pitch, buffer);
    dfb_memcpy(ptr, buffer, pitch*data->height);
    data->surface->Unlock(data->surface);
}

static void* FrameThread( void *ctx )
{
     IDirectFBVideoProvider_V4L_data *data =
          (IDirectFBVideoProvider_V4L_data*)ctx;


     while (1) {
	 int c;

	 pthread_testcancel();

	 /* Set up for the next frame */
	 if (ioctl(data->fd, VIDIOCMCAPTURE, &data->mmap_buf1)<0) {
	     fprintf (stderr, "Error VIDIOCMCAPTURE\n");
	 }

	 /* Wait for the first frame */
	 if ((c = ioctl(data->fd, VIDIOCSYNC, &data->mmap_buf0.frame)) < 0) {
	     fprintf(stderr, "%d: !!%d\n", __LINE__, c);
	 }

	 if (data->callback)
	     data->callback( data->ctx );
	     
	 /* process first frame */
	 handleBufferCopy (data, data->cbuffer0);

	 pthread_testcancel();

	 /* Set up for first frame again */
	 if (ioctl(data->fd, VIDIOCMCAPTURE, &data->mmap_buf0)<0) {
	     fprintf (stderr, "Error VIDIOCMCAPTURE\n");
	 }

	 /* Wait for second frame */
	 if ((c = ioctl(data->fd, VIDIOCSYNC, &data->mmap_buf1.frame)) < 0) {
	     fprintf(stderr, "%d: !!%d\n", __LINE__, c);
	 }

	 if (data->callback)
	     data->callback( data->ctx );

	 /* process second frame */
	 handleBufferCopy (data, data->cbuffer1);
     }
}

static ReactionResult v4l_surface_listener( const void *msg_data, void *ctx )
{
     v4l_stop( (IDirectFBVideoProvider_V4L_data*)ctx );

     return RS_REMOVE;
}

/************/

static DFBResult v4l_to_surface( CoreSurface *surface, DFBRectangle *rect,
                                 IDirectFBVideoProvider_V4L_data *data )
{
     DFBResult ret;
     int bpp, palette;
     SurfaceBuffer *buffer = surface->back_buffer;

     dfb_surfacemanager_lock( surface->manager );
     ret = dfb_surfacemanager_assure_video( surface->manager, buffer );
     dfb_surfacemanager_unlock( surface->manager );
     if (ret)
          return ret;

     v4l_stop( data );

     data->height = rect->h;
     {
	 struct video_capability capability;
	 
	 if( ioctl(data->fd, VIDIOCGCAP, &capability) == -1 )
	 {
	     fprintf(stderr, "Error: ioctl(fd,VIDIOCGCAP,&capability)\n");
	     exit(1);
	 }
     }

     /*
      * Getting fetch buffers.
      */
     {
	 struct video_mbuf mbuf;
	 if (ioctl( data->fd, VIDIOCGMBUF, &mbuf ) < 0) {
	     DFBResult ret = errno2dfb( errno );

	     PERRORMSG( "DirectFB/v4l: VIDIOCGPICT failed!\n" );

	     return ret;
	 }
	 fprintf (stderr, "mbuf.size=%d\n", mbuf.size);
	 fprintf (stderr, "mbuf.frames=%d\n", mbuf.frames);

	 if (mbuf.frames > 2) {
	     PERRORMSG( "DirectFB/v4l: No support for more than 2 frames!\n");
	     return -1;
	 }

	 data->mapSize = mbuf.size;
	 data -> cbuffer0 = mmap(0, mbuf.size, PROT_READ|PROT_WRITE, 
				 MAP_SHARED, data->fd, 0);
	 if (0 == data->cbuffer0) {
	     DFBResult ret = errno2dfb( errno );
	     PERRORMSG( "DirectFB/v4l: mmap returned 0\n");
	     return ret;
	 }
	 if (mbuf.frames > 1) {
	     data -> cbuffer1 = data->cbuffer0 + mbuf.offsets[1];
	 }
	 fprintf(stderr, "cbuffer0 = %p, cbuffer1 = %p\n", data->cbuffer0, data->cbuffer1);
     }

     switch (surface->format) {
	 case DSPF_YUY2:
	     bpp = 16;
	     palette = VIDEO_PALETTE_YUYV;
	     break;
	 case DSPF_UYVY:
	     bpp = 16;
	     palette = VIDEO_PALETTE_UYVY;
	     break;
	 case DSPF_RGB15:
	     bpp = 15;
	     palette = VIDEO_PALETTE_RGB555;
	     break;
	 case DSPF_RGB16:
	     bpp = 16;
	     palette = VIDEO_PALETTE_RGB565;
	     break;
	 case DSPF_RGB24:
	     bpp = 24;
	     palette = VIDEO_PALETTE_RGB24;
	     break;
	 case DSPF_ARGB:
	 case DSPF_RGB32:
	     bpp = 32;
	     palette = VIDEO_PALETTE_RGB32;
	     break;
	 default:
	     BUG( "unknown pixel format" );
	     return DFB_BUG;
     }
	     
     data->mmap_buf0.width = data->mmap_buf1.width = surface->width;
     data->mmap_buf0.height = data->mmap_buf1.height = surface->height;
     data->mmap_buf0.format = data->mmap_buf1.format = palette;
     data->mmap_buf0.frame = 0;
     data->mmap_buf1.frame = 1;

     fprintf(stderr,"%dx%d - %d\n", surface->width, surface->height, palette);

     {
	 /*
	  * Try a first capture.
	  */
	 if( ioctl( data->fd, VIDIOCMCAPTURE, &data->mmap_buf0) < 0 ) {
               DFBResult ret = errno2dfb( errno );

               PERRORMSG( "DirectFB/v4l: "
                          "VIDIOCMCAPTURE failed!\n" );

               return ret;
	 }
     }

     if (!data->cleanup)
          data->cleanup = dfb_core_cleanup_add( v4l_cleanup, data, 1 );

     data->destination = surface;

     pthread_create( &data->thread, NULL, FrameThread, data );

     return DFB_OK;
}

static DFBResult v4l_stop( IDirectFBVideoProvider_V4L_data *data )
{
     if (data->thread != -1) {
          pthread_cancel( data->thread );
          pthread_join( data->thread, NULL );
          data->thread = -1;
     }

     if (0 != data->surface) data->surface->Unlock(data->surface);

     if (0 != data->cbuffer0) {
	 munmap (data->cbuffer0, data->mapSize);
     }

     data->destination = NULL;

     return DFB_OK;
}

static void v4l_deinit( IDirectFBVideoProvider_V4L_data *data )
{
     if (data->fd == -1) {
          BUG( "v4l_deinit with 'fd == -1'" );
          return;
     }

     v4l_stop( data );

     close( data->fd );
     data->fd = -1;
}

static void v4l_cleanup( void *ctx, int emergency )
{
     IDirectFBVideoProvider_V4L_data *data =
          (IDirectFBVideoProvider_V4L_data*)ctx;

     if (emergency)
          v4l_stop( data );
     else
          v4l_deinit( data );
}
