/*
 * ARM memcpy asm replacement.
 *
 * Copyright (C) 2009 Bluush Dev Team.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <sys/time.h>

#include "armasm_memcpy.h"




typedef struct
{
	void *(*func)(void*,const void*,size_t);
	char *name;	
} memcpy_info_t;



static memcpy_info_t info[] =
{
	{ memcpy,   "libc" },
	{ direct_armasm_memcpy,  "armasm" },
	{ direct_armasm_memcpy2, "armasm2" },
};


#define INFO_SIZE(i)	(sizeof((i)) / sizeof(memcpy_info_t))


static void test_memcpy(memcpy_info_t *info, uint8_t *dst, uint8_t *src, size_t size, struct timeval *tv)
{
	struct timeval start, stop;
	int i;
	
	fprintf(stderr, "Testing %s (0x%08x <==> 0x%08x : %d):\n", info->name, (uint32_t)src, (uint32_t)dst, size);
	gettimeofday(&start, NULL);
	for(i=0 ; i<100 ; i++)
		info->func(dst, src, size);
	gettimeofday(&stop, NULL);
	timersub(&stop, &start, tv);
	fprintf(stderr, "\x1b[1A\x1b[65C%d.%06d sec\n", tv->tv_sec, tv->tv_usec);
}



static void show_result(struct timeval tv[])
{
	int i, fast;
	
	fast = 0;
	
	for(i=1 ; i<INFO_SIZE(info) ; i++)
	{
		if(timercmp(&tv[fast], &tv[i], >))
		{
			fast = i;
		}
	}

	fprintf(stderr, "The faster routine is %s\n\n", info[fast].name);
}



int main(int argc, char *argv[])
{
	struct timeval tv[INFO_SIZE(info)];
	uint8_t *src, *dst;
	size_t size = 500;
	int i;

	if(argc > 1)
		size = strtoul(argv[1], NULL, 0);
	
	src = (uint8_t*)malloc(sizeof(uint8_t*) * size + 10);
	dst = (uint8_t*)malloc(sizeof(uint8_t*) * size + 10);

	if(!src || !dst)
	{
		fprintf(stderr, "### ERROR: Allocation failure\n");
		exit(EXIT_FAILURE);
	}
	
	fprintf(stderr, "32bit src/dst Aligned test:\n");
	for(i=0 ; i<INFO_SIZE(info) ; i++)	
		test_memcpy(&info[i],dst,src,size,&tv[i]);
	show_result(tv);

	fprintf(stderr, "16bit src/dst Aligned test:\n");
	for(i=0 ; i<INFO_SIZE(info) ; i++)	
		test_memcpy(&info[i],dst+2,src+2,size,&tv[i]);
	show_result(tv);

	fprintf(stderr, "8bit src/dst Aligned test:\n");
	for(i=0 ; i<INFO_SIZE(info) ; i++)	
		test_memcpy(&info[i],dst+1,src+1,size,&tv[i]);
	show_result(tv);


	
	fprintf(stderr, "16bit src Aligned test:\n");
	for(i=0 ; i<INFO_SIZE(info) ; i++)	
		test_memcpy(&info[i],dst,src+2,size,&tv[i]);
	show_result(tv);

	fprintf(stderr, "8bit src Aligned test:\n");
	for(i=0 ; i<INFO_SIZE(info) ; i++)	
		test_memcpy(&info[i],dst,src+1,size,&tv[i]);
	show_result(tv);	


	fprintf(stderr, "16bit dst Aligned test:\n");
	for(i=0 ; i<INFO_SIZE(info) ; i++)	
		test_memcpy(&info[i],dst+2,src,size,&tv[i]);
	show_result(tv);

	fprintf(stderr, "8bit dst Aligned test:\n");
	for(i=0 ; i<INFO_SIZE(info) ; i++)	
		test_memcpy(&info[i],dst+1,src,size,&tv[i]);
	show_result(tv);	

	free(src); free(dst);
	return 0;
}

