package org.displaytag.model;

import java.util.StringTokenizer;
// TODD added this
import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;
import java.beans.PropertyDescriptor;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;
import org.apache.commons.beanutils.PropertyUtils;
import org.displaytag.decorator.TableDecorator;
import org.displaytag.exception.DecoratorException;
import org.displaytag.exception.ObjectLookupException;
import org.displaytag.util.Anchor;
import org.displaytag.util.Href;
import org.displaytag.util.HtmlAttributeMap;
import org.displaytag.util.HtmlTagUtil;
import org.displaytag.util.LinkUtil;
import org.displaytag.util.LookupUtil;
import org.displaytag.util.TagConstants;

/**
 * Represents a column in a table.
 * @author Fabrizio Giustina
 * @version $Revision: 1.9 $ ($Author: fgiust $)
 */
public class Column
{
    /**
     * Row this column belongs to.
     */
    private Row row;

    /**
     * Header of this column. The header cell contains all the attributes common to all cells in the same column
     */
    private HeaderCell header;

    /**
     * copy of the attribute map from the header cell. Needed to change attributes (title) in this cell only
     */
    private HtmlAttributeMap htmlAttributes;

    /**
     * contains the evaluated body value. Filled in getOpenTag.
     */
    private String stringValue;

    /**
     * Cell.
     */
    private Cell cell;

    /**
     * Constructor for Column.
     * @param headerCell HeaderCell
     * @param currentCell Cell
     * @param parentRow Row
     */
    public Column(HeaderCell headerCell, Cell currentCell, Row parentRow)
    {
        this.header = headerCell;
        this.row = parentRow;
        this.cell = currentCell;

        // also copy html attributes
        this.htmlAttributes = headerCell.getHtmlAttributes();
    }

    /**
     * Gets the value, after calling the table / column decorator is requested.
     * @param decorated boolean
     * @return Object
     * @throws ObjectLookupException for errors in bean property lookup
     * @throws DecoratorException if a column decorator is used and an exception is thrown during value decoration
     */
    public Object getValue(boolean decorated) throws ObjectLookupException,DecoratorException
    {
        // a static value has been set?
        if (this.cell.getStaticValue() != null)
        {
            return this.cell.getStaticValue();
        }

        Object object = null;
        TableDecorator tableDecorator = this.row.getParentTable().
            getTableDecorator();

        // TODD added this
        Boolean hasGetter = (Boolean)this.row.getParentTable().getDecoratorMethodCache().get(this.header.getBeanPropertyName());
        if ( (hasGetter == null) && (tableDecorator != null) )
        {
            hasGetter = new Boolean(tableDecorator.hasGetterFor(this.header.getBeanPropertyName()));
            this.row.getParentTable().getDecoratorMethodCache().put(this.header.getBeanPropertyName(), hasGetter);
        }
        else if ( ( hasGetter == null) && (tableDecorator == null ) )
        {
            hasGetter = new Boolean(false);
            this.row.getParentTable().getDecoratorMethodCache().put(this.header.getBeanPropertyName(), hasGetter);
        }

        // TODD added this
        Method method = null;
        String propertyName = this.header.getBeanPropertyName();
        Boolean isSimpleProperty = (Boolean)this.row.getParentTable().getSimplePropertyCache().get(propertyName);
        if (isSimpleProperty == null)
        {
            if ( (propertyName.indexOf(PropertyUtils.NESTED_DELIM) == -1) &&
                (propertyName.indexOf(PropertyUtils.MAPPED_DELIM) == -1) &&
                (propertyName.indexOf(PropertyUtils.INDEXED_DELIM) == -1))
            {
                isSimpleProperty = new Boolean(true);
                this.row.getParentTable().getSimplePropertyCache().put(propertyName, isSimpleProperty);
            }
            else
            {
                isSimpleProperty = new Boolean(false);
                this.row.getParentTable().getSimplePropertyCache().put(propertyName, isSimpleProperty);
            }
        }
        if (isSimpleProperty.booleanValue())
        {
            // Now cache the method
            PropertyDescriptor descriptor = null;
            Object bean = null;

            if (decorated && tableDecorator != null && hasGetter.booleanValue())
            {
                bean = tableDecorator;
            }
            else
            {
                bean = this.row.getObject();
            }
            if (bean instanceof Map)
            {
                object = ( (Map) bean).get(propertyName);
            }
            else
            {
                method = (Method)this.row.getParentTable().getPropertyMethodCache().get(propertyName);
                if (method == null)
                {

                    try
                    {
                        descriptor = PropertyUtils.getPropertyDescriptor(bean, propertyName);
                        method = PropertyUtils.getReadMethod(descriptor);
                        this.row.getParentTable().getPropertyMethodCache().put(propertyName, method);
                    }
                    catch (NoSuchMethodException e)
                    {
                         throw new ObjectLookupException(Column.class, bean, propertyName, e);
                    }
                    catch (InvocationTargetException e)
                    {
                        throw new ObjectLookupException(Column.class, bean, propertyName, e);
                    }
                    catch (IllegalAccessException e)
                    {
                         throw new ObjectLookupException(Column.class, bean, propertyName, e);
                    }
                }

                if (method != null)
                {
                    try
                    {
                        object = method.invoke(bean, new Object[0]);
                    }
                    catch (InvocationTargetException e)
                    {
                        throw new ObjectLookupException(Column.class, bean, propertyName, e);
                    }
                    catch (IllegalArgumentException e)
                    {
                          throw new ObjectLookupException(Column.class, bean, propertyName, e);
                    }
                    catch (IllegalAccessException e)
                    {
                         throw new ObjectLookupException(Column.class, bean, propertyName, e);
                    }
                }
            }
        }

        else
        {
            // if a decorator has been set, and if decorator has a getter for the requested property only, check decorator
            if (decorated && tableDecorator != null && hasGetter.booleanValue())
            {
                object = LookupUtil.getBeanProperty(tableDecorator,this.header.getBeanPropertyName());
            }
            else
            {
                // else check underlining object
                object = LookupUtil.getBeanProperty(this.row.getObject(),this.header.getBeanPropertyName());
            }
        }
        if (decorated && (this.header.getColumnDecorator() != null))
        {
            object = this.header.getColumnDecorator().decorate(object);
        }
        if (object == null || object.equals("null"))
        {
            if (!this.header.getShowNulls())
            {
                object = "";
            }
        }

        return object;
    }

    /**
     * Generates the cell open tag.
     * @return String td open tag
     * @throws ObjectLookupException for errors in bean property lookup
     * @throws DecoratorException if a column decorator is used and an exception is thrown during value decoration
     */
    public String getOpenTag() throws ObjectLookupException, DecoratorException
    {
        this.stringValue = createChoppedAndLinkedValue();

        // TODD changed this to cache
        String openTagString = (String)this.row.getParentTable().
            getColumnOpenTagCache().get(this.header.getBeanPropertyName());
        if (openTagString == null)
        {
            openTagString = HtmlTagUtil.createOpenTagString(TagConstants.
                TAGNAME_COLUMN, this.htmlAttributes);
            this.row.getParentTable().getColumnOpenTagCache().put(this.header.
                getBeanPropertyName(), openTagString);
        }

        return openTagString;
    }

    /**
     * Generates the cell close tag (&lt;/td>).
     * @return String td closing tag
     */
    public String getCloseTag()
    {
        this.stringValue = null;
        return this.header.getCloseTag();
    }

    /**
     * Calculates the cell content, cropping or linking the value as needed.
     * @return String
     * @throws ObjectLookupException for errors in bean property lookup
     * @throws DecoratorException if a column decorator is used and an exception is thrown during value decoration
     */
    public String createChoppedAndLinkedValue() throws ObjectLookupException,
        DecoratorException
    {

        Object choppedValue = getValue(true);

        boolean isChopped = false;
        String fullValue = "";
        if (choppedValue != null)
        {
            fullValue = choppedValue.toString();
        }

        // trim the string if a maxLength or maxWords is defined
        if (this.header.getMaxLength() > 0 &&
            fullValue.length() > this.header.getMaxLength())
        {
            choppedValue = StringUtils.abbreviate(fullValue,
                                                  this.header.getMaxLength() +
                                                  3);
            isChopped = true;
        }
        else if (this.header.getMaxWords() > 0)
        {
            StringBuffer buffer = new StringBuffer();
            StringTokenizer tokenizer = new StringTokenizer(fullValue);
            int tokensNum = tokenizer.countTokens();
            if (tokensNum > this.header.getMaxWords())
            {
                int wordsCount = 0;
                while (tokenizer.hasMoreTokens() &&
                       (wordsCount < this.header.getMaxWords()))
                {
                    buffer.append(tokenizer.nextToken() + " ");
                    wordsCount++;
                }
                buffer.append("...");
                choppedValue = buffer;
                isChopped = true;
            }
        }

        // chopped content? add the full content to the column "title" attribute
        if (isChopped)
        {
            // clone the attribute map, don't want to add title to all the columns
            this.htmlAttributes = (HtmlAttributeMap)this.htmlAttributes.clone();
            // add title
            this.htmlAttributes.put(TagConstants.ATTRIBUTE_TITLE,
                                    StringUtils.replace(fullValue, "\"",
                "&#34;"));
        }

        // Are we supposed to set up a link to the data being displayed in this column...
        if (this.header.getAutoLink())
        {
            choppedValue = LinkUtil.autoLink(choppedValue.toString());
        }
        else if (this.header.getHref() != null)
        {
            // generates the href for the link
            Href colHref = getColumnHref(fullValue);
            Anchor anchor = new Anchor(colHref, choppedValue.toString());
            choppedValue = anchor.toString();
        }

        if (choppedValue != null)
        {
            return choppedValue.toString();
        }
        return null;
    }

    /**
     * Generates the href for the column using paramName/property/scope.
     * @param columnContent column body
     * @return generated Href
     * @throws ObjectLookupException for errors in lookin up object properties
     */
    private Href getColumnHref(String columnContent) throws
        ObjectLookupException
    {
        // copy href
        Href colHref = new Href(this.header.getHref());

        // do we need to add a param?
        if (this.header.getParamName() != null)
        {

            Object paramValue;

            if (this.header.getParamProperty() != null)
            {
                // different property, go get it
                paramValue = LookupUtil.getBeanProperty(this.row.getObject(),
                    this.header.getParamProperty());

            }
            else
            {
                // same property as content
                paramValue = columnContent;
            }

            colHref.addParameter(this.header.getParamName(), paramValue);

        }
        return colHref;
    }

    /**
     * get the final value to be displayed in the table. This method can only be called after getOpenTag(), where the
     * content is evaluated
     * @return String final value to be displayed in the table
     */
    public String getChoppedAndLinkedValue()
    {
        return this.stringValue;
    }

    /**
     * returns the grouping order of this column or -1 if the column is not grouped.
     * @return int grouping order of this column or -1 if the column is not grouped
     */
    public int getGroup()
    {
        return this.header.getGroup();
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString()
    {
        return new ToStringBuilder(this, ToStringStyle.SIMPLE_STYLE)
            .append("cell", this.cell)
            .append("header", this.header)
            .append("htmlAttributes", this.htmlAttributes)
            .append("stringValue", this.stringValue)
            .toString();
    }
}
