/*
 * Copyright 2000-2001 VA Linux Systems, Inc.
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * on the rights to use, copy, modify, merge, publish, distribute, sub
 * license, and/or sell copies of the Software, and to permit persons to whom
 * the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT.  IN NO EVENT SHALL
 * VA LINUX SYSTEMS AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Authors:
 *    Keith Whitwell <keithw@valinux.com>
 */
/* $XFree86: xc/lib/GL/mesa/src/drv/mga/mgatex.c,v 1.10 2001/04/10 16:07:51 dawes Exp $ */

#include "mm.h"
#include "mgacontext.h"
#include "mgatex.h"
#include "mgaregs.h"
#include "mgatris.h"
#include "mgaioctl.h"

#include "context.h"
#include "enums.h"
#include "macros.h"
#include "mem.h"
#include "simple_list.h"
#include "texformat.h"


#ifdef MGA_USE_TABLE_FOR_FORMAT
#define TMC_nr_tformat (MESA_FORMAT_CI8 + 1)
static const unsigned TMC_tformat[ TMC_nr_tformat ] =
{
    [MESA_FORMAT_ARGB8888] = TMC_tformat_tw32 | TMC_takey_1 | TMC_tamask_0,
    [MESA_FORMAT_RGB565]   = TMC_tformat_tw16 | TMC_takey_1 | TMC_tamask_0,
    [MESA_FORMAT_ARGB4444] = TMC_tformat_tw12 | TMC_takey_1 | TMC_tamask_0,
    [MESA_FORMAT_ARGB1555] = TMC_tformat_tw15 | TMC_takey_1 | TMC_tamask_0,
    [MESA_FORMAT_CI8]      = TMC_tformat_tw8  | TMC_takey_1 | TMC_tamask_0,
};
#endif

void
mgaSetTexImages( mgaContextPtr mmesa,
		 struct gl_texture_object * tObj )
{
    mgaTextureObjectPtr t = (mgaTextureObjectPtr) tObj->DriverData;
    struct gl_texture_image *baseImage = tObj->Image[ tObj->BaseLevel ];
    GLint totalSize;
    GLint width, height;
    GLint i;
    GLint lastLevel;
    GLint log2Width, log2Height;
    GLuint txformat = 0;
    GLint ofs;

    /* Set the hardware texture format
     */
#ifndef MGA_USE_TABLE_FOR_FORMAT
    switch (baseImage->TexFormat->MesaFormat) {

	case MESA_FORMAT_ARGB8888: txformat = TMC_tformat_tw32;	break;
	case MESA_FORMAT_RGB565:   txformat = TMC_tformat_tw16; break;
	case MESA_FORMAT_ARGB4444: txformat = TMC_tformat_tw12;	break;
	case MESA_FORMAT_ARGB1555: txformat = TMC_tformat_tw15; break;
	case MESA_FORMAT_CI8:      txformat = TMC_tformat_tw8;  break;

	default:
	_mesa_problem(NULL, "unexpected texture format in mgaTexImage2D");
	return;
    }
#else
    if ( (baseImage->TexFormat->MesaFormat >= TMC_nr_tformat)
	 || (TMC_tformat[ baseImage->TexFormat->MesaFormat ] == 0) )
    {
	_mesa_problem(NULL, "unexpected texture format in " __FUNCTION__ );
	return;
    }

    txformat = TMC_tformat[ baseImage->TexFormat->MesaFormat ];

#endif

   /* We are going to upload all levels that are present, even if
    * later levels wouldn't be used by the current filtering mode.  This
    * allows the filtering mode to change without forcing another upload
    * of the images.
    */
   lastLevel = MGA_TEX_MAXLEVELS-1;

   totalSize = 0;
   for ( i = 0 ; i <= lastLevel ; i++ ) {
      const struct gl_texture_image *texImage;

      texImage = tObj->Image[i];
      if ( texImage == NULL ) {
	 lastLevel = i - 1;
	 break;
      }

      t->offsets[i] = totalSize;
      t->base.dirty_images |= (1<<i);

      totalSize += ((MAX2( texImage->Width, 8 ) *
		     MAX2( texImage->Height, 8 ) *
		     baseImage->TexFormat->TexelBytes) + 31) & ~31;
   }

   t->base.totalSize = totalSize;
   t->lastLevel = lastLevel;


   /* setup hardware register values */
   t->setup.texctl &= (TMC_tformat_MASK & TMC_tpitch_MASK 
		       & TMC_tpitchext_MASK);
   t->setup.texctl |= txformat;

   if ( baseImage->WidthLog2 >= 3 )
      t->setup.texctl |= ((baseImage->WidthLog2 - 3) << TMC_tpitch_SHIFT);
   else
      t->setup.texctl |= (TMC_tpitchlin_enable |
			  (baseImage->Width << TMC_tpitchext_SHIFT));


   if ( mmesa->glCtx->Light.Model.ColorControl == GL_SEPARATE_SPECULAR_COLOR )
      t->setup.texctl2 |= TMC_specen_enable;

   t->setup.texfilter &= TF_mapnb_MASK;
   t->setup.texfilter |= (lastLevel << TF_mapnb_SHIFT);

   /* warp texture registers */
   ofs = MGA_IS_G200(mmesa) ? 28 : 11;

   width = baseImage->Width;
   height = baseImage->Height;

   log2Width = baseImage->WidthLog2;
   log2Height = baseImage->HeightLog2;

   t->setup.texwidth = (MGA_FIELD(TW_twmask, width - 1) |
			MGA_FIELD(TW_rfw, (10 - log2Width - 8) & 63 ) |
			MGA_FIELD(TW_tw, (log2Width + ofs ) | 0x40 ));

   t->setup.texheight = (MGA_FIELD(TH_thmask, height - 1) |
			 MGA_FIELD(TH_rfh, (10 - log2Height - 8) & 63 ) |
			 MGA_FIELD(TH_th, (log2Height + ofs ) | 0x40 ));

   mgaUploadTexImages( mmesa, t );
}


/* ================================================================
 * Texture unit state management
 */

static void mgaUpdateTextureEnvG200( GLcontext *ctx, GLuint unit )
{
   struct gl_texture_object *tObj = ctx->Texture.Unit[0]._Current;
   mgaTextureObjectPtr t;

   if (!tObj || !tObj->DriverData)
      return;

   t = (mgaTextureObjectPtr)tObj->DriverData;

   t->setup.texctl2 &= ~TMC_decalblend_enable;

   switch (ctx->Texture.Unit[0].EnvMode) {
   case GL_REPLACE:
      t->setup.texctl &= ~TMC_tmodulate_enable;
      break;
   case GL_MODULATE:
      t->setup.texctl |= TMC_tmodulate_enable;
      break;
   case GL_DECAL:
      t->setup.texctl &= ~TMC_tmodulate_enable;
      t->setup.texctl2 |= TMC_decalblend_enable;
      break;
   case GL_BLEND:
      FALLBACK( ctx, MGA_FALLBACK_TEXTURE, GL_TRUE );
      break;
   default:
      break;
   }
}


#define MGA_DISABLE		0
#define MGA_REPLACE		1
#define MGA_MODULATE		2
#define MGA_DECAL		3
#define MGA_BLEND		4
#define MGA_ADD			5
#define MGA_MAX_COMBFUNC	6

static const GLuint g400_color_combine[][MGA_MAX_COMBFUNC] =
{
   /* Unit 0:
    */
   {
      /* Disable combiner stage
       */
      (0),

      /* GL_REPLACE
       */
      (TD0_color_sel_arg1 |
       TD0_alpha_arg2_diffuse |
       TD0_alpha_sel_arg2 ),
      
      /* GL_MODULATE
       */
      (TD0_color_arg2_diffuse |
       TD0_color_sel_mul |
       TD0_alpha_arg2_diffuse |
       TD0_alpha_sel_mul),
      
      /* GL_DECAL
       */
      (TD0_color_sel_arg1 |
       TD0_alpha_arg2_diffuse |
       TD0_alpha_sel_arg2),
      
      /* GL_BLEND
       */
      (0),
      
      /* GL_ADD
       */
      (TD0_color_arg2_diffuse |
       TD0_color_add_add |
       TD0_color_sel_add |
       TD0_alpha_arg2_diffuse |
       TD0_alpha_sel_mul),
   },
   
   /* Unit 1:
    */
   {
      /* Disable combiner stage
       */
      (0),
       
      /* GL_REPLACE
       */
      (TD0_color_sel_arg1 |
       TD0_alpha_arg2_diffuse |
       TD0_alpha_sel_arg2 ),
      
      /* GL_MODULATE
       */
      (TD0_color_arg2_prevstage |
       TD0_color_alpha_prevstage |
       TD0_color_sel_mul |
       TD0_alpha_arg2_prevstage |
       TD0_alpha_sel_mul),

      /* GL_DECAL
       */
      (TD0_color_sel_arg1 |
       TD0_alpha_arg2_prevstage |
       TD0_alpha_sel_arg2 ),
      
      /* GL_BLEND
       */
      (0),
      
      /* GL_ADD
       */
      (TD0_color_arg2_prevstage |
       TD0_color_alpha_prevstage |
       TD0_color_add_add |
       TD0_color_sel_add |
       TD0_alpha_arg2_prevstage |
       TD0_alpha_sel_mul),
   },
};

static const GLuint g400_alpha_combine[][MGA_MAX_COMBFUNC] =
{
   /* Unit 0:
    */
   {
      /* Disable combiner stage
       */
      (0),

      /* GL_REPLACE
       */
      (TD0_color_sel_arg2 |
       TD0_color_arg2_diffuse |
       TD0_alpha_sel_arg1 ),
      
      /* GL_MODULATE
       * FIXME: Is this correct?
       */
      (TD0_color_arg2_diffuse |
       TD0_color_sel_mul |
       TD0_alpha_arg2_diffuse |
       TD0_alpha_sel_mul),

      /* GL_DECAL
       */
      (TD0_color_arg2_diffuse |
       TD0_color_sel_arg2 |
       TD0_alpha_arg2_diffuse |
       TD0_alpha_sel_arg2),

      /* GL_BLEND
       */
      (TD0_color_arg2_diffuse |
       TD0_color_sel_mul |
       TD0_alpha_arg2_diffuse |
       TD0_alpha_sel_mul),

      /* GL_ADD
       */
      (TD0_color_arg2_diffuse |
       TD0_color_sel_mul |
       TD0_alpha_arg2_diffuse |
       TD0_alpha_sel_mul),
   },

   /* Unit 1:
    */
   {
      /* Disable combiner stage
       */
      (0),

      /* GL_REPLACE
       */
      (TD0_color_sel_arg2 |
       TD0_color_arg2_diffuse |
       TD0_alpha_sel_arg1 ),
      
      /* GL_MODULATE
       * FIXME: Is this correct?
       */
      (TD0_color_arg2_prevstage |
       TD0_color_alpha_prevstage |
       TD0_color_sel_mul |
       TD0_alpha_arg2_prevstage |
       TD0_alpha_sel_mul),

      /* GL_DECAL
       */
      (TD0_color_arg2_prevstage |
       TD0_color_sel_arg2 |
       TD0_alpha_arg2_prevstage |
       TD0_alpha_sel_arg2),

      /* GL_BLEND
       */
      (TD0_color_arg2_diffuse |
       TD0_color_sel_mul |
       TD0_alpha_arg2_diffuse |
       TD0_alpha_sel_mul),

      /* GL_ADD
       */
      (TD0_color_arg2_prevstage |
       TD0_color_sel_mul |
       TD0_alpha_arg2_prevstage |
       TD0_alpha_sel_mul),
   },
};

static void mgaUpdateTextureEnvG400( GLcontext *ctx, GLuint unit )
{
   mgaContextPtr mmesa = MGA_CONTEXT( ctx );
   GLuint *reg = ((GLuint *)&mmesa->setup.tdualstage0 + unit);
   GLuint source = mmesa->tmu_source[unit];
   struct gl_texture_object *tObj = ctx->Texture.Unit[source]._Current;
   GLenum format;

   if ( tObj != ctx->Texture.Unit[source].Current2D || !tObj ) 
      return;

   format = tObj->Image[tObj->BaseLevel]->Format;

   switch (ctx->Texture.Unit[source].EnvMode) {
   case GL_REPLACE:
      if (format == GL_RGB || format == GL_LUMINANCE) {
	 *reg = g400_color_combine[unit][MGA_REPLACE];
      }
      else if (format == GL_ALPHA) {
         *reg = g400_alpha_combine[unit][MGA_REPLACE];
      }
      else {
         *reg = (TD0_color_sel_arg1 |
                 TD0_alpha_sel_arg1 );
      }
      break;

   case GL_MODULATE:
      *reg = g400_color_combine[unit][MGA_MODULATE];
      break;
   case GL_DECAL:
      if (format == GL_RGB) {
	 *reg = g400_color_combine[unit][MGA_DECAL];
      }
      else if ( format == GL_RGBA ) {
#if 0
         if (unit == 0) {
            /* this doesn't work */
            *reg = (TD0_color_arg2_diffuse |
                    TD0_color_alpha_currtex |
                    TD0_color_alpha2inv_enable |
                    TD0_color_arg2mul_alpha2 |
                    TD0_color_arg1mul_alpha1 |
                    TD0_color_blend_enable |
                    TD0_color_arg1add_mulout |
                    TD0_color_arg2add_mulout |
                    TD0_color_add_add |
                    TD0_color_sel_mul |
                    TD0_alpha_arg2_diffuse |
                    TD0_alpha_sel_arg2 );
         }
         else {
            *reg = (TD0_color_arg2_prevstage |
                    TD0_color_alpha_currtex |
                    TD0_color_alpha2inv_enable |
                    TD0_color_arg2mul_alpha2 |
                    TD0_color_arg1mul_alpha1 |
                    TD0_color_add_add |
                    TD0_color_sel_add |
                    TD0_alpha_arg2_prevstage |
                    TD0_alpha_sel_arg2 );
         }
#else
         /* s/w fallback, pretty sure we can't do in h/w */
	 FALLBACK( ctx, MGA_FALLBACK_TEXTURE, GL_TRUE );
	 if ( MGA_DEBUG & DEBUG_VERBOSE_FALLBACK )
	    fprintf( stderr, "FALLBACK: GL_DECAL RGBA texture, unit=%d\n",
		     unit );
#endif
      }
      else {
	*reg = g400_alpha_combine[unit][MGA_DECAL];
      }
      break;

   case GL_ADD:
     if (format == GL_INTENSITY) {
	if (unit == 0) {
	   *reg = ( TD0_color_arg2_diffuse |
		   TD0_color_add_add |
		   TD0_color_sel_add |
		   TD0_alpha_arg2_diffuse |
		   TD0_alpha_add_enable |
		   TD0_alpha_sel_add);
	}
	else {
	   *reg = ( TD0_color_arg2_prevstage |
		   TD0_color_add_add |
		   TD0_color_sel_add |
		   TD0_alpha_arg2_prevstage |
		   TD0_alpha_add_enable |
		   TD0_alpha_sel_add);
	}
     }      
     else if (format == GL_ALPHA) {
	*reg = g400_alpha_combine[unit][MGA_ADD];
     }
     else {
	*reg = g400_color_combine[unit][MGA_ADD];
     }
     break;

   case GL_BLEND:
      if (format == GL_ALPHA) {
	 *reg = g400_alpha_combine[unit][MGA_BLEND];
      }
      else {
	 FALLBACK( ctx, MGA_FALLBACK_TEXTURE, GL_TRUE );
	 if ( MGA_DEBUG & DEBUG_VERBOSE_FALLBACK )
	    fprintf( stderr, "FALLBACK: GL_BLEND envcolor=0x%08x\n",
		     mmesa->envcolor );

         /* Do singletexture GL_BLEND with 'all ones' env-color
          * by using both texture units.  Multitexture gl_blend
          * is a fallback.
          */
         if (unit == 0) {
            /* Part 1: R1 = Rf ( 1 - Rt )
             *         A1 = Af At
             */
            *reg = ( TD0_color_arg2_diffuse |
                     TD0_color_arg1_inv_enable |
                     TD0_color_sel_mul |
                     TD0_alpha_arg2_diffuse |
                     TD0_alpha_sel_arg1);
         } else {
            /* Part 2: R2 = R1 + Rt
             *         A2 = A1
             */
            *reg = ( TD0_color_arg2_prevstage |
                     TD0_color_add_add |
                     TD0_color_sel_add |
                     TD0_alpha_arg2_prevstage |
                     TD0_alpha_sel_arg2);
         }
      }
      break;
   default:
      break;
   }
}


static void mgaUpdateTextureUnit( GLcontext *ctx, int hw_unit )
{
   mgaContextPtr mmesa = MGA_CONTEXT( ctx );
   struct gl_texture_unit *texUnit = &ctx->Texture.Unit[ mmesa->tmu_source[hw_unit] ];


   if ( texUnit->_ReallyEnabled == TEXTURE0_2D) {
      struct gl_texture_object	*tObj = texUnit->_Current;
      mgaTextureObjectPtr t = (mgaTextureObjectPtr) tObj->DriverData;

       /* Fallback if there's a texture border */
       if (tObj->Image[tObj->BaseLevel]->Border > 0) {
	   FALLBACK( ctx, MGA_FALLBACK_TEXTURE, GL_TRUE );
	   if ( MGA_DEBUG & DEBUG_VERBOSE_FALLBACK )
	       fprintf( stderr, "FALLBACK: texture border\n" );
	   return;
       }
       

       /* Upload teximages (not pipelined)
	*/
       if (t->base.dirty_images) {
	   FLUSH_BATCH( mmesa );
	   mgaSetTexImages( mmesa, tObj );
	   if ( t->base.memBlock == NULL ) {
	       FALLBACK( ctx, MGA_FALLBACK_TEXTURE, GL_TRUE );
	       return;
	   }
       }

       /* Update state if this is a different texture object to last
	* time.
	*/
       if ( mmesa->CurrentTexObj[hw_unit] != t ) {
	   if ( mmesa->CurrentTexObj[hw_unit] != NULL ) {
	       /* The old texture is no longer bound to this texture unit.
		* Mark it as such.
		*/

	       mmesa->CurrentTexObj[hw_unit]->base.bound &= ~(1UL << hw_unit);
	   }

	   mmesa->CurrentTexObj[hw_unit] = t;
	   t->base.bound |= (1UL << hw_unit);
	   driUpdateTextureLRU( (driTextureObject *) t ); /* done too often */
       }

       t->setup.texctl2 &= ~TMC_dualtex_enable;
       if (ctx->Texture._ReallyEnabled == (TEXTURE0_2D|TEXTURE1_2D)) 
	   t->setup.texctl2 |= TMC_dualtex_enable;

       t->setup.texctl2 &= ~TMC_specen_enable;
       if (ctx->Light.Model.ColorControl == GL_SEPARATE_SPECULAR_COLOR)
	   t->setup.texctl2 |= TMC_specen_enable;

       if (MGA_IS_G400(mmesa)) {
	  mgaUpdateTextureEnvG400( ctx, hw_unit );
       } else {
	  mgaUpdateTextureEnvG200( ctx, hw_unit );
       }
   }
   else if ( texUnit->_ReallyEnabled ) {
      FALLBACK( ctx, MGA_FALLBACK_TEXTURE, GL_TRUE );
      return;
   }
   else {
      /* Texture unit disabled */

      if ( mmesa->CurrentTexObj[hw_unit] != NULL ) {
	  /* The old texture is no longer bound to this texture unit.
	   * Mark it as such.
	   */

	  mmesa->CurrentTexObj[hw_unit]->base.bound &= ~(1UL << hw_unit);
	  mmesa->CurrentTexObj[hw_unit] = NULL;
      }
   }

   mmesa->dirty |= MGA_UPLOAD_CONTEXT | (MGA_UPLOAD_TEX0 << hw_unit);
}

/* The G400 is now programmed quite differently wrt texture environment.
 */
void mgaUpdateTextureState( GLcontext *ctx )
{
   mgaContextPtr mmesa = MGA_CONTEXT( ctx );
   FALLBACK( ctx, MGA_FALLBACK_TEXTURE, GL_FALSE );


   /* This works around a quirk with the MGA hardware.  If only OpenGL 
    * TEXTURE1 is enabled, then the hardware TEXTURE0 must be used.  The
    * hardware TEXTURE1 can ONLY be used when hardware TEXTURE0 is also used.
    */

   if (ctx->Texture._ReallyEnabled == TEXTURE1_2D) {
      mmesa->tmu_source[0] = 1;
   } else {
      mmesa->tmu_source[0] = 0;
   }

   mgaUpdateTextureUnit( ctx, 0 );
   if (MGA_IS_G400(mmesa)) {
      mmesa->setup.tdualstage1 = mmesa->setup.tdualstage0;

      mgaUpdateTextureUnit( ctx, 1 );
   }

   /* FIXME: I have a feeling that this should be in mgaUpdateTextureUnit. */

   mmesa->setup.dwgctl &= DC_opcod_MASK;
   mmesa->setup.dwgctl |= (ctx->Texture._ReallyEnabled
			   ? DC_opcod_texture_trap
			   : DC_opcod_trap);
}
