#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

#include <GL/glew.h>
#include <GL/glut.h>

#include "util.h"

enum {
	TEXTURE_PRIMARY,
	TEXTURE_SECONDARY,
	TEXTURE_MAX,
};

static GLuint program, textures[TEXTURE_MAX];
static double yaw;

void init_shaders(void)
{
	int vertex, fragment;
	const char *glsl_vertex, *glsl_fragment;

	glsl_vertex =
"void main(void)"
"{"
"	gl_Position = ftransform();"
"	gl_FrontColor = gl_Color;"
"	gl_TexCoord[0] = gl_TextureMatrix[0] * gl_MultiTexCoord0;"
"}";

	glsl_fragment =
"uniform bool useSecondary;"
"uniform sampler2D tex0, tex1;"
"void main(void)"
"{"
"	vec4 primary   = texture2D(tex0, gl_TexCoord[0].st);"
"	vec4 secondary = texture2D(tex1, gl_TexCoord[0].st);"
"	vec3 colour    = (1-primary.a)*gl_Color.rgb + primary.a*primary.rgb;"

	/*
	 * Removing the "if useSecondary" here (but keeping the
	 * multiplication) causes the shader to work.
	 * The failure does not depend on the value assigned to useSecondary.
	 */
"	if (useSecondary) {"
"		colour *= secondary.rgb;"
"	}"

"	gl_FragColor = vec4(colour, 1);"
"}";

	vertex   = create_shader(GL_VERTEX_SHADER,   glsl_vertex);
	fragment = create_shader(GL_FRAGMENT_SHADER, glsl_fragment);
	program  = create_program(2, vertex, fragment);
}

void init_textures(void)
{
	int rc;

	glGenTextures(TEXTURE_MAX, textures);
	rc = load_png_texture(textures[TEXTURE_PRIMARY],
		TEXTURE_MODE_REPEAT | TEXTURE_MODE_MIPMAP, "primary.png");
	if (rc == -1) {
		exit(EXIT_FAILURE);
	}

	rc = load_png_texture(textures[TEXTURE_SECONDARY],
		TEXTURE_MODE_REPEAT | TEXTURE_MODE_MIPMAP, "secondary.png");
	if (rc == -1) {
		exit(EXIT_FAILURE);
	}
}

void tick(double delta)
{
	yaw = remainder(yaw + 20*delta, 360);
	glutPostRedisplay();
}

void timeout(int val)
{
	static double lasttime = INFINITY;
	struct timespec tm;
	double current;

	if (clock_gettime(CLOCK_MONOTONIC, &tm) == -1) {
		perror("clock_gettime");
		return;
	}

	current = tm.tv_sec + (double) tm.tv_nsec / 1000000000;

	if (isfinite(lasttime))
		tick(current - lasttime);
	lasttime = current;

	glutTimerFunc(10, timeout, 0);
}

void print_gl_errors(int print)
{
	GLenum err;

	while ((err = glGetError()) != GL_NO_ERROR) {
		if (print) {
			printf("GL error: %s\n", gluErrorString(err));
		}
	}
}

void display(void)
{
	GLint var;

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();

	glTranslated(0, 0, -6);
	glRotated(yaw, 0, 1, 0);

	glColor3d(1,0,0);

	print_gl_errors(0);
	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, textures[TEXTURE_PRIMARY]);
	glActiveTexture(GL_TEXTURE1);
	glBindTexture(GL_TEXTURE_2D, textures[TEXTURE_SECONDARY]);
	print_gl_errors(1);

	glUseProgram(program);
	var = glGetUniformLocation(program, "tex0");
	glUniform1i(var, 0);
	var = glGetUniformLocation(program, "tex1");
	glUniform1i(var, 1);
	var = glGetUniformLocation(program, "useSecondary");
	glUniform1i(var, 1);

	draw_cube(2);

	glUseProgram(0);

	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, 0);
	glActiveTexture(GL_TEXTURE1);
	glBindTexture(GL_TEXTURE_2D, 0);

	glutSwapBuffers();
}

void reshape(int width, int height)
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glViewport(0, 0, width, height);
	gluPerspective(40.0, (double)width/height, 0.1, 1000.0);
	glMatrixMode(GL_MODELVIEW);
}

int main(int argc, char **argv)
{
	GLenum err;
	int window;

	glutInit(&argc, argv);
	glutInitDisplayMode(GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH);

	window = glutCreateWindow("GLTest");
	if (window == 0)
		return EXIT_FAILURE;
	glutSetWindow(window);

	err = glewInit();
	if (err != GLEW_OK) {
		fprintf(stderr, "glewInit: %s\n", glewGetErrorString(err));
		return EXIT_FAILURE;
	}

	init_shaders();
	init_textures();

	glEnable(GL_NORMALIZE);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_COLOR_MATERIAL);

	glActiveTexture(GL_TEXTURE0);
	glEnable(GL_TEXTURE_2D);
	glActiveTexture(GL_TEXTURE1);
	glEnable(GL_TEXTURE_2D);

	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	glClearColor(0.7, 0.7, 1.0, 0);

	glutDisplayFunc(display);
	glutReshapeFunc(reshape);
	glutTimerFunc(10, timeout, 0);

	glutMainLoop();
	return 0;
}
