#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <png.h>

#include <GL/glew.h>

#include "util.h"

static void draw_face(double size)
{
	glBegin(GL_QUADS);
	glNormal3d(0,0,1);
	glTexCoord2d(0, 0);
	glVertex3d(-size/2, -size/2, size/2);
	glTexCoord2d(1, 0);
	glVertex3d( size/2, -size/2, size/2);
	glTexCoord2d(1, 1);
	glVertex3d( size/2,  size/2, size/2);
	glTexCoord2d(0, 1);
	glVertex3d(-size/2,  size/2, size/2);
	glEnd();
}

void draw_cube(double size)
{
	glPushMatrix();

	draw_face(size);
	glRotated(90,  0, 1.0, 0);
	draw_face(size);
	glRotated(90,  0, 1.0, 0);
	draw_face(size);
	glRotated(90,  0, 1.0, 0);
	draw_face(size);
	glRotated(90,  1.0, 0, 0);
	draw_face(size);
	glRotated(180, 1.0, 0, 0);
	draw_face(size);

	glPopMatrix();
}

int load_png_texture(unsigned texture_id, unsigned mode, const char *filename)
{
	unsigned char *data = NULL;
	FILE *f;

	png_uint_32 width, height;
	int depth, ctype, itype;
	unsigned rowbytes;

	png_structp png;
	png_infop  info;

	png = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	if (!png) {
		fprintf(stderr, "failed to init libpng.\n");
		return -1;
	}

	info = png_create_info_struct(png);
	if (!info) {
		fprintf(stderr, "failed to init libpng.\n");
		png_destroy_read_struct(&png, NULL, NULL);
		return -1;
	}

	if (setjmp(png_jmpbuf(png))) {
		free(data);
		png_destroy_read_struct(&png, &info, NULL);
		return -1;
	}

	f = fopen(filename, "rb");
	if (!f) {
		char buf[128];
		snprintf(buf, sizeof buf, "%s: %s\n", filename,
			strerror(errno));
		png_error(png, buf);
		return -1;
	}

	png_init_io(png, f);
	png_read_info(png, info);
	png_get_IHDR(png, info, &width, &height, &depth, &ctype, &itype, 0, 0);

	if ((mode & TEXTURE_MODE_NO_ALPHA) && (ctype & PNG_COLOR_MASK_ALPHA)) {
		png_set_strip_alpha(png);
		ctype &= ~PNG_COLOR_MASK_ALPHA;
	}

	if (depth > 8)
		png_set_strip_16(png);
	png_set_expand(png);

	png_read_update_info(png, info);
	rowbytes = png_get_rowbytes(png, info);
	data = malloc(rowbytes * height);
	if (!data) {
		png_error(png, strerror(errno));
	}

	for (unsigned i = 0; i < height; i++) {
		png_read_row(png, data+i*rowbytes, NULL);
	}

	glPushAttrib(GL_TEXTURE_BIT);
	glBindTexture(GL_TEXTURE_2D, texture_id);
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

	if (mode & TEXTURE_MODE_REPEAT) {
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	} else {
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	}

	if (mode & TEXTURE_MODE_MIPMAP) {
		glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR_MIPMAP_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	} else {
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	}

	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	if (ctype & PNG_COLOR_MASK_ALPHA) {
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, width, height, 0,
			GL_RGBA, GL_UNSIGNED_BYTE, data);
	} else {
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, width, height, 0,
			GL_RGB, GL_UNSIGNED_BYTE, data);
	}
	glPopAttrib();

	free(data);
	png_destroy_read_struct(&png, &info, NULL);
	return 0;
}

unsigned create_shader(int type, const char *text)
{
	GLuint shader;
	GLint status;

	shader = glCreateShader(type);
	if (!shader)
		return 0;

	glShaderSource(shader, 1, &text, NULL);
	glCompileShader(shader);
	glGetShaderiv(shader, GL_COMPILE_STATUS, &status);
	if (!status) {
		GLint size;
		glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &size);

		char log[size];
		glGetShaderInfoLog(shader, size, NULL, log);
		fprintf(stderr, "failed to compile shader:\n%s", log);
		glDeleteShader(shader);
		return 0;
	}

	return shader;
}

unsigned create_program(unsigned num, ...)
{
	GLuint program;
	GLint status;
	va_list ap;

	program = glCreateProgram();

	va_start(ap, num);
	for (unsigned i = 0; i < num; i++) {
		glAttachShader(program, va_arg(ap, unsigned));
	}
	va_end(ap);

	glLinkProgram(program);
	glGetProgramiv(program, GL_LINK_STATUS, &status);
	if (!status) {
		GLint size;
		glGetProgramiv(program, GL_INFO_LOG_LENGTH, &size);

		char log[size];
		glGetShaderInfoLog(program, size, NULL, log);
		fprintf(stderr, "failed to link shader:\n%s", log);
		glDeleteProgram(program);
		return 0;
	}

	return program;
}
