package dbcodegen.db;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.dbcp.BasicDataSource;
import org.apache.empire.commons.ErrorObject;

import util.StringUtils;

public class Database {
	private BasicDataSource dataSource;
	private Connection connection;
	private String schemaName;
	
	private Map<String, Table> tableMap = new HashMap<String, Table>();

	public Database(String jdbcDriver, String dbUrl, 
			String userId, String password, String schemaName) {
		this.schemaName = schemaName;
		
		// Set up the data source
		dataSource = new BasicDataSource();
		dataSource.setDriverClassName(jdbcDriver);
		dataSource.setUrl(dbUrl);
		dataSource.setUsername(userId);
		dataSource.setPassword(password);	

		// Enable exceptions
		ErrorObject.setExceptionsEnabled(true);		
	}
	
	/**
	 * Gets DB connection.
	 * @return DB connection.
	 */
	public Connection getConnection() {
		if (this.connection == null) {
			try {
				this.connection = dataSource.getConnection();
			} catch (SQLException e) {
				e.printStackTrace();
				throw new RuntimeException(e);
			}
		}
		return this.connection;
	}

	/**
	 * Populates meta data for tables in the database.
	 * @param pkColName Primary key column name.  Note: We assume a single
	 * 		auto-generated PK column with the same name is used for every
	 * 		table in the DB.
	 * @param lockColName Lock column name used for optimistic locking.
	 * 		Note: We assume a single timestamp column with the same name 
	 * 		is used for every table in the DB.
	 */
	public void populateTableMetaData(String pkColName, String lockColName) {
		Table.setPkColName(pkColName);
		Table.setLockColName(lockColName);
		DatabaseMetaData dbMeta = this.getDbMetaData();
		try {
			ResultSet tables = dbMeta.getTables(null, this.schemaName, "", 
					new String[] {"TABLE"});
			while (tables.next()) {
				String tableName = tables.getString("TABLE_NAME");
				Table table = new Table(tableName, this.schemaName, dbMeta);
				this.tableMap.put(tableName.toUpperCase(), table);				
			}
			
			for (String tableName: this.tableMap.keySet()) {
				Table table = this.tableMap.get(tableName);
				for (Column fkCol: table.getFkCols()) {
					Table parentTable = this.tableMap.get(
							fkCol.getFkTableName());
					parentTable.addChildTable(fkCol.getName(), table);
				}
			}
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}
	/**
	 * Gets the database meta data.
	 * @return The database meta data.
	 */
	public DatabaseMetaData getDbMetaData() {
		try {
			return this.getConnection().getMetaData();
		} catch (SQLException e) {
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}
	
	public String getClassName() {
		return StringUtils.javaClassName(this.schemaName) + "Database";
	}
	public String getBaseTableClassName() {
		return StringUtils.javaClassName(this.schemaName) + "Table";
	}
	public String getInstanceName() {
		return StringUtils.deriveAttributeNameFromClass(this.getClassName());
	}
	public String getJdbcDriverName() {
		return this.dataSource.getDriverClassName();
	}
	public String getUrl() {
		return this.dataSource.getUrl();
	}
	public String getUserId() {
		return this.dataSource.getUsername();
	}
	public String getPassword() {
		return this.dataSource.getPassword();
	}

	public Collection<Table> getTables() {
		return tableMap.values();
	}
	public Table getTable(String name) {
		return this.tableMap.get(name.toUpperCase());
	}
	
	public String getSchemaName() {
		return schemaName;
	}
}
