/*
 *  Copyright (C) 2006 Martin Schoen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: ephy-rss-reader-extension.c,v 1.5 2005/01/28 22:30:17 chpe Exp $
 */

#include <gtk/gtkaction.h>
#include <gtk/gtkactiongroup.h>
#include <gtk/gtkuimanager.h>

#include "ephy-debug.h"

#include <epiphany/ephy-embed.h>
#include <epiphany/ephy-window.h>
#include <epiphany/ephy-tab.h>

#include "ephy-rss-reader-extension.h"
#include "feed-manager.h"
 
#define EPHY_RSS_READER_EXTENSION_GET_PRIVATE(object)(G_TYPE_INSTANCE_GET_PRIVATE ((object), EPHY_TYPE_RSS_READER_EXTENSION, EphyRssReaderExtensionPrivate))
static GObjectClass *parent_class = NULL;
static GType type = 0;

#define WINDOW_DATA_KEY "EphyRssReaderExtensionWindowData"

/**
 *		function prototypes
 */

static void feed_loaded_cb (guint position);
static void feeds_updated_cb (const gpointer tab, const gchar *file);
static void show_rss_page_cb (GtkAction *action, EphyWindow *window);

/**
 *		data structures
 */

static GtkActionEntry action_entries [] =
{
	{ "RSSPage",
	  NULL,
	  N_("News"),
	  NULL, /* shortcut key */
	  N_("Opens the RSS Summary."),
	  G_CALLBACK (show_rss_page_cb) }
};
static const guint n_action_entries = G_N_ELEMENTS (action_entries);

typedef struct
{
	GtkActionGroup *action_group;
	guint ui_id;
	EphyRssReaderExtension *extension;
} WindowData;
static void free_window_data (WindowData *data);

struct _EphyRssReaderExtensionPrivate
{
	FeedManager *manager;
	EphyTab *tab;
};

/**
 *		gobject methods
 */

static void
ephy_rss_reader_extension_init (EphyRssReaderExtension *extension)
{
	LOG ("EphyRssReaderExtension initialising");
	extension->priv = EPHY_RSS_READER_EXTENSION_GET_PRIVATE (extension);
	
	extension->priv->manager = g_object_new (TYPE_FEED_MANAGER, NULL);
	g_signal_connect (extension->priv->manager, 
							"feed-loaded", 
							G_CALLBACK (feed_loaded_cb), 
							extension->priv->manager);
	g_signal_connect (extension->priv->manager, 
							"feeds-updated", 
							G_CALLBACK (feeds_updated_cb), 
							extension->priv->manager);
	
	/*	add the feeds to the list
	 *
	 *		FIXME: add storage (gconf? opml?) support and ui
	 */
	
	feed_manager_add_feed (extension->priv->manager, "http://www.tagesschau.de/newsticker.rdf");
	feed_manager_add_feed (extension->priv->manager, "http://www.spiegel.de/schlagzeilen/rss/0,5291,,00.xml");
	feed_manager_add_feed (extension->priv->manager, "http://www.userfriendly.org/rss/uf.rss");
	feed_manager_add_feed (extension->priv->manager, "http://unununium.freepgs.com/bashrss/");
	feed_manager_add_feed (extension->priv->manager, "http://german-bash.org/latest-quotes.xml");
	feed_manager_add_feed (extension->priv->manager, "http://thedailywtf.com/rss.aspx");
	feed_manager_add_feed (extension->priv->manager, "http://feeds.feedburner.com/Bildblog");
	feed_manager_add_feed (extension->priv->manager, "http://www.daujones.com/daubeitraege.rss");
	feed_manager_add_feed (extension->priv->manager, "http://www.heise.de/newsticker/heise-atom.xml");
	feed_manager_add_feed (extension->priv->manager, "http://planet.gnome.org/news/rss20.xml");
	feed_manager_add_feed (extension->priv->manager, "http://planet.gnome.org/rss20.xml");
	feed_manager_add_feed (extension->priv->manager, "http://www.golem.de/rss.php?feed=RSS1.0");
	feed_manager_add_feed (extension->priv->manager, "http://www.gentoo.org/rdf/en/gentoo-news.rdf");
	feed_manager_add_feed (extension->priv->manager, "http://www.linuxtoday.com/backend/biglt.rss");
	feed_manager_add_feed (extension->priv->manager, "http://rss.slashdot.org/linux/KhmQ");
	feed_manager_add_feed (extension->priv->manager, "http://rss.slashdot.org/Slashdot/slashdot");
}

static void
ephy_rss_reader_extension_finalize (GObject *object)
{
	EphyRssReaderExtension *extension = EPHY_RSS_READER_EXTENSION (object);
	
	LOG ("EphyRssReaderExtension finalising");
	
	g_object_unref (extension->priv->manager);
		
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
impl_attach_window (EphyExtension *ext,
		    EphyWindow *window)
{
	LOG ("EphyRssReaderExtension attach_window");
	/* Add RSS item to "go" menu */
	GtkActionGroup *action_group;
	GtkUIManager *manager;
	guint merge_id;
	WindowData *data;

	data = g_new (WindowData, 1);
	
	manager = GTK_UI_MANAGER (ephy_window_get_ui_manager (window));

	data->action_group = action_group =
		gtk_action_group_new ("EphyRSSReaderExtensionActions");

	gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE);
	gtk_action_group_add_actions (action_group, action_entries,
				      n_action_entries, window);

	gtk_ui_manager_insert_action_group (manager, action_group, 0);

	data->ui_id = merge_id = gtk_ui_manager_new_merge_id (manager);

	gtk_ui_manager_add_ui (manager, merge_id, "/menubar/GoMenu",
			       "RSSPage", "RSSPage",
			       GTK_UI_MANAGER_MENUITEM, FALSE);

	data->extension = (EphyRssReaderExtension *)ext;
	
	g_object_set_data_full (G_OBJECT (window), WINDOW_DATA_KEY, data,
				(GDestroyNotify) free_window_data);
}

static void
impl_detach_window (EphyExtension *ext,
					EphyWindow *window)
{
	LOG ("EphyRssReaderExtension detach_window");
	
	/* Remove RSS item from "go" menu */
	GtkUIManager *manager;
	WindowData *data;

	manager = GTK_UI_MANAGER (ephy_window_get_ui_manager (window));

	data = (WindowData *) g_object_get_data (G_OBJECT (window), WINDOW_DATA_KEY);
	g_return_if_fail (data != NULL);

	gtk_ui_manager_remove_ui (manager, data->ui_id);
	gtk_ui_manager_remove_action_group (manager, data->action_group);

	g_object_set_data (G_OBJECT (window), WINDOW_DATA_KEY, NULL);
}

static void
impl_attach_tab (EphyExtension *ext,
					EphyWindow *window,
					EphyTab *tab)
{
}

static void
impl_detach_tab (EphyExtension *ext,
					EphyWindow *window,
					EphyTab *tab)
{
}

static void
ephy_rss_reader_extension_iface_init (EphyExtensionIface *iface)
{
	iface->attach_window = impl_attach_window;
	iface->detach_window = impl_detach_window;
	iface->attach_tab = impl_attach_tab;
	iface->detach_tab = impl_detach_tab;
}

static void
ephy_rss_reader_extension_class_init (EphyRssReaderExtensionClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = ephy_rss_reader_extension_finalize;
	
	g_type_class_add_private (object_class, sizeof (EphyRssReaderExtensionPrivate));
}

GType
ephy_rss_reader_extension_get_type (void)
{
	return type;
}

GType
ephy_rss_reader_extension_register_type (GTypeModule *module)
{
	static const GTypeInfo our_info =
	{
		sizeof (EphyRssReaderExtensionClass),
		NULL, /* base_init */
		NULL, /* base_finalize */
		(GClassInitFunc) ephy_rss_reader_extension_class_init,
		NULL,
		NULL, /* class_data */
		sizeof (EphyRssReaderExtension),
		0, /* n_preallocs */
		(GInstanceInitFunc) ephy_rss_reader_extension_init
	};

	static const GInterfaceInfo extension_info =
	{
		(GInterfaceInitFunc) ephy_rss_reader_extension_iface_init,
		NULL,
		NULL
	};

	type = g_type_module_register_type (module,
					    G_TYPE_OBJECT,
					    "EphyRssReaderExtension",
					    &our_info, 0);

	g_type_module_add_interface (module,
				     type,
				     EPHY_TYPE_EXTENSION,
				     &extension_info);

	return type;
}

/**
 * 	callbacks
 */

static void feed_loaded_cb (guint position)
{
	LOG ("feed_loaded_cb");
}

static void
feeds_updated_cb (const gpointer tab, const gchar *file)
{
	/*
	 * FIXME? rendering a large buffer does not work with gtkmozembed.
	 * See Bug #245960
	 * 
	 * FIXME: manual reload necessary (only once) to enable redraw on window resize
	 * 		this happens only when the rss tab is the first opened tab in the window
	 */
	
	/*EphyTab *rss_tab = (EphyTab*)tab;
	
	ephy_embed_load_url (ephy_tab_get_embed (rss_tab), file);*/
}

static void
show_rss_page_cb (GtkAction *action, 
					EphyWindow *window)
{
	WindowData *data = g_object_get_data (G_OBJECT (window), WINDOW_DATA_KEY);
	EphyRssReaderExtension* self = EPHY_RSS_READER_EXTENSION(data->extension);
	EphyTab *tab = ephy_window_get_active_tab (window);
	self->priv->tab = tab;
	feed_manager_refresh (self->priv->manager, tab);
}

/**
 *		internal (private) helper functions
 */

static void
free_window_data (WindowData *data)
{
	data->extension = NULL;
	g_object_unref (data->action_group);
	g_free (data);
}
