/*
 *  Copyright (C) 2006 Martin Schoen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "config.h"

#include <glib/gstdio.h>

#include <libxml/tree.h>
#include <libxml/xmlsave.h>

#include <epiphany/ephy-embed.h>
#include <epiphany/ephy-embed-factory.h>
#include <epiphany/ephy-embed-persist.h>
#include <epiphany/ephy-window.h>
#include <epiphany/ephy-tab.h>
#include <epiphany/ephy-statusbar.h>
#include "ephy-debug.h"

#include "feed-manager-marshal.h"
#include "feed-manager.h"
#include "page-builder.h"

#define FEED_MANAGER_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE ((object), TYPE_FEED_MANAGER, FeedManagerPrivate))

/** below this interval (in seconds) no refresh is done, even if requested */
#define REFRESH_INTERVAL_MIN	1800
/** TODO: feeds will automatically be refreshed after this time */
#define REFRESH_INTERVAL_MAX	3600*24

/**
 *		function prototypes
 */

static void feed_loaded_cb (EphyEmbedPersist* persist, gpointer data);

static void free_feed (gpointer data, gpointer user_data);
static void update_feeds (FeedManager *self);

/**
 *		data structures
 */

struct _FeedManagerPrivate
{
	PageBuilder *builder;
	EphyEmbed *embed;
	EphyEmbedPersist *persist;
	EphyTab *tab;
	GList *feeds, *current_feed;
	time_t last_update;
	gchar *temp_file;
};

static GObjectClass *parent_class = NULL;

static GType type = 0;

/**
 *		signals
 */

enum {
	FEED_LOADED,
	FEEDS_UPDATED,
	LAST_SIGNAL
};
static guint signals [LAST_SIGNAL] = { 0 };

/**
 *		gobject methods
 */

static void
feed_manager_init (FeedManager *manager)
{
	LOG ("FeedManager initializing");
	manager->priv = FEED_MANAGER_GET_PRIVATE (manager);
	
	char temp_name[] = "ephy-rss-reader-XXXXXX";
	mktemp (temp_name);
	manager->priv->temp_file = g_build_filename (g_get_tmp_dir (), 
	 												temp_name, 
	 												NULL);
	 
	manager->priv->persist = EPHY_EMBED_PERSIST(ephy_embed_factory_new_object (EPHY_TYPE_EMBED_PERSIST));
	ephy_embed_persist_set_flags (manager->priv->persist, 
										EPHY_EMBED_PERSIST_NO_VIEW |
										EPHY_EMBED_PERSIST_COPY_PAGE | 
										EPHY_EMBED_PERSIST_NO_CERTDIALOGS);
	ephy_embed_persist_set_dest (manager->priv->persist, manager->priv->temp_file);
	g_signal_connect (manager->priv->persist, 
							"completed", 
							G_CALLBACK (feed_loaded_cb), 
							manager);
	
	manager->priv->builder = g_object_new (TYPE_PAGE_BUILDER, NULL);
}

static void
feed_manager_finalize (GObject *object)
{
	LOG ("FeedManager finalizing");
	FeedManagerPrivate *priv = FEED_MANAGER_GET_PRIVATE (FEED_MANAGER (object));
	
	g_list_foreach (priv->feeds, &free_feed, NULL);
	g_list_free (priv->feeds);	
	
	g_object_unref (priv->builder);
	g_object_unref (priv->persist);
	g_unlink (priv->temp_file);
	g_free (priv->temp_file);
	
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
feed_manager_class_init (FeedManagerClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = feed_manager_finalize;
	
	g_type_class_add_private (object_class, sizeof (FeedManagerPrivate));
	
	signals[FEED_LOADED] = g_signal_new ("feed-loaded",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
				G_STRUCT_OFFSET (FeedManagerClass, feed_loaded),
				NULL,
				NULL,
				g_cclosure_marshal_VOID__UINT,
				G_TYPE_NONE,
				1,
				G_TYPE_UINT);
		signals[FEEDS_UPDATED] = g_signal_new ("feeds-updated",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
				G_STRUCT_OFFSET (FeedManagerClass, feeds_updated),
				NULL,
				NULL,
				g_cclosure_user_marshal_VOID__POINTER_STRING,
				G_TYPE_NONE,
				2,
				G_TYPE_POINTER,
				G_TYPE_STRING);
}

GType
feed_manager_get_type (void)
{
	return type;
}

GType
feed_manager_register_type (GTypeModule *module)
{
	static const GTypeInfo our_info =
	{
		sizeof (FeedManagerClass),
		NULL, /* base_init */
		NULL, /* base_finalize */
		(GClassInitFunc) feed_manager_class_init,
		NULL, /* class_finalize */
		NULL, /* class_data */
		sizeof (FeedManager),
		0, /* n_preallocs */
		(GInstanceInitFunc) feed_manager_init
	};

	type = g_type_module_register_type (module,
					    G_TYPE_OBJECT,
					    "FeedManager",
					    &our_info, 0);

	return type;
}

/**
 *		public methods
 */

void 
feed_manager_add_feed (FeedManager *self, const gchar *uri)
{
	Feed *feed = g_new (Feed, 1);
	feed->uri = g_strdup (uri);
	self->priv->feeds = g_list_append (self->priv->feeds, feed);
	LOG ("Added feed \"%s\"", feed->uri);
}

void 
feed_manager_refresh (FeedManager *self, EphyTab *tab)
{
	LOG ("Starting update");
	
	EphyWindow* window = (EphyWindow*) gtk_widget_get_toplevel ((GtkWidget*)tab);
	EphyStatusbar* statusbar = (EphyStatusbar*) ephy_window_get_statusbar (window);
	ephy_statusbar_set_progress (statusbar, 0);
	
	if (time (NULL) > (self->priv->last_update + REFRESH_INTERVAL_MIN))
	{
		self->priv->current_feed = g_list_first (self->priv->feeds);
	} else
	{
		self->priv->current_feed = NULL;
	}
	
	self->priv->tab = tab;
	ephy_embed_persist_set_embed (self->priv->persist, 
										ephy_tab_get_embed (tab));
	update_feeds (self);
}

/**
 *		callbacks
 */
 
static void
feed_loaded_cb (EphyEmbedPersist* persist, gpointer data)
{	
	LOG ("Feed loaded");
	FeedManager *self = (FeedManager*)data;
	
	/** load and process feed */
	Feed *current = (Feed*)self->priv->current_feed->data;
	xmlDocPtr feed = xmlParseFile (self->priv->temp_file);
	current->data = page_builder_process_feed (self->priv->builder, feed, self->priv->last_update);
	xmlFreeDoc (feed);
	
	guint current_number = g_list_position (self->priv->feeds, 
																										self->priv->current_feed);
	guint total_number = g_list_length (self->priv->feeds);
	
	EphyWindow* window = (EphyWindow*)gtk_widget_get_toplevel ((GtkWidget*)self->priv->tab);
	EphyStatusbar* statusbar = (EphyStatusbar*) ephy_window_get_statusbar (window);
	ephy_statusbar_set_progress (statusbar, (current_number+1)*100/total_number);
	g_signal_emit (self,
								signals[FEED_LOADED],
								0,
								(current_number-1)*100/total_number,
								NULL);
	
	self->priv->current_feed = g_list_next (self->priv->current_feed);
	update_feeds (self);
}

/**
 *		internal (private) helper functions
 */

static void
free_feed (gpointer data, gpointer user_data)
{
	Feed *feed = (Feed*)data;
	LOG ("Feed \"%s\" freed", feed->uri);
	g_free (feed->uri);
	g_return_if_fail (feed->data != NULL);
	xmlFreeDoc (feed->data);
}

static void update_feeds (FeedManager *self)
{
	if (self->priv->current_feed == NULL)
	{
		LOG ("Building page");
		/** end of list reached, feeds are all updated */
		xmlDocPtr page = page_builder_create_page (self->priv->builder, self->priv->feeds);
		
		g_return_if_fail (page != NULL);
		
		xmlSaveCtxtPtr context = xmlSaveToFilename (self->priv->temp_file, 
												"utf-8",
												XML_SAVE_FORMAT | XML_SAVE_NO_DECL);
		xmlSaveDoc (context, page);
		xmlSaveFlush (context);
		xmlSaveClose (context);
		xmlFreeDoc (page);
		
		self->priv->last_update = time (NULL);
		
		LOG ("Finished updating");
		
		ephy_embed_load_url (ephy_tab_get_embed (self->priv->tab), self->priv->temp_file);
		
		g_signal_emit (self,
									signals[FEEDS_UPDATED],
									0,
									self->priv->tab,
									self->priv->temp_file,
									NULL);
	} else
	{
		Feed *current = (Feed*)self->priv->current_feed->data;
		LOG ("Updating feed \"%s\"", current->uri);
		ephy_embed_persist_set_source (self->priv->persist, current->uri);
		ephy_embed_persist_save (self->priv->persist);
		LOG ("Feed loading");
	}
}
