/*
 *  Copyright (C) 2006 Martin Schoen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <gtk/gtk.h>

#include <epiphany/ephy-embed.h>
#include <epiphany/ephy-window.h>
#include <epiphany/ephy-statusbar.h>
#include <epiphany/ephy-tab.h>
#include <epiphany/ephy-bookmarks.h>
#include <epiphany/ephy-shell.h>

#include "ephy-rss-reader-extension.h"
#include "feed.h"
#include "feed-manager.h"

#include "ephy-debug.h"
 
#define EPHY_RSS_READER_EXTENSION_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE ((object), EPHY_TYPE_RSS_READER_EXTENSION, EphyRssReaderExtensionPrivate))

static GObjectClass *parent_class = NULL;
static GType type = 0;

#define WINDOW_DATA_KEY		"EphyRssReaderExtensionWindowData"
#define FEED_LIST_DATA_KEY	"EphyRssReaderExtensionFeedList"

#define STATUSBAR_ICON_FILENAME	"rss-icon-16.png"

/**
 *		function prototypes
 */

static void tab_changed_cb (EphyWindow *window, gpointer data);
static void page_content_changed_cb (EphyEmbed *embed,	gchar *uri, EphyWindow *window);

static void feed_found_cb (EphyEmbed *embed, gchar *type, gchar *title, gchar *address, gpointer user_data);
static void feed_selected_cb (GtkMenuItem *menuitem, gpointer user_data);

static void progress_changed_cb (FeedManager *manager, guint progress, gpointer user_data);
static void page_ready_cb (FeedManager *manager, const gchar *file, gpointer user_data);

static gboolean statusbar_icon_clicked_cb (GtkWidget *widget,
																								GdkEventButton *event,
																								EphyWindow *window,
																								gpointer user_data);
static void menu_item_clicked_cb (GtkAction *action, EphyWindow *window);

static void feed_list_free (GList *list);

/**
 *		data structures
 */

static GtkActionEntry action_entries [] =
{
	{ "RSSSummaryPage",
	  NULL,
	  N_("News"),
	  NULL,
	  N_("Shows the News Summary."),
	  G_CALLBACK (menu_item_clicked_cb) }
};
static const guint n_action_entries = G_N_ELEMENTS (action_entries);

typedef struct
{
	GtkActionGroup *action_group;
	GtkWidget *frame, *evbox;
	guint ui_id;
	EphyRssReaderExtension *extension;
} WindowData;
static void free_window_data (WindowData *data);

struct _EphyRssReaderExtensionPrivate
{
	FeedManager *manager;
	EphyTab *tab;
};

/**
 *		gobject methods
 */

static void
ephy_rss_reader_extension_init (EphyRssReaderExtension *extension)
{
	LOG ("EphyRssReaderExtension initialising");
	extension->priv = EPHY_RSS_READER_EXTENSION_GET_PRIVATE (extension);
	
	extension->priv->manager = g_object_new (TYPE_FEED_MANAGER, NULL);
	g_signal_connect (extension->priv->manager, 
												"progress-changed", 
												G_CALLBACK (progress_changed_cb), 
												extension);
	g_signal_connect (extension->priv->manager, 
												"page-ready", 
												G_CALLBACK (page_ready_cb), 
												extension);
}

static void
ephy_rss_reader_extension_finalize (GObject *object)
{
	EphyRssReaderExtension *extension = EPHY_RSS_READER_EXTENSION (object);
	
	LOG ("EphyRssReaderExtension finalising");
	
	g_signal_handlers_disconnect_by_func (extension->priv->manager, 
																							G_CALLBACK (progress_changed_cb),
																							extension);
	g_signal_handlers_disconnect_by_func (extension->priv->manager, 
																							G_CALLBACK (page_ready_cb),
																							extension);
	g_object_unref (extension->priv->manager);
	
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
impl_attach_window (EphyExtension *ext,
		    									EphyWindow *window)
{
	/** add RSS item to "go" menu */
	WindowData *data = g_new (WindowData, 1);
	
	GtkUIManager *manager = GTK_UI_MANAGER (ephy_window_get_ui_manager (window));

	GtkActionGroup *action_group = data->action_group = 
		gtk_action_group_new ("EphyRSSReaderExtensionActions");

	gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE);
	gtk_action_group_add_actions (action_group, 
																			action_entries,
				      														n_action_entries, 
				      														window);

	gtk_ui_manager_insert_action_group (manager, action_group, 0);

	guint merge_id = data->ui_id = gtk_ui_manager_new_merge_id (manager);

	gtk_ui_manager_add_ui (manager, 
															merge_id, 
															"/menubar/GoMenu",
															"RSSSummaryPage", 
															"RSSSummaryPage",
															GTK_UI_MANAGER_MENUITEM, 
															FALSE);

	data->extension = (EphyRssReaderExtension *) ext;
	
	/** create statusbar icon */			
	EphyStatusbar *statusbar = EPHY_STATUSBAR (ephy_window_get_statusbar (window));
	g_return_if_fail (statusbar != NULL);
	
	data->frame = gtk_frame_new (NULL);
	data->evbox = gtk_event_box_new ();
	gtk_event_box_set_visible_window (GTK_EVENT_BOX (data->evbox), FALSE);
	gtk_container_add (GTK_CONTAINER (data->frame), data->evbox);
	gtk_widget_show (data->evbox);
	
	gchar *filename = g_build_filename (SHARE_DIR, STATUSBAR_ICON_FILENAME, NULL);
	GtkWidget *icon = gtk_image_new_from_file (filename);
	g_free (filename);
	gtk_container_add (GTK_CONTAINER (data->evbox), icon);
	gtk_widget_show (icon);
	
	gchar *tooltip = g_strdup_printf (_("Add a News Feed from this Site to your Bookmarks."));
	gtk_tooltips_set_tip (statusbar->tooltips, data->evbox, tooltip, NULL);
	g_free (tooltip);
	
	ephy_statusbar_add_widget (statusbar, data->frame);
	
	g_signal_connect_after (data->evbox, 
															"button-press-event",
															G_CALLBACK (statusbar_icon_clicked_cb),
															window);
	g_signal_connect_after (window, "notify::active-tab",
															G_CALLBACK (tab_changed_cb), 
															NULL);
															
	g_object_set_data_full (G_OBJECT (window), 
														WINDOW_DATA_KEY, 
														data,
														(GDestroyNotify) free_window_data);
}

static void
impl_detach_window (EphyExtension *ext,
													EphyWindow *window)
{
	g_return_if_fail (window != NULL);
	
	g_signal_handlers_disconnect_by_func (window, 
																							G_CALLBACK (tab_changed_cb), 
																							NULL);
	
	/** remove RSS item from "go" menu */
	GtkUIManager *manager = GTK_UI_MANAGER (ephy_window_get_ui_manager (window));

	WindowData *data = (WindowData *) g_object_get_data (G_OBJECT (window), WINDOW_DATA_KEY);
	g_return_if_fail (data != NULL);

	gtk_ui_manager_remove_ui (manager, data->ui_id);
	gtk_ui_manager_remove_action_group (manager, data->action_group);

	/** remove statusbar icon */
	EphyStatusbar *statusbar = EPHY_STATUSBAR (ephy_window_get_statusbar (window));
	g_return_if_fail (statusbar != NULL);
	g_return_if_fail (data->frame != NULL);
	g_return_if_fail (data->evbox != NULL);	
	gtk_tooltips_set_tip (statusbar->tooltips, GTK_WIDGET (data->evbox), NULL, NULL);
	ephy_statusbar_remove_widget (statusbar, GTK_WIDGET (data->frame));
	g_signal_handlers_disconnect_by_func (data->evbox, 
																							G_CALLBACK (statusbar_icon_clicked_cb),
																							window);
	
	g_object_set_data (G_OBJECT (window), WINDOW_DATA_KEY, NULL);
}

static void
impl_attach_tab (EphyExtension *ext,
					EphyWindow *window,
					EphyTab *tab)
{
	EphyEmbed *embed = ephy_tab_get_embed (tab);
	g_signal_connect_after (embed, "ge-content-change",
														G_CALLBACK (page_content_changed_cb), 
														window);
	g_signal_connect_after (embed, 
														"ge-feed-link", 
														G_CALLBACK (feed_found_cb), 
														NULL);
}

static void
impl_detach_tab (EphyExtension *ext,
											EphyWindow *window,
											EphyTab *tab)
{
	EphyEmbed *embed = ephy_tab_get_embed (tab);
	
	g_signal_handlers_disconnect_by_func (embed, 
																							G_CALLBACK (page_content_changed_cb), 
																							window);
	g_signal_handlers_disconnect_by_func (embed, 
																							G_CALLBACK (feed_found_cb), 
																							NULL);
	
	g_object_set_data (G_OBJECT (tab), 
															FEED_LIST_DATA_KEY, 
															NULL);
}

static void
ephy_rss_reader_extension_iface_init (EphyExtensionIface *iface)
{
	iface->attach_window = impl_attach_window;
	iface->detach_window = impl_detach_window;
	iface->attach_tab = impl_attach_tab;
	iface->detach_tab = impl_detach_tab;
}

static void
ephy_rss_reader_extension_class_init (EphyRssReaderExtensionClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = ephy_rss_reader_extension_finalize;
	
	g_type_class_add_private (object_class, sizeof (EphyRssReaderExtensionPrivate));
}

GType
ephy_rss_reader_extension_get_type (void)
{
	return type;
}

GType
ephy_rss_reader_extension_register_type (GTypeModule *module)
{
	static const GTypeInfo our_info =
	{
		sizeof (EphyRssReaderExtensionClass),
		NULL,
		NULL,
		(GClassInitFunc) ephy_rss_reader_extension_class_init,
		NULL,
		NULL,
		sizeof (EphyRssReaderExtension),
		0,
		(GInstanceInitFunc) ephy_rss_reader_extension_init
	};

	static const GInterfaceInfo extension_info =
	{
		(GInterfaceInitFunc) ephy_rss_reader_extension_iface_init,
		NULL,
		NULL
	};

	type = g_type_module_register_type (module,
					    G_TYPE_OBJECT,
					    "EphyRssReaderExtension",
					    &our_info, 0);

	g_type_module_add_interface (module,
				     type,
				     EPHY_TYPE_EXTENSION,
				     &extension_info);

	return type;
}

/**
 * 	callbacks
 */

static void
tab_changed_cb (EphyWindow *window,
										gpointer user_data)
{
	EphyTab *tab = ephy_window_get_active_tab (window);
	
	GList *feed_list = (GList*) g_object_get_data (G_OBJECT (tab), FEED_LIST_DATA_KEY);
	
	WindowData *data = (WindowData *) g_object_get_data (G_OBJECT (window), WINDOW_DATA_KEY);
	
	if (g_list_length (feed_list) > 0)
	{
		gtk_widget_show (data->frame);
	} else 
	{
		gtk_widget_hide (data->frame);
	}
}

static void
page_content_changed_cb (EphyEmbed *embed,
															gchar *uri,
															EphyWindow *window)
{
	EphyTab *tab = ephy_tab_for_embed (embed);
	
	g_object_set_data (G_OBJECT (tab), FEED_LIST_DATA_KEY, NULL);
	
	WindowData *data = (WindowData *) g_object_get_data (G_OBJECT (window), WINDOW_DATA_KEY);
	gtk_widget_hide (data->frame);
}

static void
feed_found_cb (EphyEmbed *embed,
												gchar *type,
												gchar *title,
												gchar *address,
												gpointer user_data)
{
	LOG ("Feed found: \n\t%s\n\t%s\n\t%s", type, title, address);
	
	EphyTab *tab = ephy_tab_for_embed (embed);
	GList *feed_list = (GList*) g_object_steal_data (G_OBJECT (tab), FEED_LIST_DATA_KEY);
	
	Feed *feed = g_object_new (TYPE_FEED, NULL);
	g_object_set (G_OBJECT (feed), "title", title, "address", address, NULL);
	
	feed_list = g_list_append (feed_list, feed);
	g_object_set_data_full (G_OBJECT (tab), 
															FEED_LIST_DATA_KEY, 
															feed_list,
															(GDestroyNotify) feed_list_free);
	
	EphyWindow* window = (EphyWindow*)gtk_widget_get_toplevel ((GtkWidget*) tab);
	WindowData *data = (WindowData *) g_object_get_data (G_OBJECT (window), WINDOW_DATA_KEY);
	gtk_widget_show (data->frame);
}											

static void
feed_selected_cb (GtkMenuItem *menuitem, gpointer user_data)
{
	Feed *feed = (Feed*) user_data;
	
	feed_add_to_bookmarks (feed);
}

/*static gboolean
context_menu_cb (EphyEmbed *embed,
											EphyEmbedEvent *event,
											EphyWindow *window)
{	
	const GValue *value;
	const char *address;
	gboolean active = FALSE;
	
	WindowData *data = (WindowData*) g_object_get_data (G_OBJECT (window), WINDOW_DATA_KEY);
	g_return_val_if_fail (data != NULL, FALSE);
	
	GList *feed_list = (GList*) g_object_steal_data (G_OBJECT (tab), FEED_LIST_DATA_KEY);
	
	if ((ephy_embed_event_get_context (event) & EPHY_EMBED_CONTEXT_LINK) && (list != NULL))
	{
		 = ephy_embed_event_get (event, "link", address);
		address = g_value_get_string (value);
		
		active = rss_feedlist_contains (list, address);
		
		g_object_set(data->subscribe_action, "sensitive", active, "visible", active, NULL);	
	}
	
	return FALSE;
}*/

static void
progress_changed_cb (FeedManager *manager, 
													guint progress, 
													gpointer user_data)
{
	EphyRssReaderExtension *self = EPHY_RSS_READER_EXTENSION (user_data);
	
	EphyWindow* window = (EphyWindow*)gtk_widget_get_toplevel ((GtkWidget*) self->priv->tab);
	g_return_if_fail (window != NULL);
	EphyStatusbar* statusbar = (EphyStatusbar*) ephy_window_get_statusbar (window);
	g_return_if_fail (statusbar != NULL);
	ephy_statusbar_set_progress (statusbar, progress);
}

static void
page_ready_cb (FeedManager *manager, 
										const gchar *file, 
										gpointer user_data)
{
	/**
	 * FIXME? rendering a large buffer does not work with gtkmozembed.
	 * See Bug #245960
	 * 
	 * FIXME: manual reload necessary (only once) to enable redraw on window resize
	 * 		this happens only when the rss tab is the first opened tab in the window
	 */
	
	EphyRssReaderExtension *self = EPHY_RSS_READER_EXTENSION (user_data);
	ephy_embed_load_url (ephy_tab_get_embed (self->priv->tab), file);
}

static gboolean
statusbar_icon_clicked_cb (GtkWidget *widget,
																GdkEventButton *event,
																EphyWindow *window,
																gpointer user_data)
{
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (event != NULL, FALSE);
	
	if (event->type == GDK_BUTTON_PRESS)
	{
		EphyTab *tab = ephy_window_get_active_tab (window);
		
		GtkMenu *menu = (GtkMenu*) gtk_menu_new ();
		
		EphyShell *shell = ephy_shell_get_default ();
		EphyBookmarks *bookmarks = ephy_shell_get_bookmarks (shell);
		
		GList *feed_list = (GList*) g_object_get_data (G_OBJECT (tab), FEED_LIST_DATA_KEY);
		GList *current = g_list_first (feed_list);
		
		while (current != NULL)
		{
			Feed *feed = FEED (current->data);
			
			gchar *title;
			g_object_get (G_OBJECT (feed), "title", &title, NULL);
			gchar *label = g_strdup_printf (_("Add \"%s\" to your News Bookmarks"), title);
			GtkWidget *menu_item =  gtk_menu_item_new_with_label (label);
			g_free (label);
			
			g_signal_connect (menu_item, 
														"activate", 
														G_CALLBACK (feed_selected_cb), 
														current->data);
			gtk_menu_shell_append (GTK_MENU_SHELL (menu), menu_item);

			/** set insensitive if bookmark already exists */
			gboolean sensitive = feed_is_in_bookmarks (feed);
			gtk_widget_set_sensitive (menu_item, sensitive);
			
			gtk_widget_show (menu_item);
			
			current = g_list_next (current);
		}
		gtk_menu_popup (menu, NULL, NULL, NULL, NULL, 
													event->button, 
													event->time);
		return TRUE;
	}
	
	return FALSE;
}

static void
menu_item_clicked_cb (GtkAction *action, 
														EphyWindow *window)
{
	WindowData *data = g_object_get_data (G_OBJECT (window), WINDOW_DATA_KEY);
	EphyRssReaderExtension* self = EPHY_RSS_READER_EXTENSION (data->extension);
	
	EphyTab *tab = ephy_window_get_active_tab (window);
	self->priv->tab = tab;
	
	feed_manager_refresh (self->priv->manager);
}

/**
 *		internal (private) helper functions
 */

static void 
feed_list_free (GList *list)
{
	GList *current = g_list_first (list);
	while (current != NULL)
	{
		Feed *feed = current->data;
		g_object_unref (feed);
		current = g_list_next (current);
	}
	g_list_free (list);
}

static void
free_window_data (WindowData *data)
{
	g_object_unref (data->action_group);
	g_free (data);
}
