/*
 *  Copyright (C) 2006 Martin Schoen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "config.h"

#include <glib/gstdio.h>

#include <libxml/tree.h>
#include <libxml/xmlsave.h>

#include <epiphany/ephy-embed.h>
#include <epiphany/ephy-embed-factory.h>
#include <epiphany/ephy-embed-persist.h>
#include <epiphany/ephy-window.h>
#include <epiphany/ephy-tab.h>
#include <epiphany/ephy-statusbar.h>
#include <epiphany/ephy-bookmarks.h>
#include <epiphany/ephy-shell.h>

#include "feed-manager.h"
#include "page-builder.h"
#include "common.h"

#include "ephy-debug.h"

#define FEED_MANAGER_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE ((object), TYPE_FEED_MANAGER, FeedManagerPrivate))

/**
 *		function prototypes
 */

static void feed_loaded_cb (Feed *feed, gpointer user_data);
static void feed_manager_build_page (FeedManager *self);

/**
 *		data structures
 */

struct _FeedManagerPrivate
{
	PageBuilder *builder;
	GList *feeds;
	time_t last_update;
	gchar *temp_file;
};

static GObjectClass *parent_class = NULL;

static GType type = 0;

/**
 *		signals
 */

enum {
	PROGRESS_CHANGED,
	PAGE_READY,
	LAST_SIGNAL
};
static guint signals [LAST_SIGNAL] = { 0 };

/**
 *		gobject methods
 */

static void
feed_manager_init (FeedManager *manager)
{
	LOG ("FeedManager initializing");
	FeedManagerPrivate *priv = FEED_MANAGER_GET_PRIVATE (manager);
	
	char temp_name[] = "ephy-rss-reader-XXXXXX";
	mktemp (temp_name);
	priv->temp_file = g_build_filename (g_get_tmp_dir (), 
	 																					temp_name, 
	 																					NULL);
}

static void
feed_manager_finalize (GObject *object)
{
	LOG ("FeedManager finalizing");
	FeedManagerPrivate *priv = FEED_MANAGER_GET_PRIVATE (FEED_MANAGER (object));
	
	if (priv->feeds != NULL)
	{
		GList *current = g_list_first (priv->feeds);
		while (current != NULL)
		{
			g_object_unref (current->data);
			current = g_list_next (current);
		}
		g_list_free (priv->feeds);	
	}
	
	g_object_unref (priv->builder);
	
	if (priv->temp_file != NULL)
	{
		g_unlink (priv->temp_file);
		g_free (priv->temp_file);
	}
	
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
feed_manager_class_init (FeedManagerClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = feed_manager_finalize;
	
	g_type_class_add_private (object_class, sizeof (FeedManagerPrivate));
	
	signals[PROGRESS_CHANGED] = g_signal_new ("progress-changed",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
				G_STRUCT_OFFSET (FeedManagerClass, progress_changed),
				NULL,
				NULL,
				g_cclosure_marshal_VOID__UINT,
				G_TYPE_NONE,
				1,
				G_TYPE_UINT);
	signals[PAGE_READY] = g_signal_new ("page-ready",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
				G_STRUCT_OFFSET (FeedManagerClass, page_ready),
				NULL,
				NULL,
				g_cclosure_marshal_VOID__STRING,
				G_TYPE_NONE,
				1,
				G_TYPE_STRING);
}

GType
feed_manager_get_type (void)
{
	return type;
}

GType
feed_manager_register_type (GTypeModule *module)
{
	static const GTypeInfo our_info =
	{
		sizeof (FeedManagerClass),
		NULL,
		NULL,
		(GClassInitFunc) feed_manager_class_init,
		NULL,
		NULL,
		sizeof (FeedManager),
		0,
		(GInstanceInitFunc) feed_manager_init
	};

	type = g_type_module_register_type (module,
					    G_TYPE_OBJECT,
					    "FeedManager",
					    &our_info, 0);

	return type;
}

/**
 *		public methods
 */

void 
feed_manager_refresh (FeedManager *self)
{
	FeedManagerPrivate *priv = FEED_MANAGER_GET_PRIVATE (self);
	
	if (priv->builder == NULL)
	{
		priv->builder = g_object_new (TYPE_PAGE_BUILDER, NULL);
	}
	
	/** build feed list from bookmarks */
	EphyShell *shell = ephy_shell_get_default ();
	EphyBookmarks *bookmarks = ephy_shell_get_bookmarks (shell);
	EphyNode *keyword = ephy_bookmarks_find_keyword	(bookmarks,
																															RSS_BOOKMARK_KEYWORD,
																															FALSE);
	if (keyword != NULL)
	{
		EphyNode *bookmark_list = ephy_bookmarks_get_bookmarks (bookmarks);
		EphyNode *bookmark = ephy_node_get_nth_child (bookmark_list, 1);
		while (bookmark != NULL)
		{
			if (ephy_bookmarks_has_keyword (bookmarks, keyword, bookmark))
			{
				Feed *feed = g_object_new (TYPE_FEED, NULL);
				g_object_set (G_OBJECT (feed), "bookmark", bookmark, NULL);
				priv->feeds = g_list_append (priv->feeds, feed);
				g_signal_connect (feed, 
														"feed-loaded", 
														G_CALLBACK (feed_loaded_cb), 
														self);
			}
			bookmark = ephy_node_get_next_child (bookmark_list, bookmark);
		}
	}
	g_return_if_fail (priv->feeds != NULL);

	g_signal_emit (self,
										signals[PROGRESS_CHANGED],
										0,
										1,
										NULL);
	
	GList *current = g_list_first (priv->feeds);
	while (current != NULL)
	{
		feed_refresh (FEED (current->data));
		current = g_list_next (current);
	}
}

/**
 *		callbacks
 */

static void
feed_loaded_cb (Feed *feed, gpointer user_data)
{
	static guint feeds_loaded = 0;
	
	FeedManager *self = FEED_MANAGER (user_data);
	FeedManagerPrivate *priv = FEED_MANAGER_GET_PRIVATE (self);
	
	page_builder_process_feed (priv->builder, feed);
	
	feeds_loaded++;
	guint total_number = g_list_length (priv->feeds);
	
	g_signal_emit (self,
										signals[PROGRESS_CHANGED],
										0,
										feeds_loaded*100/total_number,
										NULL);
	
	if (feeds_loaded >= total_number)
	{
		feeds_loaded = 0;
		feed_manager_build_page (self);
		g_object_unref (priv->builder);
	}
}


/**
 *		internal (private) helper functions
 */

static void 
feed_manager_build_page (FeedManager *self)
{	
	LOG ("Building page.");
	
	FeedManagerPrivate *priv = FEED_MANAGER_GET_PRIVATE (self);
	
	xmlDocPtr page = page_builder_create_page (priv->builder, priv->feeds);
	g_return_if_fail (page != NULL);
	
	xmlSaveCtxtPtr context = xmlSaveToFilename (priv->temp_file, 
																											"utf-8",
																											XML_SAVE_FORMAT | XML_SAVE_NO_DECL);
	xmlSaveDoc (context, page);
	xmlSaveFlush (context);
	xmlSaveClose (context);
	xmlFreeDoc (page);
		
	g_signal_emit (self,
										signals[PAGE_READY],
										0,
										priv->temp_file,
										NULL);
	
	g_object_unref (priv->builder);
}
