/*
 *  Copyright (C) 2006 Martin Schoen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "config.h"

#include "feed.h"

#include <glib/gstdio.h>
#include <libxml/tree.h>
#include <epiphany/ephy-bookmarks.h>
#include <epiphany/ephy-shell.h>
#include <epiphany/ephy-embed.h>
#include <epiphany/ephy-embed-factory.h>
#include <epiphany/ephy-embed-persist.h>

#include "common.h"
#include "ephy-debug.h"

#define FEED_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE ((object), TYPE_FEED, FeedPrivate))

/** below this interval (in seconds) no refresh is done, even if requested */
#define REFRESH_INTERVAL_MIN	60*30
/** TODO: feeds will automatically be refreshed after this time */
#define REFRESH_INTERVAL_MAX	60*60*24

/**
 *		function prototypes
 */

static void feed_loaded_cb (EphyEmbedPersist *persist, gpointer user_data);
static void feed_load (Feed *self);

/**
 *		data structures
 */

enum {
	FEED_TITLE = 1,
	FEED_ADDRESS,
	FEED_BOOKMARK,
	FEED_XML_DATA,
	FEED_HTML_DATA
};

enum {
	FEED_LOADED,
	LAST_SIGNAL
};
static guint signals [LAST_SIGNAL] = { 0 };

struct _FeedPrivate
{
	gchar *title, *address;
	EphyNode *bookmark;
	xmlDocPtr xml_data, html_data;
	
	guint last_update, update_interval;
	
	EphyEmbedPersist *persist;
	gchar *temp_file;
};

static GObjectClass *parent_class = NULL;

static GType type = 0;

/**
 *		gobject methods
 */

static void
feed_init (Feed *feed)
{
	FeedPrivate *priv = FEED_GET_PRIVATE (FEED (feed));
	
	priv->last_update = 0;
	priv->update_interval = REFRESH_INTERVAL_MIN;
}

static void
feed_finalize (GObject *object)
{
	FeedPrivate *priv = FEED_GET_PRIVATE (FEED (object));
	
	g_free (priv->title);
	g_free (priv->address);
	xmlFreeDoc (priv->xml_data);
	xmlFreeDoc (priv->html_data);
	
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
feed_get_property (GObject *object,
											guint property_id,
											GValue *value,
											GParamSpec *pspec)
{
	Feed *self = FEED (object);
	FeedPrivate *priv = FEED_GET_PRIVATE (FEED (object));
	
	switch (property_id)
	{
 		case FEED_TITLE:
 			/** bookmark data always has precedence over local data */
 			if (priv->bookmark != NULL)
 			{
 				gchar *title = ephy_node_get_property_string (priv->bookmark, EPHY_NODE_BMK_PROP_TITLE);
 				g_value_set_string (value, title);
 			} else
 			{
 				g_value_set_string (value, priv->title);
 			}
 			break;
 		case FEED_ADDRESS:
 			if (priv->bookmark != NULL)
 			{
 				gchar *address = ephy_node_get_property_string (priv->bookmark, EPHY_NODE_BMK_PROP_LOCATION);
 				g_value_set_string (value, address);
 			} else
 			{
 				g_value_set_string (value, priv->address);
 			}
 			break;
		case FEED_BOOKMARK:
 			break;
		case FEED_XML_DATA:
			if (priv->xml_data == NULL)
 			{
 				feed_load (self);
 			}
			g_value_set_pointer (value, priv->xml_data);
 			break;
 		case FEED_HTML_DATA:
 			g_value_set_pointer (value, priv->html_data);
 			break;
 		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object,property_id,pspec);
	}
}

static void
feed_set_property (GObject *object,
											guint property_id,
											const GValue *value,
											GParamSpec *pspec)
{
	FeedPrivate *priv = FEED_GET_PRIVATE (FEED (object));
	
	switch (property_id)
	{
 		case FEED_TITLE:
 			g_free (priv->title);
			priv->title = g_value_dup_string (value);
 			break;
 		case FEED_ADDRESS:
 			g_free (priv->address);
			priv->address = g_value_dup_string (value);
 			break;
		case FEED_BOOKMARK:
			priv->bookmark = (EphyNode*) g_value_get_pointer (value);
 			break;
		case FEED_XML_DATA:
 			break;
 		case FEED_HTML_DATA:
 			priv->html_data = (xmlDocPtr) g_value_get_pointer (value);
 			break;
 		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
	}
}

static void
feed_class_init (FeedClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = feed_finalize;
	
	g_type_class_add_private (object_class, sizeof (FeedPrivate));
	
	GParamSpec *pspec;
	
	object_class->set_property = feed_set_property;
	object_class->get_property = feed_get_property;
	
	pspec = g_param_spec_string ("title",
																		"title",
																		"Set/Get Feed's Title",
																		"",
																		G_PARAM_READWRITE);
	g_object_class_install_property (object_class,
																			FEED_TITLE,
																			pspec);
	pspec = g_param_spec_string ("address",
																		"address",
																		"Set/Get Feed's Address",
																		"",
																		G_PARAM_READWRITE);
	g_object_class_install_property (object_class,
																			FEED_ADDRESS,
																			pspec);
	pspec = g_param_spec_pointer ("bookmark",
																		"bookmark",
																		"Get/Set Feed's Bookmark",
																		G_PARAM_READWRITE);
	g_object_class_install_property (object_class,
																				FEED_BOOKMARK,
																				pspec);
	pspec = g_param_spec_pointer ("xml-data",
																		"xml-data",
																		"Get Feed's XML Data",
																		G_PARAM_READABLE);
	g_object_class_install_property (object_class,
																				FEED_XML_DATA,
																				pspec);
	pspec = g_param_spec_pointer ("html-data",
																		"html-data",
																		"Get/Set Feed's HTML Data",
																		G_PARAM_READWRITE);
	g_object_class_install_property (object_class,
																				FEED_HTML_DATA,
																				pspec);
	
	signals[FEED_LOADED] = g_signal_new ("feed-loaded",
																								G_TYPE_FROM_CLASS (klass),
																								G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
																								G_STRUCT_OFFSET (FeedClass, feed_loaded),
																								NULL,
																								NULL,
																								g_cclosure_marshal_VOID__VOID,
																								G_TYPE_NONE,
																								0,
																								NULL);
}

GType
feed_get_type (void)
{
	return type;
}

GType
feed_register_type (GTypeModule *module)
{
	static const GTypeInfo our_info =
	{
		sizeof (FeedClass),
		NULL,
		NULL,
		(GClassInitFunc) feed_class_init,
		NULL,
		NULL,
		sizeof (Feed),
		0,
		(GInstanceInitFunc) feed_init
	};

	type = g_type_module_register_type (module,
					    G_TYPE_OBJECT,
					    "Feed",
					    &our_info, 0);

	return type;
}

/**
 *		public methods
 */

void
feed_add_to_bookmarks (Feed *self)
{
	FeedPrivate *priv = FEED_GET_PRIVATE (self);
	
	g_return_if_fail (priv->title != NULL);
	g_return_if_fail (priv->address != NULL);
	
	EphyShell *shell = ephy_shell_get_default ();
	EphyBookmarks *bookmarks = ephy_shell_get_bookmarks (shell);
	
	g_return_if_fail (ephy_bookmarks_find_bookmark (bookmarks, priv->address) == NULL);
	
	EphyNode *bookmark = ephy_bookmarks_add	(bookmarks,
																											priv->title,
																											priv->address);
	g_object_ref (bookmark);
	g_return_if_fail (bookmark != NULL);
	priv->bookmark = bookmark;	
	
	EphyNode *keyword = ephy_bookmarks_find_keyword	(bookmarks,
																															RSS_BOOKMARK_KEYWORD,
																															FALSE);
	if (keyword == NULL)
	{
		keyword = ephy_bookmarks_add_keyword (bookmarks, RSS_BOOKMARK_KEYWORD);
	}
	ephy_bookmarks_set_keyword	(bookmarks, 
																				keyword,
																				bookmark);
}

gboolean
feed_is_in_bookmarks (Feed *self)
{
	FeedPrivate *priv = FEED_GET_PRIVATE (self);
	
	g_return_val_if_fail (priv->bookmark == NULL, TRUE);
	g_return_val_if_fail (priv->address != NULL, FALSE);
	
	EphyShell *shell = ephy_shell_get_default ();
	EphyBookmarks *bookmarks = ephy_shell_get_bookmarks (shell);
	return (ephy_bookmarks_find_bookmark (bookmarks, priv->address) == NULL);
}

void feed_refresh (Feed *self)
{
	FeedPrivate *priv = FEED_GET_PRIVATE (self);
	
	if ((priv->xml_data == NULL) || ((time (NULL) + priv->update_interval) > priv->last_update))
	{
		feed_load (self);
	} else 
	{
		g_signal_emit (self,
										signals[FEED_LOADED],
										0,
										NULL);
	}
}

/**
 *		callbacks
 */

static void
feed_loaded_cb (EphyEmbedPersist *persist, gpointer user_data)
{	
	Feed *self = FEED (user_data);
	FeedPrivate *priv = FEED_GET_PRIVATE (self);
		
	priv->xml_data = xmlParseFile (priv->temp_file);
	
	g_unlink (priv->temp_file);
	g_object_unref (priv->persist);
	
	g_signal_emit (self,
										signals[FEED_LOADED],
										0,
										NULL);
}

/**
 *		internal (private) helper functions
 */

static void 
feed_load (Feed *self)
{
	FeedPrivate *priv = FEED_GET_PRIVATE (self);
	
	if (priv->temp_file == NULL)
	{
		char temp_name[] = "ephy-rss-reader-XXXXXX";
		mktemp (temp_name);
		priv->temp_file = g_build_filename (g_get_tmp_dir (), 
	 																					temp_name, 
	 																					NULL);
	}
	
	gchar *address;
	if (priv->bookmark != NULL)
	{
		address = ephy_node_get_property_string (priv->bookmark, EPHY_NODE_BMK_PROP_LOCATION);
	} else if (priv->address != NULL)
	{
		address = priv->address;
	} else 
	{
		return;
	}
	
	priv->persist = EPHY_EMBED_PERSIST (ephy_embed_factory_new_object (EPHY_TYPE_EMBED_PERSIST));
	ephy_embed_persist_set_flags (priv->persist, 
																			EPHY_EMBED_PERSIST_NO_VIEW |
																			EPHY_EMBED_PERSIST_NO_CERTDIALOGS);
	ephy_embed_persist_set_source (priv->persist, address);
	ephy_embed_persist_set_dest (priv->persist, priv->temp_file);
	g_signal_connect (priv->persist, 
												"completed", 
												G_CALLBACK (feed_loaded_cb), 
												self);
	ephy_embed_persist_save (priv->persist);
}
