/*
 *  Copyright (C) 2006 Martin Schoen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include "page-builder.h"

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <locale.h>

#include <glib/gi18n-lib.h>

#include <libxml/tree.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>
#include <libxslt/xslt.h>
#include <libxslt/transform.h>

#include "ephy-debug.h"

#define PAGE_BUILDER_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE ((object), TYPE_PAGE_BUILDER, PageBuilderPrivate))

#define STYLESHEET_FILENAME					"rss-stylesheet.xsl"
#define MENU_GENERATOR_FILENAME	"rss-menu-generator.xsl"
#define HTML_TEMPLATE_FILENAME		"rss-template.html"
#define RSS_ICON_FILENAME							"rss-icon.png"

/**
 *		function prototypes
 */

static gpointer load_files (gpointer data);
static time_t parse_date (const gchar* date);

/**
 *		data structures
 */

struct _PageBuilderPrivate
{
	xsltStylesheetPtr stylesheet, menu_generator;
	xmlDocPtr html_template;
	GHashTable *known_titles;
};

static GObjectClass *parent_class = NULL;

static GType type = 0;

/**
 *		gobject methods
 */

static void
page_builder_init (PageBuilder *builder)
{
	LOG ("PageBuilder initializing");
	builder->priv = PAGE_BUILDER_GET_PRIVATE (builder);
	
	/** initialize libxml */
	LIBXML_TEST_VERSION
	xmlInitParser ();
	xmlSubstituteEntitiesDefault (1);
	xmlLoadExtDtdDefaultValue = 1;
	xmlKeepBlanksDefault (1);
	
	/** load stylesheets and template in the background */
	g_thread_create (&load_files,
											builder,
											FALSE,
											NULL);
}

static void
page_builder_finalize (GObject *object)
{
	LOG ("PageBuilder finalizing");
	PageBuilderPrivate *priv = PAGE_BUILDER_GET_PRIVATE (PAGE_BUILDER (object));
	
	xsltFreeStylesheet (priv->stylesheet);
	xsltFreeStylesheet (priv->menu_generator);
	xmlFreeDoc (priv->html_template);
	g_hash_table_remove_all (priv->known_titles);
	
	xsltCleanupGlobals ();
	xmlCleanupParser ();
	
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
page_builder_class_init (PageBuilderClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = page_builder_finalize;
	
	g_type_class_add_private (object_class, sizeof (PageBuilderPrivate));
}

GType
page_builder_get_type (void)
{
	return type;
}

GType
page_builder_register_type (GTypeModule *module)
{
	static const GTypeInfo our_info =
	{
		sizeof (PageBuilderClass),
		NULL,
		NULL,
		(GClassInitFunc) page_builder_class_init,
		NULL,
		NULL,
		sizeof (PageBuilder),
		0,
		(GInstanceInitFunc) page_builder_init
	};

	type = g_type_module_register_type (module,
					    G_TYPE_OBJECT,
					    "PageBuilder",
					    &our_info, 0);

	return type;
}

/**
 *		public methods
 */

void
page_builder_process_feed (PageBuilder *self, Feed *feed)
{
	g_return_if_fail (self->priv->stylesheet != NULL);
	
	gchar *title;
	xmlDocPtr xml_data;
	g_object_get (G_OBJECT (feed), "title", &title, "xml-data", &xml_data, NULL);
	
	g_return_if_fail (xml_data != NULL);
	
	gchar* title_param = g_strconcat ("\'", title, "\'", NULL);
	const char *params[3] = {"feed-title", title_param, NULL};
	
	xmlDocPtr html_data = xsltApplyStylesheet (self->priv->stylesheet, 
																									xml_data, 
																									params);
	g_return_if_fail (html_data != NULL);
	
	/** format entry date */
	xmlXPathContextPtr xpathCtx = xmlXPathNewContext (html_data);
	xmlXPathObjectPtr xpathObj = xmlXPathEvalExpression ("/div/div/span[attribute::class='date']", 
																																xpathCtx);
	gint size = (xpathObj->nodesetval) ? xpathObj->nodesetval->nodeNr : 0;
	gint i = 0;
	for(i = size - 1; i >= 0; i--)
	{
		gchar* date = xmlNodeGetContent (xpathObj->nodesetval->nodeTab[i]);
		time_t entry_date = parse_date (date);
		gchar format_date[30];
		strftime (format_date, 29, "%c", localtime (&entry_date));
		xmlNodeSetContent (xpathObj->nodesetval->nodeTab[i], 
															format_date);
	}
	xmlXPathFreeObject(xpathObj);
	
	/** mark old entries */
	xpathObj = xmlXPathEvalExpression ("/div/div/div[attribute::class='entrytitle']", 
																						xpathCtx);
	size = (xpathObj->nodesetval) ? xpathObj->nodesetval->nodeNr : 0;
	i = 0;
	for(i = size - 1; i >= 0; i--)
	{
		title = xmlNodeGetContent (xpathObj->nodesetval->nodeTab[i]);
		gboolean *obsolete = (gboolean*)g_hash_table_lookup (self->priv->known_titles, title);
		if (obsolete == NULL)
		{
			// not in table, insert 
			gboolean *obs = g_malloc (sizeof (gboolean));
			*obs = FALSE;
			gchar *key = g_strdup (title);
			g_hash_table_insert (self->priv->known_titles, key, obs);
		} else
		{
			xmlNodePtr cur = xpathObj->nodesetval->nodeTab[i]->parent;
			xmlSetProp (cur, "class", "entry_hidden");
			*obsolete = FALSE;
		}
	}
	xmlXPathFreeObject(xpathObj);
	
	xmlXPathFreeContext(xpathCtx);
	
	g_object_set (G_OBJECT (feed), "html-data", html_data, NULL);
}

xmlDocPtr page_builder_create_page (PageBuilder *self, const GList *feeds)
{
	g_return_val_if_fail (self->priv->menu_generator != NULL, NULL);
	g_return_val_if_fail (self->priv->html_template != NULL, NULL);
	
	/** build page contents */
	xmlDocPtr page_content = xmlNewDoc ("1.0");
	xmlNodePtr page_content_root = xmlNewNode (NULL, "div");
	xmlNewProp (page_content_root, "id", "content");
	xmlDocSetRootElement (page_content, page_content_root);
	
	GList *current_feed = g_list_first (feeds);
	while (current_feed != NULL)
	{
		xmlDocPtr html_data;
		g_object_get (G_OBJECT (current_feed->data), "html-data", &html_data, NULL);
		
		if (html_data != NULL)
		{
			xmlNodePtr html_data_copy = xmlDocCopyNode (xmlDocGetRootElement (html_data), 
																														page_content, 
																														1);
			xmlAddChild (page_content_root, html_data_copy);
			
			/** html data is no longer needed */
			xmlFreeDoc (html_data);
			g_object_set (G_OBJECT (current_feed->data), "html-data", NULL, NULL);
		}
		current_feed = g_list_next (current_feed);
	}
	
	/** copy template to document */
	xmlDocPtr page = xmlNewDoc ("1.0");
	xmlNodePtr page_root = xmlDocCopyNode (xmlDocGetRootElement (self->priv->html_template), 
																									page, 
																									1);
	xmlDocSetRootElement (page, page_root);
	xmlNodePtr body = page_root->last->prev;
	
	/** add menu */
	gchar* filename = g_build_filename (SHARE_DIR, RSS_ICON_FILENAME, NULL);
	gchar* icon_file = g_strconcat ("\'", filename, "\'", NULL);
	g_free (filename);
	
	gchar* show_all = g_strconcat ("\'", _("All Feeds"), "\'", NULL);
	gchar* show_old = g_strconcat ("\'", _("show old Items"), "\'", NULL);
	const char *params[] = {"rss-icon", icon_file, "show-all", show_all, "show-old", show_old, NULL};
	xmlDocPtr menu = xsltApplyStylesheet (self->priv->menu_generator, 
																							page_content, 
																							params);
	xmlNodePtr menu_content = xmlDocCopyNode (xmlDocGetRootElement (menu), 
																											page, 
																											1);
	xmlAddChild (body, menu_content);
	g_free (icon_file);
	g_free (show_all);
	g_free (show_old);

	/** add contents */
	xmlNodePtr content = xmlDocCopyNode (page_content_root, 
																								page, 
																								1);
	xmlAddChild (body, content);
	
	xmlFreeDoc (page_content);
	xmlFreeDoc (menu);
	return page;
}

/**
 *		internal (private) helper functions
 */
 
static gpointer
load_files (gpointer data)
{
	PageBuilder* self = (PageBuilder*)data;
	
	gchar* filename;
	
	filename = g_build_filename (SHARE_DIR, STYLESHEET_FILENAME, NULL);
	self->priv->stylesheet = xsltParseStylesheetFile (filename);
	g_free (filename);
	
	filename = g_build_filename (SHARE_DIR, MENU_GENERATOR_FILENAME, NULL);
	self->priv->menu_generator = xsltParseStylesheetFile (filename);
	g_free (filename);
	
	filename = g_build_filename (SHARE_DIR, HTML_TEMPLATE_FILENAME, NULL);
	self->priv->html_template = xmlParseFile (filename);
	g_free (filename);
	
	if (self->priv->stylesheet && self->priv->menu_generator && self->priv->html_template)
	{
		LOG ("Stylesheets loaded");
	} else
	{
		LOG ("Error loading stylesheets");
	}
	
	/** load list of known entry titles */
	self->priv->known_titles = g_hash_table_new_full (g_str_hash, 
																														g_str_equal,
																														g_free,
																														g_free);
	/** TODO: parse list
	for ()
	{
		gchar *key = title;
		
		gboolean *obs = g_malloc (sizeof (gboolean));
		*obs = TRUE;
		g_hash_table_insert (self->priv->known_titles, key, obs);	
	}*/
	return NULL;
}

static time_t
parse_date (const gchar* date_string)
{
	const gchar* pos = NULL;
	struct tm *date;
	
	/**
	 * 	formats are:
	 *			rfc3339 format: 					YYYY-MM-DDThh:mm:ss[.s](Z/(+/-)hh:mm)
	 *			rfc822 format:						[WWW,] [D]D MMM [YY] hh:mm[:ss] (TTT/(+/-)hhmm)
	 *
	 *		two standards, no one cares :(
	 *			modified rfc822 formats:	[WWW,] MMM [D]D ([YY]/[YYYY]) hh:mm[:ss] (TTT/(+/-)hhmm)
	 *																MMM [D]D[ext], hh:mm:ss
	 */
	
	if (date_string == NULL || strlen (date_string) == 0)
	{
		return time (NULL);
	}
	
	time_t loctime = time (NULL);
	date = localtime (&loctime);
	
	/** set locale to english */
	gchar *locale = g_strdup (setlocale (LC_TIME, NULL));
	setlocale (LC_TIME, "C");
	
	/** skip leading day name */
	if ((pos = g_utf8_strchr (date_string, -1, ',')) != NULL)
	{
			pos++;
	} else 
	{
		pos = date_string;
	}
	
	static gchar *formats[] = {"%t%Y-%m-%dT%T%t",		/** rfc3999 formats */
														"%t%Y-%m-%dT%T%t",
														"%t%Y-%m-%dT%R%t",
														"%t%Y-%m-%d%t",
														"%t%d %b %y %T%t",			/** rfc822 formats */
														"%t%d %b %y %R%t",
														"%t%d %b %Y %T%t",
														"%t%d %b %Y %R%t",
														"%t%b %d %y %T%t",
														"%t%b %d %y %R%t",
														"%t%b %d %Y %T%t",
														"%t%b %d %Y %R%t",
														"%t%b %d %T%t",
														"%t%b %d %R%t",
														"%t%R%t",
														"%t%T%t"};
	
	gchar *result = NULL;
	gint i = 0;
	while  (result == NULL && i < 16)
	{
		result = strptime (pos, formats[i], date);
		i++;
	}
	
	setlocale (LC_TIME, locale);
	g_free (locale);
	
	static struct {
		gchar* name;
		gint offset;
	} timezone[] = {
					{"UT", 0},
					{"GMT", 0},
					{"EST", -5},
					{"EDT", -4},
					{"CST", -6},
					{"CDT", -5},
					{"MST", -7},
					{"MDT", -6},
					{"PST", -8},
					{"PDT", -7},
					{"Z", 0},
					{"A", -1},
					{"M", -12},
					{"N", 1},
					{"Y", 12}};
	
	/** handle timezone */
	/*if ((pos = g_utf8_strrchr (date_string, -1, '+')))
	{
		tm.tm_gmtoff = (atoi (pos[1]) * 10 + atoi (pos[2]))*3600;
	} else if ((pos = g_utf8_strrchr (date_string, -1, '-')))
	{
		tm.tm_gmtoff = (-1)*(atoi (pos[1]) * 10 + atoi (pos[2]))*3600;
	} else
	{
		tm.tm_gmtoff = 0;
		gint index = 0;
		for (index = 0; index < 15; index++)
		{
			if (strncmp (pos, timezone[index].name, strlen (timezone[index].name)) == 0)
			{
				tm.tm_gmtoff = timezone->offset*3600;
			}
		}
	}*/
	return mktime (date);
}
