/*
  cx2100psu.c - A user-space program to interact with a CX2100 power supply
                unit, LCD and nav buttons
  
  Copyright (C) 2013  Graeme Foot, Kinetic Engineering Design Ltd <graeme@touchcut.com>
 
  based on code from I2C TOOLS FOR LINUX

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
  MA 02110-1301 USA.
*/

#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "i2c-dev.h"


#define VERSION "1.0.0"



/** print out usage information
 */
static void usage(void) __attribute__ ((noreturn));
static void usage(void)
{
  fprintf(stderr,
    "Usage:\n"
    "  cx2100_lcd [-Vhvp(L|l)b] [-t(1|2) text]\n"
    "  cx2100_lcd [--versionInfo] [--powerInfo] [--light(On|Off)] [--buttons] [--lcdLine(1|2) text]\n"
    "\n"
    "  -V                 Application version\n"
    "  -h                 Usage information\n"
    "  -v, --versionInfo  Display the power modules Type, Serial No and Firware Version\n"
    "  -p, --powerInfo    Display the power modules voltage, temperature, current and power values\n"
    "  -L, --lightOn      Turn the LCD backlight on\n"
    "  -l, --lightOff     Turn the LCD backlight off\n"
    "  -b, --buttons      Return the current state of the nav buttons where:\n"
    "                       0x01 - Right button pressed\n"
    "                       0x02 - Left button pressed\n"
    "                       0x04 - Down button pressed\n"
    "                       0x08 - Up button pressed\n"
    "                       0x10 - Enter button pressed\n"
    "  -tX, --lcdLineX    Set the text on line 1/2\n"
    "                       text - The text to set.  If the text has spaces, enclose\n"
    "                              it in quotes.  The text can only be a maximum of 16\n"
    "                              ASCII characters (extra characters will be ignored)\n");
  exit(1);
}



/** open an i2c device file
 */
static int open_i2c_dev(int i2cbus, char *filename, size_t size)
{
  int file;

  // try opening using /dev/i2c-# format
  snprintf(filename, size, "/dev/i2c-%d", i2cbus);
  filename[size - 1] = '\0';
  file = open(filename, O_RDWR);

  if ( (file < 0) && ( (errno == ENOENT) || (errno == ENOTDIR) ) )
  {
    // retry using /dev/i2c/# format
    sprintf(filename, "/dev/i2c/%d", i2cbus);
    file = open(filename, O_RDWR);
  }

  if (file < 0)
  {
    if (errno == ENOENT)
    {
      fprintf(stderr, "Error: Could not open file `/dev/i2c-%d' or `/dev/i2c/%d': %s\n",
              i2cbus, i2cbus, strerror(ENOENT));
    }
    else
    {
      fprintf(stderr, "Error: Could not open file `%s': %s\n", filename, strerror(errno));
      if (errno == EACCES)
      {
        fprintf(stderr, "Run as root?\n");
      }
    }
  }

  return file;
}



#define MISSING_FUNC_FMT	"Error: Device does not have %s capability\n"

/** check device capabilities
 * returning whether the device supports PEC
 */
static int checkCapabilites(int file, int *supportsPEC)
{
  unsigned long int funcs;

  // get device functionality
  if (ioctl(file, I2C_FUNCS, &funcs) < 0) 
  {
    fprintf(stderr, "Error: Could not get the device functionality matrix: %s\n", 
            strerror(errno));
    return -1;
  }

  // we want to be able to read bytes/words/blocks
  // we want to be able to write bytes/blocks
  if (!(funcs & I2C_FUNC_SMBUS_READ_BYTE)) 
  {
    fprintf(stderr, MISSING_FUNC_FMT, "SMBus receive byte");
    return -1;
  }
  else if (!(funcs & I2C_FUNC_SMBUS_WRITE_BYTE)) 
  {
    fprintf(stderr, MISSING_FUNC_FMT, "SMBus send byte");
    return -1;
  }
  
  else if (!(funcs & I2C_FUNC_SMBUS_READ_BYTE_DATA)) 
  {
    fprintf(stderr, MISSING_FUNC_FMT, "SMBus read byte");
    return -1;
  }
  else if (!(funcs & I2C_FUNC_SMBUS_WRITE_BYTE_DATA))
  {
    fprintf(stderr, MISSING_FUNC_FMT, "SMBus write byte");
    return -1;
  }
  
  else if (!(funcs & I2C_FUNC_SMBUS_READ_WORD_DATA)) 
  {
    fprintf(stderr, MISSING_FUNC_FMT, "SMBus read word");
    return -1;
  }
  else if (!(funcs & I2C_FUNC_SMBUS_WRITE_WORD_DATA)) 
  {
    fprintf(stderr, MISSING_FUNC_FMT, "SMBus write word");
    return -1;
  }
  
  else if (!(funcs & I2C_FUNC_SMBUS_READ_BLOCK_DATA)) 
  {
    fprintf(stderr, MISSING_FUNC_FMT, "SMBus read block");
    return -1;
  }
  else if (!(funcs & I2C_FUNC_SMBUS_WRITE_BLOCK_DATA)) 
  {
    fprintf(stderr, MISSING_FUNC_FMT, "SMBus write block");
    return -1;
  }
  

  // check if the module supports PEC (Packet Error Checking)
  *supportsPEC = (funcs & (I2C_FUNC_SMBUS_PEC | I2C_FUNC_I2C));
  

  return 0;
}



/** set the slave address, and choose whether to allow forcing
 * 
 * forcing is required to allow read/write when it is already attached to 
 * a driver
 */
static int setSlaveAddr(int file, int address, int force)
{
  if (ioctl(file, force ? I2C_SLAVE_FORCE : I2C_SLAVE, address) < 0)
  {
    fprintf(stderr, "Error: Could not set address to 0x%02x: %s\n",
            address, strerror(errno));
    return -errno;
  }

  return 0;
}



/** read and output a DWord
 */
static int readDWord(int file, int dataAddr, const char *paramName)
{
  unsigned char block[I2C_SMBUS_BLOCK_MAX];
  unsigned int  value;
  
  if (i2c_smbus_read_block_data(file, dataAddr, block) < 0)
  {
    fprintf(stderr, "Error: Could not get %s: %s\n", paramName, strerror(errno));
    return -errno;
  }
  
  value = block[0] + (block[1] << 8) + (block[2] << 16) + (block[3] << 24);
  fprintf(stdout, "%s: 0x%08x %u\n", paramName, value, value);
  
  return 0;
}


/** read and output a Word
 */
static int readWord(int file, int dataAddr, const char *paramName)
{
  int value;
  
  value = i2c_smbus_read_word_data(file, dataAddr);
  if (value < 0)
  {
    fprintf(stderr, "Error: Could not get %s: %s\n", paramName, strerror(errno));
    return -errno;
  }
  fprintf(stdout, "%s: 0x%04x %u\n", paramName, value, value);
  
  return 0;
}


/** read and output a Byte
 */
static int readByte(int file, int dataAddr, const char *paramName, int isSigned)
{
  int value;
  
  value = i2c_smbus_read_byte_data(file, dataAddr);
  if (value < 0)
  {
    fprintf(stderr, "Error: Could not get %s: %s\n", paramName, strerror(errno));
    return -errno;
  }
  
  if (isSigned) fprintf(stdout, "%s: %hhd\n", paramName, (char)(value & 0xFF));
  else          fprintf(stdout, "%s: 0x%02x %u\n", paramName, value, value);
  
  return 0;
}


/** write an LCD line, Note: max 16 chars + \0
 */
static int writeLCDLine(int file, int dataAddr, const char *paramName, char *lineText)
{
  unsigned char block[I2C_SMBUS_BLOCK_MAX];
  int idx = 0;
  
  while ( (*lineText != '\0') && (idx < 16) )
  {
    block[idx] = *lineText;
    lineText++;
    idx++;
  }
  
  // add null char
  block[idx] = '\0';
  idx++;
  
  // write data
  if (i2c_smbus_write_block_data(file, dataAddr, idx, block) < 0)
  {
    fprintf(stderr, "Error: Could not set %s: %s\n", paramName, strerror(errno));
    return -errno;
  }
  
  return 0;
}


    
/** app entry point
 */
int main(int argc, char *argv[])
{
  // i2c vars
  int   i2cbus;
  int   address;
  int   file;
  char  filename[20];
  int   usePEC;
  int   value;
  unsigned char byte;
  
  // param indecies
  int   paramIdx;
  char *charPtr;
  
  // options
  int   force       = 0;
  int   versionInfo = 0;
  int   powerInfo   = 0;
  int   lightOnOff  = -1;
  int   buttons     = 0;
  int   line1       = 0;
  int   line2       = 0;
  char *line1Text;
  char *line2Text;
  char **textX;
  
  
  // parse command line
  paramIdx = 1;
  while (paramIdx < argc)
  {
    // double --?
    if      (strcmp(argv[paramIdx], "--force") == 0)       force = 1;
    else if (strcmp(argv[paramIdx], "--versionInfo") == 0) versionInfo = 1;
    else if (strcmp(argv[paramIdx], "--powerInfo") == 0)   powerInfo = 1;
    else if (strcmp(argv[paramIdx], "--lightOn") == 0)     lightOnOff = 1;
    else if (strcmp(argv[paramIdx], "--lightOff") == 0)    lightOnOff = 0;
    else if (strcmp(argv[paramIdx], "--buttons") == 0)     buttons = 1;
    else if (strncmp(argv[paramIdx], "--lcdLine", 9) == 0)
    {
      // which line
      if (argv[paramIdx][9] == '1')
      {
        line1 = 1;
        textX = &line1Text;
      }
      else if (argv[paramIdx][9] == '2')
      {
        line2 = 1;
        textX = &line2Text;
      }
      else
      {
        fprintf(stderr, "Error: Unsupported lcdLine number \"%c\"!\n", argv[paramIdx][9]);
        usage();
        exit(1);
      }
        
      // get text param
      paramIdx++;
      if (paramIdx >= argc)
      {
        fprintf(stderr, "Error: Missing lcdLine text!\n");
        usage();
        exit(1);
      }
      
      *textX = argv[paramIdx];
    }
    else if (strncmp(argv[paramIdx], "--", 2) == 0)
    {
      fprintf(stderr, "Error: Unsupported option \"%s\"!\n", argv[paramIdx]);
      usage();
      exit(1);
    }
    
    // single -?
    else if (argv[paramIdx][0] == '-')
    {
      charPtr = argv[paramIdx];
      charPtr++;
      do
      {
        // whats the char
        switch (*charPtr)
        {
          case 'V' :
          {
            // display app version
            fprintf(stderr, "cx2100_lcd version %s\n", VERSION);
            exit(0);
          } break;
          
          case 'h' :
          {
            // display usage information
            usage();
            exit(0);
          } break;
          
          case 'f' : { force = 1; } break;
          case 'v' : { versionInfo = 1; } break;
          case 'p' : { powerInfo = 1; } break;
          case 'L' : { lightOnOff = 1; } break;
          case 'l' : { lightOnOff = 0; } break;
          case 'b' : { buttons = 1; } break;
          
          case 't' :
          {
            // which line
            charPtr++;
            if (*charPtr == '1')
            {
              line1 = 1;
              textX = &line1Text;
            }
            else if (*charPtr == '2')
            {
              line2 = 1;
              textX = &line2Text;
            }
            else
            {
              fprintf(stderr, "Error: Unsupported lcdLine number \"%c\" (0x%02x)!\n", *charPtr, *charPtr);
              usage();
              exit(1);
            }

            // we should now be at the end of the single - list
            charPtr++;
            if (*charPtr != '\0')
            {
              fprintf(stderr, "Error: Expected lcdLine text but got \"%c\" (0x%02x)!\n", *charPtr, *charPtr);
              usage();
              exit(1);
            }
            // so that the loop will break on the '\0'
            charPtr--;
            
            // get text param
            paramIdx++;
            if (paramIdx >= argc)
            {
              fprintf(stderr, "Error: Missing lcdLine text!\n");
              usage();
              exit(1);
            }

            *textX = argv[paramIdx];
          } break;
          
          default :
          {
            fprintf(stderr, "Error: Unsupported option \"-%c\" (0x%02x)!\n", *charPtr, *charPtr);
            usage();
            exit(1);
          }
        }
        charPtr++;
      } while (*charPtr != '\0');
    }
    
    else
    {
      usage();
      exit(1);
    }
    
    paramIdx++;
  }

  
  // open the i2c file
  // Note: we are accessing the primary i2c bus (bus 0)
  i2cbus  = 0;
  address = 0x1A;
  file = open_i2c_dev(i2cbus, filename, sizeof(filename));
  if (file < 0)
  {
    exit(1);
  }
  
  // check it's capabilites and set the slave address
  usePEC = 0;
  if ( checkCapabilites(file, &usePEC) || setSlaveAddr(file, address, force) )
  {
    close(file);
    exit(1);
  }
  
  // try to enable PEC if it is available
  if ( usePEC && (ioctl(file, I2C_PEC, 1) < 0) )
  {
    fprintf(stderr, "Error: Could not set PEC (Packet Error Checking): %s\n", strerror(errno));
    close(file);
    exit(1);
  }

  
  // get the module version info
  if (versionInfo)
  {
    if ( readDWord(file, 0x10, "Module Type") ||
         readDWord(file, 0x11, "Serial Number") ||
         readWord(file, 0x12, "Firmware Version") )
    {
      close(file);
      exit(1);
    }
  }
  
  // get the module power info
  if (powerInfo)
  {
    if ( readWord(file, 0x30, "Sense 5V") ||
         readWord(file, 0x31, "Sense 5V Max") ||
         readWord(file, 0x32, "Sense 12V") ||
         readWord(file, 0x33, "Sense 12V Max") ||
         readWord(file, 0x34, "Sense 24V") ||
         readWord(file, 0x35, "Sense 24V Max") ||
         readByte(file, 0x36, "Temp", 1) ||
         readByte(file, 0x37, "Temp Min", 1) ||
         readByte(file, 0x38, "Temp Max", 1) ||
         readWord(file, 0x39, "Current") ||
         readWord(file, 0x3A, "Current Max") ||
         readDWord(file, 0x3B, "Power") ||
         readDWord(file, 0x3C, "Power Max") ||
         readWord(file, 0x3D, "DCurrent") ||
         readWord(file, 0x3E, "DCurrent Max") ||
         readDWord(file, 0x3B, "DPower") ||
         readDWord(file, 0x3C, "DPower Max") )
    {
      close(file);
      exit(1);
    }
  }
  
  // set the backlight to on
  if ( (lightOnOff == 1) && (i2c_smbus_write_byte_data(file, 0x60, 0xFF) < 0) )
  {
    fprintf(stderr, "Error: Could not turn the Backlight on: %s\n", strerror(errno));
    close(file);
    exit(1);
  }
  // set the backlight to off
  else if ( (lightOnOff == 0) && (i2c_smbus_write_byte_data(file, 0x60, 0x00) < 0) )
  {
    fprintf(stderr, "Error: Could not turn the Backlight off: %s\n", strerror(errno));
    close(file);
    exit(1);
  }
  
  // read button state
  if (buttons)
  {
    value = i2c_smbus_read_byte_data(file, 0x63);
    if (value < 0)
    {
      fprintf(stderr, "Error: Could not get %s: %s\n", "Button States", strerror(errno));
      close(file);
      exit(1);
    }
    
    byte = (unsigned char)(value & 0xFF);
    fprintf(stdout, "%s: 0x%02x %hhu\n", "Button States", byte, byte);
    
    fprintf(stdout, "%s: %d\n", "Up Button", (byte & 0x08) ? 1 : 0);
    fprintf(stdout, "%s: %d\n", "Down Button", (byte & 0x04) ? 1 : 0);
    fprintf(stdout, "%s: %d\n", "Left Button", (byte & 0x02) ? 1 : 0);
    fprintf(stdout, "%s: %d\n", "Right Button", (byte & 0x01) ? 1 : 0);
    fprintf(stdout, "%s: %d\n", "Enter Button", (byte & 0x10) ? 1 : 0);
  }
  
  // write lcd line 1 text
  if ( (line1) && writeLCDLine(file, 0x61, "LCD Line 1", line1Text) )
  {
    close(file);
    exit(1);
  }
  if ( (line2) && writeLCDLine(file, 0x62, "LCD Line 2", line2Text) )
  {
    close(file);
    exit(1);
  }


  // all done
  close(file);
  exit(0);
}
