#!/usr/bin/env python
## XML converter 'tc-xml2h.py'.
## Generates a header file for etherlab from a exported xml file.
## Carlos Herkt, Maximilian Pachl
## 2014-09-11
## 
##  This script is free software; you can redistribute it and/or
##  modify it under the terms of the GNU General Public License version 2, as
##  published by the Free Software Foundation.
##
##  This script is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
##  Public License for more details.
##
##  You should have received a copy of the GNU General Public License along
##  with this script; if not, write to the Free Software
##  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##
##  ---
##
##  The license mentioned above concerns the source code only. Using the
##  EtherCAT technology and brand is only permitted in compliance with the
##  industrial property and similar rights of Beckhoff Automation GmbH.


import xml.etree.ElementTree as etree
import sys, time


# ======================================================================
# Constants
# ======================================================================

## header output filename
OUT_HEADER_FILE = 'ethercat_config.h'

## number of sync managers
NUM_SYNC_MANAGERS = 4


# ======================================================================
# main
# ======================================================================

## default entry point for application
def main():
    # variables containing the generated source code
    devicesStr = ""
    sdosStr = ""
    pdoEntriesStr = ""
    pdosStr = ""
    syncManagerStr = ""

    # for each file on command line parse and generate source
    for argIndex in range(1, len(sys.argv)):
        print("Converting file " + sys.argv[argIndex] + "...");
        pdos, sms, sdos, device = parsexml(sys.argv[argIndex])

        # generate sourcefile from parsed xml
        devicesStr += generateDevice(device, argIndex)
        sdosStr += generateSDO(sdos, argIndex)
        pdoEntriesStr += "/* Type: " +  device["type"] + "\n * Vendor ID:\t" + hex(device["vendorId"]) + "\n * Product Code:\t" + hex(device["productCode"]) + "\n */\n\n"
        pdoEntriesStr += generateEntryArrays(pdos, argIndex)
        pdosStr += generatePDOArrays(sms, argIndex)
        syncManagerStr += generateSyncManager(sms, argIndex)

    # write the generated source to the file
    fh = open(OUT_HEADER_FILE, 'w')

    # print file header
    fh.write("#ifndef ETHERCATCONFIG_H\n")
    fh.write("#define ETHERCATCONFIG_H\n\n")
    fh.write("/*****************************************************************************\n")
    fh.write(" *\n")
    fh.write(" * ethercat_config.h\n")
    fh.write(" *\n")
    fh.write(" * Generated by tc-xml2h v1.0 at " + time.strftime("%c") + "\n")
    fh.write(" * This file contains CoE initialization commands, PDO, SyncManager and PDOEntry configuration.\n")
    fh.write(" *\n")
    fh.write(" ****************************************************************************/\n\n\n")

    # write the terminal definitions
    fh.write("/****************************************************************************/\n// Terminal information\n\n")
    fh.write(devicesStr)

    # write the CoE initilaisation SDOs
    fh.write("\n\n/****************************************************************************/\n// SDO configuration (CoE init commands)\n\n")
    fh.write(sdosStr)

    # write the PDOs and there entries
    fh.write("\n\n/****************************************************************************/\n// PDOs\n\n")
    fh.write(pdoEntriesStr)
    fh.write(pdosStr)

    # write the syncmagers
    fh.write("\n/****************************************************************************/\n// SyncManagers\n\n")
    fh.write(syncManagerStr)

    # closing if
    fh.write("#endif\n")

    # byebye, generation finished
    fh.close()
    print("finished source generation")


# ======================================================================
# program specific functions
# ======================================================================

# parses a xml file
def parsexml(file):
    tree = etree.parse(file)
    root = tree.getroot()

    # the ethercat slave is our terminal
    xmlSlave = root.find("EtherCAT").find("Slave")

    # find all necessary information about the terminal/device we are using
    deviceInfoXml = xmlSlave.find("Info")
    deviceInfo = {
        "type": deviceInfoXml.find("Type").text,
        "vendorId": int(deviceInfoXml.find("VendorId").text),
        "productCode": int(deviceInfoXml.find("ProductCode").text)
    }

    # grab all PDOs and syncmanagers and put it into the list
    pdos = []
    syncManagers = {}
    for pdo in xmlSlave.find("ProcessData"):
        # we are only intreseted in the TxPdo and RxPdo Tags
        if pdo.tag == "TxPdo" or pdo.tag == "RxPdo":
            # get all entries of this pdo
            entries = []
            for entry in pdo.findall("Entry"):
                try:
                    entries.append(PDOEntry(
                        hextoInt(entry.find("Index").text),
                        int(entry.find("SubIndex").text),
                        entry.find("Name").text,
                        int(entry.find("BitLen").text)
                    ))
                except AttributeError:  # a tag is missing -> log and skip
                    print("\tInvalid PDO entry!")

            # append to the overall list of pdo objects
            pdoObject = PDO(hextoInt(pdo.find("Index").text), pdo.find("Name").text, entries)
            pdos.append(pdoObject)

            # append to the right sync manager
            syncManagerIndex = int(pdo.attrib["Sm"])
            if not(syncManagerIndex in syncManagers):   # create list if new syncmanager
                syncManagers[syncManagerIndex] = list()

            syncManagers[syncManagerIndex].append(pdoObject)


    ## grab the CoE Init Commands
    sdoConfigs = []
    for initCmd in xmlSlave.find("Mailbox").find("CoE").find("InitCmds").findall("InitCmd"):
        cmdComment = initCmd.find("Comment").text
        if "CoE Init Cmd" in cmdComment:    # we are interested in the CoE Init commands
            sdoConfigs.append({"index": int(initCmd.find("Index").text), "data": initCmd.find("Data").text})

    return pdos, syncManagers, sdoConfigs, deviceInfo

## generates a pdo entry array for each pdo
def generateEntryArrays(pdos, deviceIndex):
    result = ""
    for pdo in pdos:
        result += "// " + pdo.name +"\n"
        result += "ec_pdo_entry_info_t pdo_entries_" + str(deviceIndex) + "_" + hex(pdo.index) + "[] = {" +"\n"
        for entry in pdo.entries:
           result += "\t{" + hex(entry.index) + ", " + str(entry.subIndex) + ", " + str(entry.bitLength) + "},\t// " + entry.name +"\n"

        result += "\t{}" +"\n" + "};\n\n"
    return result

## generates a pdo array for each sync manager
def generatePDOArrays(syncManagers, deviceIndex):
    result = ""
    for sm, pdos in syncManagers.items():
        result += "ec_pdo_info_t pdos_" + str(deviceIndex) + "_sm_" + str(sm) + "[] = {" +"\n"
        for pdo in pdos:
            result += "\t{" + hex(pdo.index) + ", " + str(len(pdo.entries)) + ", pdo_entries_" + str(deviceIndex) + "_" + hex(pdo.index) + "},\t//" + pdo.name +"\n"

        result += "};" +"\n\n"
    return result

## generate the sync manager array
def generateSyncManager(syncManagers, deviceIndex):
    result = "ec_sync_info_t syncs_" + str(deviceIndex) + "[] = {" +"\n"

    # we have 4 sync managers all the time!
    for smIndex in range(0, NUM_SYNC_MANAGERS):
        direction = ["EC_DIR_INPUT", "EC_DIR_OUTPUT"][smIndex%2==0]

        # only defined sync managers are properly generated
        if smIndex in syncManagers:
            result += "\t{" + str(smIndex) + ", " + direction + ", " + str(len(syncManagers[smIndex])) + ", pdos_" + str(deviceIndex) + "_sm_" + str(smIndex) + ", EC_WD_ENABLE}," +"\n"
        else:
            result += "\t{" + str(smIndex) + ", " + direction + ", 0, NULL, EC_WD_DISABLE}," +"\n"
    result += "\t{0xff}\n};\n\n"

    return result

## generates the CoE Init commans as char arrays
def generateSDO(sdos, deviceIndex):
    result = ""
    for sdo in sdos:
        result += "unsigned char sdo_config_" + str(deviceIndex) + "_" + hex(sdo["index"]) + "[] = " + strToCharArray(sdo["data"]) +"\n"

    return result

## generates the Device defines
def generateDevice(info, deviceIndex):
    return "#define Terminal_" + str(deviceIndex) + "_" + info["type"] + " " + hex(info["vendorId"]) + ", " + hex(info["productCode"]) +"\n"


# ======================================================================
# helper functions
# ======================================================================

# converts a hex string to an int
# example input: #x6000
def hextoInt(hexString):
    return int(hexString.replace("#", "0"), 0)

# converts a string containing hex characters to
# a C style char array initializer.
def strToCharArray(hexString):
    result = "{"
    i = 0
    while i < len(hexString) - 2:
        result += "0x" + hexString[i] + hexString[i + 1] + ", "
        i += 2

    result += "0x" + hexString[i] + hexString[i + 1]
    result += "};"
    return result


# ======================================================================
# Class definitions
# ======================================================================

## a pdo containing multiple pdo entries
class PDO:
    def __init__(self, index, name, entries):
        self.index = index
        self.name = name
        self.entries = entries

# a pdo entry
class PDOEntry:
    def __init__(self, index, subIndex, name, bitLength):
        self.index = index
        self.subIndex = subIndex
        self.name = name
        self.bitLength = bitLength


# ======================================================================
# Entry point
# ======================================================================

if __name__ == "__main__":
    main()