/* dialog.c - dialog windows for internal use
 *
 *  Window Maker window manager
 *
 *  Copyright (c) 1997-2003 Alfredo K. Kojima
 *  Copyright (c) 1998-2003 Dan Pascu
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */

#include "wconfig.h"

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <limits.h>

#include "WMDialogController.h"

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#include <signal.h>
#ifdef __FreeBSD__
#include <sys/signal.h>
#endif


#ifndef PATH_MAX
#define PATH_MAX DEFAULT_PATH_MAX
#endif

#include "WindowMaker.h"
#include "GNUstep.h"
#include "screen.h"
#include "dialog.h"
#include "funcs.h"
#include "stacking.h"
#include "framewin.h"
#include "window.h"
#include "actions.h"
#include "defaults.h"
#include "xinerama.h"



extern WPreferences wPreferences;


static WMPoint
getCenter(WScreen *scr, int width, int height)
{
    return wGetPointToCenterRectInHead(scr, wGetHeadForPointerLocation(scr),
                                       width, height);
}


int
wMessageDialog(WScreen *scr, char *title, char *message,
               char *defBtn, char *altBtn, char *othBtn)
{
#if 0
  /* Cannot do this because it might be called before NSApp start */
  WMDialogController *controller = [WMDialogController sharedController];
  NSString *tle, *msg;
  NSString *defaultButton, *alternateButton, *otherButton;

  if (title)
    tle = [NSString stringWithCString: title];
  else
    tle = nil;

  if (message)
    msg = [NSString stringWithCString: message];
  else
    msg = nil;

  if (defBtn)
    defaultButton = [NSString stringWithCString: defBtn];
  else
    defaultButton = nil;

  if (altBtn)
    alternateButton = [NSString stringWithCString: altBtn];
  else
    alternateButton = nil;

  if (othBtn)
    otherButton = [NSString stringWithCString: othBtn];
  else
    otherButton = nil;

  return [controller messageDialogWithTitle: tle
	                     message: msg
	 	       defaultButton: defaultButton
		     alternateButton: alternateButton
		         otherButton: otherButton];
#else
    WMAlertPanel *panel;
    Window parent;
    WWindow *wwin;
    int result;
    WMPoint center;

    panel = WMCreateAlertPanel(scr->wmscreen, NULL, title, message,
                               defBtn, altBtn, othBtn);

    parent = XCreateSimpleWindow(dpy, scr->root_win, 0, 0, 400, 180, 0, 0, 0);

    XReparentWindow(dpy, WMWidgetXID(panel->win), parent, 0, 0);


    center = getCenter(scr, 400, 180);
    wwin = wManageInternalWindow(scr, parent, None, NULL, center.x, center.y,
                                 400, 180);
    wwin->client_leader = WMWidgetXID(panel->win);

    WMMapWidget(panel->win);

    wWindowMap(wwin);

    WMRunModalLoop(WMWidgetScreen(panel->win), WMWidgetView(panel->win));

    result = panel->result;

    WMUnmapWidget(panel->win);

    wUnmanageWindow(wwin, False, False);

    WMDestroyAlertPanel(panel);

    XDestroyWindow(dpy, parent);

    return result;
#endif
}

int
wInputDialog(WScreen *scr, char *title, char *message, char **text)
{
#if 1
  NSString *tle, *msg, *txt;
  if (title)
    tle = [NSString stringWithCString: title];
  else
    tle = nil;

  if (message)
    msg = [NSString stringWithCString: message];
  else
    msg = nil;

  if (*text)
    txt = [NSString stringWithCString: *text];
  else
    txt = nil;

  WMDialogController *controller = [WMDialogController sharedController];
  NSString *string = [controller inputDialogWithTitle: tle
	                                      message: msg text: txt];
  if (string)
  {
    if (*text)
      wfree(*text);
    *text = wstrdup((char*)[string cString]);

    return True;
  }
  else
#endif
  {
    return False;
  }
}


/*
 *****************************************************************
 * Icon Selection Panel
 *****************************************************************
 */

typedef struct IconPanel {

    WScreen *scr;

    WMWindow *win;

    WMLabel *dirLabel;
    WMLabel *iconLabel;

    WMList *dirList;
    WMList *iconList;
    WMFont *normalfont;

    WMButton *previewButton;

    WMLabel *iconView;

    WMLabel *fileLabel;
    WMTextField *fileField;

    WMButton *okButton;
    WMButton *cancelButton;
#if 0
    WMButton *chooseButton;
#endif
    short done;
    short result;
    short preview;
} IconPanel;



static void
listPixmaps(WScreen *scr, WMList *lPtr, char *path)
{
    struct dirent *dentry;
    DIR *dir;
    char pbuf[PATH_MAX+16];
    char *apath;
    IconPanel *panel = WMGetHangedData(lPtr);

    panel->preview = False;

    apath = wexpandpath(path);
    dir = opendir(apath);

    if (!dir) {
        char *msg;
        char *tmp;
        tmp = ("Could not open directory ");
        msg = wmalloc(strlen(tmp)+strlen(path)+6);
        strcpy(msg, tmp);
        strcat(msg, path);

        wMessageDialog(scr, ("Error"), msg, ("OK"), NULL, NULL);
        wfree(msg);
        wfree(apath);
        return;
    }

    /* list contents in the column */
    while ((dentry = readdir(dir))) {
        struct stat statb;

        if (strcmp(dentry->d_name, ".")==0 ||
            strcmp(dentry->d_name, "..")==0)
            continue;

        strcpy(pbuf, apath);
        strcat(pbuf, "/");
        strcat(pbuf, dentry->d_name);

        if (stat(pbuf, &statb)<0)
            continue;

        if (statb.st_mode & (S_IRUSR|S_IRGRP|S_IROTH)
            && statb.st_mode & (S_IFREG|S_IFLNK)) {
            WMAddListItem(lPtr, dentry->d_name);
        }
    }
    WMSortListItems(lPtr);

    closedir(dir);
    wfree(apath);
    panel->preview = True;
}



static void
setViewedImage(IconPanel *panel, char *file)
{
    WMPixmap *pixmap;
    RColor color;

    color.red = 0xae;
    color.green = 0xaa;
    color.blue = 0xae;
    color.alpha = 0;
    pixmap = WMCreateBlendedPixmapFromFile(WMWidgetScreen(panel->win),
                                           file, &color);
    if (!pixmap) {
        WMSetButtonEnabled(panel->okButton, False);

        WMSetLabelText(panel->iconView, ("Could not load image file "));

        WMSetLabelImage(panel->iconView, NULL);
    } else {
        WMSetButtonEnabled(panel->okButton, True);

        WMSetLabelText(panel->iconView, NULL);
        WMSetLabelImage(panel->iconView, pixmap);
        WMReleasePixmap(pixmap);
    }
}


static void
listCallback(void *self, void *data)
{
    WMList *lPtr = (WMList*)self;
    IconPanel *panel = (IconPanel*)data;
    char *path;

    if (lPtr==panel->dirList) {
        WMListItem *item = WMGetListSelectedItem(lPtr);

        if (item == NULL)
            return;
        path = item->text;

        WMSetTextFieldText(panel->fileField, path);

        WMSetLabelImage(panel->iconView, NULL);

        WMSetButtonEnabled(panel->okButton, False);

        WMClearList(panel->iconList);
        listPixmaps(panel->scr, panel->iconList, path);
    } else {
        char *tmp, *iconFile;
        WMListItem *item = WMGetListSelectedItem(panel->dirList);

        if (item == NULL)
            return;
        path = item->text;
        tmp = wexpandpath(path);

        item = WMGetListSelectedItem(panel->iconList);
        if (item == NULL)
            return;
        iconFile = item->text;

        path = wmalloc(strlen(tmp)+strlen(iconFile)+4);
        strcpy(path, tmp);
        strcat(path, "/");
        strcat(path, iconFile);
        wfree(tmp);
        WMSetTextFieldText(panel->fileField, path);
        setViewedImage(panel, path);
        wfree(path);
    }
}


static void
listIconPaths(WMList *lPtr)
{
    char *paths;
    char *path;

    paths = wstrdup(wPreferences.icon_path);

    path = strtok(paths, ":");

    do {
        char *tmp;

        tmp = wexpandpath(path);
        /* do not sort, because the order implies the order of
         * directories searched */
        if (access(tmp, X_OK)==0)
            WMAddListItem(lPtr, path);
        wfree(tmp);
    } while ((path=strtok(NULL, ":"))!=NULL);

    wfree(paths);
}


void
drawIconProc(WMList *lPtr, int index, Drawable d, char *text, int state,
             WMRect *rect)
{
    IconPanel *panel = WMGetHangedData(lPtr);
    WScreen *scr = panel->scr;
    GC gc = scr->draw_gc;
    GC copygc = scr->copy_gc;
    char *file, *dirfile;
    WMPixmap *pixmap;
    WMColor *back;
    WMSize size;
    WMScreen *wmscr = WMWidgetScreen(panel->win);
    RColor color;
    int x, y, width, height, len;

    if(!panel->preview) return;

    x = rect->pos.x;
    y = rect->pos.y;
    width = rect->size.width;
    height = rect->size.height;

    back = (state & WLDSSelected) ? scr->white : scr->gray;

    dirfile = wexpandpath(WMGetListSelectedItem(panel->dirList)->text);
    len = strlen(dirfile)+strlen(text)+4;
    file = wmalloc(len);
    snprintf(file, len, "%s/%s", dirfile, text);
    wfree(dirfile);

    color.red = WMRedComponentOfColor(back) >> 8;
    color.green = WMGreenComponentOfColor(back) >> 8;
    color.blue = WMBlueComponentOfColor(back) >> 8;
    color.alpha = WMGetColorAlpha(back) >> 8;

    pixmap = WMCreateBlendedPixmapFromFile(wmscr, file, &color);
    wfree(file);

    if (!pixmap) {
        /*WMRemoveListItem(lPtr, index);*/
        return;
    }

    XFillRectangle(dpy, d, WMColorGC(back), x, y, width, height);

    XSetClipMask(dpy, gc, None);
    /*XDrawRectangle(dpy, d, WMColorGC(white), x+5, y+5, width-10, 54);*/
    XDrawLine(dpy, d, WMColorGC(scr->white), x, y+height-1, x+width, y+height-1);

    size = WMGetPixmapSize(pixmap);

    XSetClipMask(dpy, copygc, WMGetPixmapMaskXID(pixmap));
    XSetClipOrigin(dpy, copygc, x + (width-size.width)/2, y+2);
    XCopyArea(dpy, WMGetPixmapXID(pixmap), d, copygc, 0, 0,
              size.width>100?100:size.width, size.height>64?64:size.height,
              x + (width-size.width)/2, y+2);

    {
        int i,j;
        int fheight = WMFontHeight(panel->normalfont);
        int tlen = strlen(text);
        int twidth = WMWidthOfString(panel->normalfont, text, tlen);
        int ofx, ofy;

        ofx = x + (width - twidth)/2;
        ofy = y + 64 - fheight;

        for(i=-1;i<2;i++)
            for(j=-1;j<2;j++)
                WMDrawString(wmscr, d, scr->white, panel->normalfont,
                             ofx+i, ofy+j, text, tlen);

        WMDrawString(wmscr, d, scr->black, panel->normalfont, ofx, ofy,
                     text, tlen);
    }

    WMReleasePixmap(pixmap);
    /* I hope it is better to do not use cache / on my box it is fast nuff */
    XFlush(dpy);
}


static void
buttonCallback(void *self, void *clientData)
{
    WMButton *bPtr = (WMButton*)self;
    IconPanel *panel = (IconPanel*)clientData;


    if (bPtr==panel->okButton) {
        panel->done = True;
        panel->result = True;
    } else if (bPtr==panel->cancelButton) {
        panel->done = True;
        panel->result = False;
    } else if (bPtr==panel->previewButton) {
        /**** Previewer ****/
        WMSetButtonEnabled(bPtr, False);
        WMSetListUserDrawItemHeight(panel->iconList, 68);
        WMSetListUserDrawProc(panel->iconList, drawIconProc);
        WMRedisplayWidget(panel->iconList);
        /* for draw proc to access screen/gc */
        /*** end preview ***/
    }
#if 0
    else if (bPtr==panel->chooseButton) {
        WMOpenPanel *op;

        op = WMCreateOpenPanel(WMWidgetScreen(bPtr));

        if (WMRunModalFilePanelForDirectory(op, NULL, "/usr/local", NULL, NULL)) {
            char *path;
            path = WMGetFilePanelFile(op);
            WMSetTextFieldText(panel->fileField, path);
            setViewedImage(panel, path);
            wfree(path);
        }
        WMDestroyFilePanel(op);
    }
#endif
}


static void
keyPressHandler(XEvent *event, void *data)
{
    IconPanel *panel = (IconPanel*)data;
    char buffer[32];
    int count;
    KeySym ksym;
    int iidx;
    int didx;
    int item = 0;
    WMList *list = NULL;

    if (event->type == KeyRelease)
        return;

    buffer[0] = 0;
    count = XLookupString(&event->xkey, buffer, sizeof(buffer), &ksym, NULL);


    iidx = WMGetListSelectedItemRow(panel->iconList);
    didx = WMGetListSelectedItemRow(panel->dirList);

    switch (ksym) {
    case XK_Up:
        if (iidx > 0)
            item = iidx-1;
        else
            item = iidx;
        list = panel->iconList;
        break;
    case XK_Down:
        if (iidx < WMGetListNumberOfRows(panel->iconList) - 1)
            item = iidx+1;
        else
            item = iidx;
        list = panel->iconList;
        break;
    case XK_Home:
        item = 0;
        list = panel->iconList;
        break;
    case XK_End:
        item = WMGetListNumberOfRows(panel->iconList) - 1;
        list = panel->iconList;
        break;
    case XK_Next:
        if (didx < WMGetListNumberOfRows(panel->dirList) - 1)
            item = didx + 1;
        else
            item = didx;
        list = panel->dirList;
        break;
    case XK_Prior:
        if (didx > 0)
            item = didx - 1;
        else
            item = 0;
        list = panel->dirList;
        break;
    case XK_Return:
        WMPerformButtonClick(panel->okButton);
        break;
    case XK_Escape:
        WMPerformButtonClick(panel->cancelButton);
        break;
    }

    if (list) {
        WMSelectListItem(list, item);
        WMSetListPosition(list, item - 5);
        listCallback(list, panel);
    }
}



Bool
wIconChooserDialog(WScreen *scr, char **file, char *instance, char *class)
{
    WWindow *wwin;
    Window parent;
    IconPanel *panel;
    WMColor *color;
    WMFont *boldFont;
    Bool result;

    panel = wmalloc(sizeof(IconPanel));
    memset(panel, 0, sizeof(IconPanel));

    panel->scr = scr;

    panel->win = WMCreateWindow(scr->wmscreen, "iconChooser");
    WMResizeWidget(panel->win, 450, 280);

    WMCreateEventHandler(WMWidgetView(panel->win), KeyPressMask|KeyReleaseMask,
                         keyPressHandler, panel);


    boldFont = WMBoldSystemFontOfSize(scr->wmscreen, 12);
    panel->normalfont = WMSystemFontOfSize(WMWidgetScreen(panel->win), 12);

    panel->dirLabel = WMCreateLabel(panel->win);
    WMResizeWidget(panel->dirLabel, 200, 20);
    WMMoveWidget(panel->dirLabel, 10, 7);
    WMSetLabelText(panel->dirLabel, ("Directories"));
    WMSetLabelFont(panel->dirLabel, boldFont);
    WMSetLabelTextAlignment(panel->dirLabel, WACenter);

    WMSetLabelRelief(panel->dirLabel, WRSunken);

    panel->iconLabel = WMCreateLabel(panel->win);
    WMResizeWidget(panel->iconLabel, 140, 20);
    WMMoveWidget(panel->iconLabel, 215, 7);
    WMSetLabelText(panel->iconLabel, ("Icons"));
    WMSetLabelFont(panel->iconLabel, boldFont);
    WMSetLabelTextAlignment(panel->iconLabel, WACenter);

    WMReleaseFont(boldFont);

    color = WMWhiteColor(scr->wmscreen);
    WMSetLabelTextColor(panel->dirLabel, color);
    WMSetLabelTextColor(panel->iconLabel, color);
    WMReleaseColor(color);

    color = WMDarkGrayColor(scr->wmscreen);
    WMSetWidgetBackgroundColor(panel->iconLabel, color);
    WMSetWidgetBackgroundColor(panel->dirLabel, color);
    WMReleaseColor(color);

    WMSetLabelRelief(panel->iconLabel, WRSunken);

    panel->dirList = WMCreateList(panel->win);
    WMResizeWidget(panel->dirList, 200, 170);
    WMMoveWidget(panel->dirList, 10, 30);
    WMSetListAction(panel->dirList, listCallback, panel);

    panel->iconList = WMCreateList(panel->win);
    WMResizeWidget(panel->iconList, 140, 170);
    WMMoveWidget(panel->iconList, 215, 30);
    WMSetListAction(panel->iconList, listCallback, panel);

    WMHangData(panel->iconList,panel);

    panel->previewButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->previewButton, 75, 26);
    WMMoveWidget(panel->previewButton, 365, 130);
    WMSetButtonText(panel->previewButton, ("Preview"));
    WMSetButtonAction(panel->previewButton, buttonCallback, panel);

    panel->iconView = WMCreateLabel(panel->win);
    WMResizeWidget(panel->iconView, 75, 75);
    WMMoveWidget(panel->iconView, 365, 40);
    WMSetLabelImagePosition(panel->iconView, WIPOverlaps);
    WMSetLabelRelief(panel->iconView, WRSunken);
    WMSetLabelTextAlignment(panel->iconView, WACenter);

    panel->fileLabel = WMCreateLabel(panel->win);
    WMResizeWidget(panel->fileLabel, 80, 20);
    WMMoveWidget(panel->fileLabel, 10, 210);
    WMSetLabelText(panel->fileLabel, ("File Name:"));

    panel->fileField = WMCreateTextField(panel->win);
    WMSetViewNextResponder(WMWidgetView(panel->fileField), WMWidgetView(panel->win));
    WMResizeWidget(panel->fileField, 345, 20);
    WMMoveWidget(panel->fileField, 95, 210);
    WMSetTextFieldEditable(panel->fileField, False);

    panel->okButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->okButton, 80, 26);
    WMMoveWidget(panel->okButton, 360, 240);
    WMSetButtonText(panel->okButton, ("OK"));
    WMSetButtonEnabled(panel->okButton, False);
    WMSetButtonAction(panel->okButton, buttonCallback, panel);

    panel->cancelButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->cancelButton, 80, 26);
    WMMoveWidget(panel->cancelButton, 270, 240);
    WMSetButtonText(panel->cancelButton, ("Cancel"));
    WMSetButtonAction(panel->cancelButton, buttonCallback, panel);
#if 0
    panel->chooseButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->chooseButton, 110, 26);
    WMMoveWidget(panel->chooseButton, 150, 240);
    WMSetButtonText(panel->chooseButton, ("Choose File"));
    WMSetButtonAction(panel->chooseButton, buttonCallback, panel);
#endif
    WMRealizeWidget(panel->win);
    WMMapSubwidgets(panel->win);

    parent = XCreateSimpleWindow(dpy, scr->root_win, 0, 0, 450, 280, 0, 0, 0);

    XReparentWindow(dpy, WMWidgetXID(panel->win), parent, 0, 0);

    {
        char *tmp;
        int len = (instance ? strlen(instance) : 0)
            + (class ? strlen(class) : 0) + 32;
        WMPoint center;

        tmp = wmalloc(len);

        if (tmp && (instance || class))
            snprintf(tmp, len, "%s [%s.%s]", ("Icon Chooser"), instance, class);
        else
            strcpy(tmp, ("Icon Chooser"));

        center = getCenter(scr, 450, 280);

        wwin = wManageInternalWindow(scr, parent, None, tmp, center.x,center.y,
                                     450, 280);
        wfree(tmp);
    }

    /* put icon paths in the list */
    listIconPaths(panel->dirList);

    WMMapWidget(panel->win);

    wWindowMap(wwin);

    while (!panel->done) {
        XEvent event;

        WMNextEvent(dpy, &event);
        WMHandleEvent(&event);
    }

    if (panel->result) {
        char *defaultPath, *wantedPath;

        /* check if the file the user selected is not the one that
         * would be loaded by default with the current search path */
        *file = WMGetListSelectedItem(panel->iconList)->text;
        if (**file==0) {
            wfree(*file);
            *file = NULL;
        } else {
            defaultPath = FindImage(wPreferences.icon_path, *file);
            wantedPath = WMGetTextFieldText(panel->fileField);
            /* if the file is not the default, use full path */
            if (strcmp(wantedPath, defaultPath)!=0) {
                *file = wantedPath;
            } else {
                *file = wstrdup(*file);
                wfree(wantedPath);
            }
            wfree(defaultPath);
        }
    } else {
        *file = NULL;
    }

    result = panel->result;

    WMReleaseFont(panel->normalfont);

    WMUnmapWidget(panel->win);

    WMDestroyWidget(panel->win);

    wUnmanageWindow(wwin, False, False);

    wfree(panel);

    XDestroyWindow(dpy, parent);

    return result;
}


/*
 ***********************************************************************
 * Crashing Dialog Panel
 ***********************************************************************
 */

extern WDDomain *WDWindowAttributes;


typedef struct _CrashPanel {
    WMWindow *win;            /* main window */

    WMLabel *iconL;           /* application icon */
    WMLabel *nameL;           /* title of panel */

    WMFrame *sepF;            /* separator frame */

    WMLabel *noteL;           /* Title of note */
    WMLabel *note2L;          /* body of note with what happened */

    WMFrame *whatF;           /* "what to do next" frame */
    WMPopUpButton *whatP;     /* action selection popup button */

    WMButton *okB;            /* ok button */

    Bool done;                /* if finished with this dialog */
    int action;               /* what to do after */

    KeyCode retKey;

} CrashPanel;


static void
handleKeyPress(XEvent *event, void *clientData)
{
    CrashPanel *panel = (CrashPanel*)clientData;

    if (event->xkey.keycode == panel->retKey) {
        WMPerformButtonClick(panel->okB);
    }
}


static void
okButtonCallback(void *self, void *clientData)
{
    CrashPanel *panel = (CrashPanel*)clientData;

    panel->done = True;
}


static void
setCrashAction(void *self, void *clientData)
{
    WMPopUpButton *pop = (WMPopUpButton*)self;
    CrashPanel *panel = (CrashPanel*)clientData;

    panel->action = WMGetPopUpButtonSelectedItem(pop);
}


/* Make this read the logo from a compiled in pixmap -Dan */
static WMPixmap*
getWindowMakerIconImage(WMScreen *scr)
{
    WMPropList *dict, *key, *option, *value=NULL;
    WMPixmap *pix=NULL;
    char *path;

    if (!WDWindowAttributes || !WDWindowAttributes->dictionary)
        return NULL;

    WMPLSetCaseSensitive(True);

    key = WMCreatePLString("Logo.WMPanel");
    option = WMCreatePLString("Icon");

    dict = WMGetFromPLDictionary(WDWindowAttributes->dictionary, key);

    if (dict) {
        value = WMGetFromPLDictionary(dict, option);
    }

    WMReleasePropList(key);
    WMReleasePropList(option);

    WMPLSetCaseSensitive(False);

    if (value && WMIsPLString(value)) {
        path = FindImage(wPreferences.icon_path, WMGetFromPLString(value));

        if (path) {
            RColor gray;

            gray.red = 0xae;  gray.green = 0xaa;
            gray.blue = 0xae; gray.alpha = 0;

            pix = WMCreateBlendedPixmapFromFile(scr, path, &gray);
            wfree(path);
        }
    }

    return pix;
}


#define PWIDTH	295
#define PHEIGHT	345


int
wShowCrashingDialogPanel(int whatSig)
{
    CrashPanel *panel;
    WMScreen *scr;
    WMFont *font;
    WMPixmap *logo;
    int screen_no, scr_width, scr_height;
    int action;
    char buf[256];

    panel = wmalloc(sizeof(CrashPanel));
    memset(panel, 0, sizeof(CrashPanel));

    screen_no = DefaultScreen(dpy);
    scr_width = WidthOfScreen(ScreenOfDisplay(dpy, screen_no));
    scr_height = HeightOfScreen(ScreenOfDisplay(dpy, screen_no));

    scr = WMCreateScreen(dpy, screen_no);
    if (!scr) {
        wsyserror(("cannot open connection for crashing dialog panel. Aborting."));
        return WMAbort;
    }

    panel->retKey = XKeysymToKeycode(dpy, XK_Return);

    panel->win = WMCreateWindow(scr, "crashingDialog");
    WMResizeWidget(panel->win, PWIDTH, PHEIGHT);
    WMMoveWidget(panel->win, (scr_width - PWIDTH)/2, (scr_height - PHEIGHT)/2);

    logo = getWindowMakerIconImage(scr);
    if (logo) {
        panel->iconL = WMCreateLabel(panel->win);
        WMResizeWidget(panel->iconL, 64, 64);
        WMMoveWidget(panel->iconL, 10, 10);
        WMSetLabelImagePosition(panel->iconL, WIPImageOnly);
        WMSetLabelImage(panel->iconL, logo);
    }

    panel->nameL = WMCreateLabel(panel->win);
    WMResizeWidget(panel->nameL, 200, 30);
    WMMoveWidget(panel->nameL, 80, 25);
    WMSetLabelTextAlignment(panel->nameL, WALeft);
    font = WMBoldSystemFontOfSize(scr, 24);
    WMSetLabelFont(panel->nameL, font);
    WMReleaseFont(font);
    WMSetLabelText(panel->nameL, ("Fatal error"));

    panel->sepF = WMCreateFrame(panel->win);
    WMResizeWidget(panel->sepF, PWIDTH+4, 2);
    WMMoveWidget(panel->sepF, -2, 80);

    panel->noteL = WMCreateLabel(panel->win);
    WMResizeWidget(panel->noteL, PWIDTH-20, 40);
    WMMoveWidget(panel->noteL, 10, 90);
    WMSetLabelTextAlignment(panel->noteL, WAJustified);
#ifdef SYS_SIGLIST_DECLARED
    snprintf(buf, sizeof(buf), ("Window Maker received signal %i\n(%s)."),
             whatSig, sys_siglist[whatSig]);
#else
    snprintf(buf, sizeof(buf), ("Window Maker received signal %i."), whatSig);
#endif
    WMSetLabelText(panel->noteL, buf);

    panel->note2L = WMCreateLabel(panel->win);
    WMResizeWidget(panel->note2L, PWIDTH-20, 100);
    WMMoveWidget(panel->note2L, 10, 130);
    WMSetLabelTextAlignment(panel->note2L, WALeft);
    WMSetLabelText(panel->note2L,
                   (" This fatal error occured probably due to a bug."
                     " Please fill the included BUGFORM and "
                     "report it to bugs@windowmaker.org."));
    WMSetLabelWraps(panel->note2L, True);


    panel->whatF = WMCreateFrame(panel->win);
    WMResizeWidget(panel->whatF, PWIDTH-20, 50);
    WMMoveWidget(panel->whatF, 10, 240);
    WMSetFrameTitle(panel->whatF, ("What do you want to do now?"));

    panel->whatP = WMCreatePopUpButton(panel->whatF);
    WMResizeWidget(panel->whatP, PWIDTH-20-70, 20);
    WMMoveWidget(panel->whatP, 35, 20);
    WMSetPopUpButtonPullsDown(panel->whatP, False);
    WMSetPopUpButtonText(panel->whatP, ("Select action"));
    WMAddPopUpButtonItem(panel->whatP, ("Abort and leave a core file"));
    WMAddPopUpButtonItem(panel->whatP, ("Restart Window Maker"));
    WMAddPopUpButtonItem(panel->whatP, ("Start alternate window manager"));
    WMSetPopUpButtonAction(panel->whatP, setCrashAction, panel);
    WMSetPopUpButtonSelectedItem(panel->whatP, WMRestart);
    panel->action = WMRestart;

    WMMapSubwidgets(panel->whatF);

    panel->okB = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->okB, 80, 26);
    WMMoveWidget(panel->okB, 205, 309);
    WMSetButtonText(panel->okB, ("OK"));
    WMSetButtonImage(panel->okB, WMGetSystemPixmap(scr, WSIReturnArrow));
    WMSetButtonAltImage(panel->okB, WMGetSystemPixmap(scr, WSIHighlightedReturnArrow));
    WMSetButtonImagePosition(panel->okB, WIPRight);
    WMSetButtonAction(panel->okB, okButtonCallback, panel);

    panel->done = 0;

    WMCreateEventHandler(WMWidgetView(panel->win), KeyPressMask,
                         handleKeyPress, panel);

    WMRealizeWidget(panel->win);
    WMMapSubwidgets(panel->win);

    WMMapWidget(panel->win);

    XSetInputFocus(dpy, WMWidgetXID(panel->win), RevertToParent, CurrentTime);

    while (!panel->done) {
        XEvent event;

        WMNextEvent(dpy, &event);
        WMHandleEvent(&event);
    }

    action = panel->action;

    WMUnmapWidget(panel->win);
    WMDestroyWidget(panel->win);
    wfree(panel);

    return action;
}



/*****************************************************************************
 *			About GNUstep Panel
 *****************************************************************************/

#if 0 /* might be useful in the future */
static void
drawGNUstepLogo(Display *dpy, Drawable d, int width, int height,
                unsigned long blackPixel, unsigned long whitePixel)
{
    GC gc;
    XGCValues gcv;
    XRectangle rects[3];

    gcv.foreground = blackPixel;
    gc = XCreateGC(dpy, d, GCForeground, &gcv);

    XFillArc(dpy, d, gc, width/45, height/45,
             width - 2*width/45, height - 2*height/45, 0, 360*64);

    rects[0].x = 0;
    rects[0].y = 37*height/45;
    rects[0].width = width/3;
    rects[0].height = height - rects[0].y;

    rects[1].x = rects[0].width;
    rects[1].y = height/2;
    rects[1].width = width - 2*width/3;
    rects[1].height = height - rects[1].y;

    rects[2].x = 2*width/3;
    rects[2].y = height - 37*height/45;
    rects[2].width = width/3;
    rects[2].height = height - rects[2].y;

    XSetClipRectangles(dpy, gc, 0, 0, rects, 3, Unsorted);
    XFillRectangle(dpy, d, gc, 0, 0, width, height);

    XSetForeground(dpy, gc, whitePixel);
    XFillArc(dpy, d, gc, width/45, height/45,
             width - 2*width/45, height - 2*height/45, 0, 360*64);

    XFreeGC(dpy, gc);
}
#endif

