#include "LuceneKit/Store/LCBufferedIndexInput.h"
#include "GNUstep.h"

/** Base implementation class for buffered {@link IndexInput}. */
@implementation LCBufferedIndexInput

- (id) init
{
  self = [super init];
  buffer = [[NSMutableData alloc] init];
  bufferStart = 0;			  // position in file of buffer
  bufferLength = 0;			  // end of valid bytes
  bufferPosition = 0;		  // next byte to read
  return self;
}

- (void) dealloc
{
  RELEASE(buffer);
  [super dealloc];
}

- (char) readByte
{
  char b;
  if (bufferPosition >= bufferLength)
    [self refill];
  NSRange r = NSMakeRange(bufferPosition, 1);
  [buffer getBytes: b range: r];
  return b;
}

- (void) readBytes: (char *) b offset: (int) offset length: (int) len
{
  int i;
  long start;
  NSRange r;
  if (len < INPUT_BUFFER_SIZE) {
    r = NSMakeRange(offset, len);
    [buffer getBytes: b range: r];
  } else {					  // read all-at-once
    start = [self getFilePointer];
    [self seekInternal: start];
    [self readInternal: b offset: offset length: len];

    bufferStart = start + len;		  // adjust stream variables
    bufferPosition = 0;
    bufferLength = 0;				  // trigger refill() on read
  }
}

- (void) refill
{
  long start = bufferStart + bufferPosition;
  long end = start + INPUT_BUFFER_SIZE;
  if (end > [self length])				  // don't read past EOF
    end = [self length];
  bufferLength = (int)(end - start);
  if (bufferLength <= 0)
    {
      NSLog(@"read past EOF");
      return;
      //throw new IOException("read past EOF");
    }

  if (buffer == NULL)
    buffer = malloc(INPUT_BUFFER_SIZE*sizeof(char));  // allocate buffer lazily

  [self readInternal: buffer offset: 0 length: bufferLength];

  bufferStart = start;
  bufferPosition = 0;
}

  /** Expert: implements buffer refill.  Reads bytes from the current position
   * in the input.
   * @param b the array to read bytes into
   * @param offset the offset in the array to start storing bytes
   * @param length the number of bytes to read
   */
- (void) readInternal: (char *) b offset: (int) offset length: (int) length
{
}

- (long) getFilePointer
{ 
  return bufferStart + bufferPosition; 
}

- (void) seek: (long) pos
{
  if (pos >= bufferStart && pos < (bufferStart + bufferLength))
    bufferPosition = (int)(pos - bufferStart);  // seek within buffer
  else {
    bufferStart = pos;
    bufferPosition = 0;
    bufferLength = 0;				  // trigger refill() on read()
    [self seekInternal:pos];
  }
}

  /** Expert: implements seek.  Sets current position in this file, where the
   * next {@link #readInternal(byte[],int,int)} will occur.
   * @see #readInternal(byte[],int,int)
   */
- (void) seekInternal: (long) pos
{
}

#if 0
  public Object clone() {
    BufferedIndexInput clone = (BufferedIndexInput)super.clone();

    if (buffer != null) {
      clone.buffer = new byte[BUFFER_SIZE];
      System.arraycopy(buffer, 0, clone.buffer, 0, bufferLength);
    }

    return clone;
  }
#endif

@end
