#include "LuceneKit/Store/LCBufferedIndexOutput.h"

#define OUTPUT_BUFFER_SIZE 1024

/** Base implementation class for buffered {@link IndexOutput}. */
@implementation LCBufferedIndexOutput

- (id) init
{
  self = [super init];
  byte = malloc(OUTPUT_BUFFER_SIZE*sizeof(char));
  bufferStart = 0; // position in file of buffer
  bufferPosition = 0; // position in buffer
  return self;
}

- (void) dealloc
{
  free(byte);
  [super dealloc];
}

  /** Writes a single byte.
   * @see IndexInput#readByte()
   */
- (void) writeByte: (char) b
{
  if (bufferPosition >= OUTPUT_BUFFER_SIZE)
    [self flush];
  byte[bufferPosition++] = b;
}

  /** Writes an array of bytes.
   * @param b the bytes to write
   * @param length the number of bytes to write
   * @see IndexInput#readBytes(byte[],int,int)
   */
- (void) writeBytes: (char *) b length: (int) length
{
  int i;
  for (i = 0; i < length; i++)
    [self writeByte: b[i]];
}

  /** Forces any buffered output to be written. */
- (void) flush
{
  [self flushBuffer: byte length: bufferPosition];
  bufferStart += bufferPosition;
  bufferPosition = 0;
}

  /** Expert: implements buffer write.  Writes bytes at the current position in
   * the output.
   * @param b the bytes to write
   * @param len the number of bytes to write
   */
- (void) flushBuffer: (char *) b length: (int) len
{
}

  /** Closes this stream to further operations. */
- (void) close
{
  [self flush];
}

  /** Returns the current position in this file, where the next write will
   * occur.
   * @see #seek(long)
   */
- (long) getFilePointer
{
  return bufferStart + bufferPosition;
}

  /** Sets current position in this file, where the next write will occur.
   * @see #getFilePointer()
   */
- (void) seek: (long) pos
{
  [self flush];
  bufferStart = pos;
}

  /** The number of bytes in the file. */
- (long) length
{
  return 0;
}

@end
