package org.apache.lucene.index;

/**
 * Copyright 2004 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
#include "LuceneKit/Index/LCFilterTermDocs.h"

@implementation LCFilterTermDocs

  /** Base class for filtering {@link TermDocs} implementations. */
- (id) initWithTermDocs: (id <LCTermDocs>) docs
{
  self = [super init];
  ASSIGN(in, docs);
  return self;
}

- (void) seekTerm: (LCTerm *) term
{
  [in seekTerm: term];
}

- (void) seekTermEnum: (LCTermEnum *) enum
{
  [in seekTermEnum: enum];
}

- (int) doc
{
  return [in doc];
}

- (int) freq
{
  return [in freq];
}

- (BOOL) next
{
  return [in next];
}

- (int) readDocs: (int *) docs frequency: (int *) freqs
{
  return [in read: docs frequency: freqs];
}

- (BOOL) skipTo: (int) i
{
  return [in skipTo: i];
}

- (void) close
{
  [in close];
}

@end

  /** Base class for filtering {@link TermPositions} implementations. */
@implementation LCFilterTermPositions

- (id) initWithTermPositions: (id <LCTermPositions>) po
{
  return [super initWithTermDocs: po];
}

- (int) nextPosition
{
  return [(id <LCTermPositions>)in nextPosition];
}

@end

@implementation LCFilterTermEnum

  /** Base class for filtering {@link TermEnum} implementations. */
- (id) initWithTermEnum: (LCTermEnum *) enum
{
  self = [super init];
  ASSIGN(in, enum);
  return self;
}

- (BOOL) next
{
  return [in next];
}

- (LCTerm *) term
{
  return [in term];
}

- (int) docFreq
{
  return [in docFreq];
}

- (void) close
{
  [in close];
}

@end

@implementation LCFilterIndexReader

  /**
   * <p>Construct a FilterIndexReader based on the specified base reader.
   * Directory locking for delete, undeleteAll, and setNorm operations is
   * left to the base reader.</p>
   * <p>Note that base reader is closed if this FilterIndexReader is closed.</p>
   * @param in specified base reader.
   */
- (id) initWithIndexReader: (LCIndexReader *) reader
{
  self = [super initWithDirectory: [reader directory]];
  ASSIGN(in, reader);
  return self;
}

- (NSArray *) getTermFreqVectors: (int) docNumber
{
  return [in getTermFreqVectors: docNumber];
}

- (NSArray *) getTermFreqVector: (int) docNumber field: (NSString *) field
{
  return [in getTermFreqVector: docNumber field: field];
}

- (int) numDocs
{
  return [in numDocs];
}

- (int) maxDoc
{
  return [in maxDoc];
}

- (LCDocument *) document: (int) n
{
  return [in document: n];
}

- (BOOL) isDeleted: (int) n
{
  return [in isDeleted: n];
}

- (BOOL) hasDeletions
{
  return [in hasDeletions];
}

- (void) doUndeleteAll
{
  [in undeleteAll];
}

- (char *) norms: (NSString *) f
{
  return [in norms: f];
}

- (void) norms: (NSString *) f bytes: (char *) bytes
         offset: (int) offset
{
  [in norms: f bytes: bytes offset: offset];

}

- (void) doSetNorm: (int) d string: (NSString *) f
             byte: (char) b
{
  [in setNorm: d string: f byte: b];
}

- (LCTermEnum *) terms
{
  return [in terms];
}

- (LCTermEnum *) termsWithTerm: (LCTerm *) t
{
  return [in termsWithTerm: t];
}

- (int) docFreq: (LCTerm *) t
{
  return [in docFreq: t];
}

- (LCTermDocs *) termDocs
{
  return [in termDocs];
}

- (LCTermPositions *) termPositions
{
  return [in termPositions];
}

- (void) doDelete: (int) n
{
  [in delete: n];
}

- (void) doCommit
{
  [in commit];
}

- (void) doClose
{
  [in close];
}

- (NSArray *) getFieldNames
{
  return [in getFieldNames];
}

- (NSArray *) getFieldNamesWithIndexed: (BOOL) indexed
{
  return [in getFieldNamesWithIndexed: indexed];
}

- (NSArray *) getIndexedFieldNames: (LCTermVector_Type) tvSpec
{
  return [in getIndexedFieldNames: tvSpec];
}
  
#if 0
  public Collection getFieldNames(IndexReader.FieldOption fieldNames) {
    return in.getFieldNames(fieldNames);
  }
#endif

@end
