/** FILE: Resolution.m
 *
 * <title>Resolution</title>
 * <date>July 05, 2006</date>
 *
 * Copyright (c) 2006
 * All rights reserved.
 *
 * Written by: Chris B. Vetter <cbv@gmx.net>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *   - Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *
 *   - Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 *   - ALTERNATIVELY, this product may be distributed under the terms of the
 *     GNU Public License, in which case the provisions of the GPL are
 *     required INSTEAD OF the above restrictions.
 *     (This clause is necessary due to a potential bad interaction between
 *     the GPL and the restrictions contained in a BSD-style copyright.)
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 ***************************************************************************/

//
// Include
//

#import "Resolution.h"

#import <AppKit/NSImage.h>
#import <AppKit/NSImageView.h>
#import <AppKit/NSMenu.h>
#import <AppKit/NSMenuItem.h>

#import <Foundation/NSBundle.h>
#import <Foundation/NSInvocation.h>
#import <Foundation/NSTimer.h>

#import <xf86Parser.h>		// XF86CONF_DBLSCAN, XF86CONF_INTERLACE
#import <X11/Xlib.h>
#import <X11/extensions/xf86vmode.h>

#if defined( XINERAMA )
#  import <X11/extensions/Xinerama.h>
#endif

//
// Define
//

#if ! defined( XF86CONF_INTERLACE )
#  define XF86CONF_INTERLACE	0x0010
#endif

#if ! defined( XF86CONF_DBLSCAN )
#  define XF86CONF_DBLSCAN	0x0020
#endif

//
// Typedef
//

//
// Public
//

//
// Private
//

static Display
  *display = NULL;

//
// Private Interface
//

//
// Prototype
//

/***************************************************************************
**
** Implementation
**
*/

@implementation Resolution 

/***************************************************************************
**
** Private Methods
**
*/

/**
 * <em>Description forthcoming</em>
 *
 *
 */

- (void) loadImage
{
  NSBundle
    *bundle = nil;
  NSString
    *path = nil;
  NSImage
    *image = nil;
  
  bundle = [NSBundle bundleForClass: [self class]];
  
  if( bundle )
  {
    path = [bundle pathForResource: @"display"
                            ofType: @"tiff"];
  }
  
  if( path )
  {
    image = [[NSImage alloc] initByReferencingFile: path];
  }
  
  if( image )
  {
    [view setImage: image];
    [view setNeedsDisplay];
    
    [image release];
  }
  
  //
  // That's it
  //
  
  return;
}

/**
 * <em>Description forthcoming</em>
 *
 *
 */

- (int) getNumberOfScreens
{
  int
    screenCount = 0;
#if defined( XINERAMA )
  XineramaScreenInfo
    *screen;
  int
    screenExists[1024],
    i = 0;
#endif /* XINERAMA */
    
  
#if defined( XINERAMA )
  if( XineramaIsActive(display) )
  {
    screen = XineramaQueryScreens(display, screenExists);
    
    for( ; i == screen[i].screen_number; i++)
      ;
    
    screenCount = i;
    XFree(screen);
  }
  else
#endif /* XINERAMA */
  {
    screenCount = XScreenCount(display);
  }
  
  //
  // That's it
  //
  
  return screenCount;
}

/**
 * <em>Description forthcoming</em>
 *
 *
 */

- (void) switchResolution: (id) sender
{
  XF86VidModeModeInfo
    **vm_modelines;
  int
    screenCount = 0,
    vm_count = 0,
    i = 0,
    j = 0;
  
  screenCount = [self getNumberOfScreens];
  
  for( ; i < screenCount; i++ )
  {
    XF86VidModeGetAllModeLines(display, i, &vm_count, &vm_modelines);
    
    for( ; j < vm_count; j++ )
    {
      NSMenuItem
        *item = nil;
      
      if( j == [sender tag] )
      {
        XF86VidModeSwitchToMode(display, i, vm_modelines[j]);
        XFlush(display);
      }
    }
  }
  
  // OK, this is OOGLY but if we call -getResolutions directly, we'll crash
  // We do that to re-create the menu(s)
  {
    NSInvocation
      *inv = nil;
    NSTimer
      *timer = nil;
    
    inv = [NSInvocation invocationWithMethodSignature:
            [self methodSignatureForSelector: @selector(getResolutions)]];
    
    [inv setTarget: self];
    [inv setSelector: @selector(getResolutions)];
    
    timer = [NSTimer scheduledTimerWithTimeInterval: 2.0
                                         invocation: inv
                                            repeats: NO];
  }
  
  //
  // That's it
  //
  
  return;
}

/**
 * <em>Description forthcoming</em>
 *
 *
 */

- (NSString *) createLabel: (XF86VidModeModeInfo **) vm_modelines
                          : (XF86VidModeModeLine) modeline
                          : (int) index
{
  char
    c;
  float
    refresh = 0;
  
  refresh = vm_modelines[index]->dotclock * 1000.0 /
            ( vm_modelines[index]->htotal * vm_modelines[index]->vtotal ) *
            ( (vm_modelines[index]->flags & XF86CONF_DBLSCAN) ? 0.5 : 1 ) *
            ( (vm_modelines[index]->flags & XF86CONF_INTERLACE) ? 2.0 : 1 );
  
  c = ( (modeline.hdisplay == vm_modelines[index]->hdisplay) &&
        (modeline.vdisplay==vm_modelines[index]->vdisplay) ? '*' : ' ' );
  
  //
  // That's it
  //
  
  return [NSString stringWithFormat: @"%c %ix%i %.fHz",
           c,
           vm_modelines[index]->hdisplay,
           vm_modelines[index]->vdisplay,
           refresh];
}

/**
 * <em>Description forthcoming</em>
 *
 *
 */

- (BOOL) getResolutions
{
  if( NULL == (display = XOpenDisplay("")) )
  {
    return NO; 
  }
  else
  {
    NSMenu
      *menu = nil,
      *submenu = nil;
    NSString
      *menuLabel = nil,
      *itemLabel = nil;
    int
      screen = 0,
      screenCount = 0,
      vm_count = 0,
      dotclock = 0;
    XF86VidModeModeInfo
      **vm_modelines;
    XF86VidModeModeLine
      modeline;
    
    screenCount = [self getNumberOfScreens];
    
    menu = [[NSMenu alloc] initWithTitle: @"Resolution"];
    [menu autorelease];
    
    for( ; screen < screenCount; screen++ )
    {
      int
        mode = 0;
      
      if( ! XF86VidModeGetAllModeLines(display, screen, &vm_count, &vm_modelines) )
      {
        return NO;
      }
      
      if( ! XF86VidModeGetModeLine(display, screen, &dotclock, &modeline) )
      {
        return NO;
      }
      
      if( 1 < screenCount )
      {
        menuLabel = [NSString stringWithFormat: @"Screen %i", screen];
        
        [menu addItemWithTitle: menuLabel
                        action: NULL
                 keyEquivalent: nil];
        
        submenu = [[NSMenu alloc] init];
      }
      
      for( ; mode < vm_count; mode++ )
      {
        NSMenuItem
          *item = nil;
        
        item = [[NSMenuItem alloc] init];
        itemLabel = [self createLabel: vm_modelines : modeline : mode];
        
        [item setTitle: itemLabel];
        [item setTarget: self];
        [item setAction: @selector(switchResolution:)];
        [item setKeyEquivalent: nil];
        [item setTag: mode];
        [item setEnabled: YES];
        
        if( 1 < screenCount )
        {
          [submenu addItem: item];
        }
        else
        {
          [menu addItem: item];
        }
        
        [item release];
      }
      
      if( 1 < screenCount )
      {
        [menu setSubmenu: submenu
                 forItem: [menu itemWithTitle: menuLabel]];
        
        [submenu release];
      }
    }
    
    [view setMenu: menu];
  }
    
  //
  // That's it
  //
  
  return YES;
}

/***************************************************************************
**
** Factory Methods
**
*/

/***************************************************************************
**
** Instance Methods
**
*/

/***************************************************************************
**
** Accessor Methods
**
*/

/***************************************************************************
**
** Protocol Methods
**
*/

/**
 * <em>Description forthcoming</em>
 *
 *
 */

- (NSView *) menuletView
{
  //
  // That's it
  //
  
  return [[view retain] autorelease];
}

/***************************************************************************
**
** Delegate Methods
**
*/

/***************************************************************************
**
** Override Methods
**
*/

/**
 * <em>Description forthcoming</em>
 *
 *
 */

- (id) init
{
  if( (self = [super init]) )
  {
    view = [[NSImageView alloc] initWithFrame: NSMakeRect(0, 0, 25, 20)];
    {
      [self loadImage];
      
      if( ! [self getResolutions] )
      {
        NSLog(@"Unable to get list of resolutions");
        
        [self release];
        return nil;
      }
    }
    
    return self;
  }
  
  //
  // That's it
  //
  
  return nil;
}

/**
 * <em>Description forthcoming</em>
 *
 *
 */

- (void) dealloc
{
  if( view )
  {
    [view release];
    view = nil;
    
    XCloseDisplay(display);
    free(display);
  }
  [super dealloc];
  
  //
  // That's it
  //
  
  return;
}

@end

/***************************************************************************
**
** Functions
**
*/

/*
** End of File.
**
****************************************************************************/
