/* 
 * makedumpfile.c
 *
 * Copyright (C) 2006  NEC Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * TODO
 *   1. Get a size of struct and a offset of struct member.
 *      (by the feature of GDB).
 *   2. Get the memory management info by symbol "pgdat_list".
 *   3. Exclude Free Pages.
 */

#include "makedumpfile.h"

/*
 *  Strip line-ending linefeeds in a string.
 */
char *
strip_linefeeds(char *line)
{
	char *p;

	if (line == NULL || strlen(line) == 0)
		return(line);

	p = &LASTCHAR(line);

	while (*p == '\n') {
		*p = NULLCHAR;
		if (--p < line)
			break; 
	}

	return(line);
}

/*
 *  Strip line-beginning and line-ending whitespace and linefeeds.
 */
char *
clean_line(char *line)
{
	line = g_strchug(line);
	strip_linefeeds(line);
	line = g_strchomp(line);
	return(line);
}

/*
 *  Parse a line into tokens, populate the passed-in argv[] array, and return
 *  the count of arguments found.  This function modifies the passed-string 
 *  by inserting a NULL character at the end of each token.  Expressions 
 *  encompassed by parentheses, and strings encompassed by apostrophes, are 
 *  collected into single tokens.
 */
int
parse_line(char *str, char *argv[])
{
	int i, j;
	int string;
	int expression;

	for (i = 0; i < MAXARGS; i++)
		argv[i] = NULL;

	clean_line(str);

	if (str == NULL || strlen(str) == 0)
		return(0);

	i = j = 0;
	string = expression = FALSE;
	argv[j++] = str;

	while (TRUE) {
		if (j == MAXARGS)
			ERRMSG("too many arguments in string!\n");

		while (str[i] != ' ' && str[i] != '\t' && str[i] != NULLCHAR) {
			i++;
		}

		switch (str[i])
		{
		case ' ':
		case '\t':
			str[i++] = NULLCHAR;
	
		    if (str[i] == '"') {    
			str[i] = ' ';
			string = TRUE;
			i++;
		    }

		    if (str[i] == '(') {     
			expression = TRUE;
		    }

		    while (str[i] == ' ' || str[i] == '\t') {
			i++;
		    }

		    if (str[i] != NULLCHAR && str[i] != '\n') {
			argv[j++] = &str[i];
			if (string) {
				string = FALSE;
				while (str[i] != '"' && str[i] != NULLCHAR)
					i++;
				if (str[i] == '"')
					str[i] = ' ';
			}
			if (expression) {
				expression = FALSE;
				while (str[i] != ')' && str[i] != NULLCHAR)
					i++;
			}
			break;
		    }
		/* else fall through */
		case '\n':
		    str[i] = NULLCHAR;
		/* keep falling... */
		case NULLCHAR:
		    argv[j] = NULLCHAR;
		    return(j);
		}
	}
}

/*
 * Convert Physical Address to File Offest.
 *  If this function returns 0x0, File Offset isn't found.
 *  The File Offset 0x0 is in the ELF header.
 *  It is not in the memory image.
 */
off_t
paddr_to_fileoffset(struct DumpInfo *info,  unsigned long paddr)
{
	int i;
	off_t offset;
	struct pt_load_segment *pls;

	for (i = offset = 0; i < info->num_pt_load_segments; i++) {
		pls = &info->pt_load_segments[i];
		if ((paddr >= pls->phys_start)
		    && (paddr < pls->phys_end)) {
			offset = (off_t)(paddr - pls->phys_start) +
				pls->file_offset;
				break;
		}
	}
	return offset;
}

/*
 * Convert Virtual Address to File Offest.
 *  If this function returns 0x0, File Offset isn't found.
 *  The File Offset 0x0 is the ELF header.
 *  It is not in the memory image.
 */
off_t
vaddr_to_fileoffset(struct DumpInfo *info,  unsigned long vaddr)
{
	int i;
	off_t offset;
	struct pt_load_segment *pls;

	for (i = offset = 0; i < info->num_pt_load_segments; i++) {
		pls = &info->pt_load_segments[i];
		if ((vaddr >= pls->virt_start)
		    && (vaddr < pls->virt_end)) {
			offset = (off_t)(vaddr - pls->virt_start) +
				pls->file_offset;
				break;
		}
	}
	return offset;
}

/*
 * Get the number of the page descriptors from the ELF info.
 */
unsigned int
num_page_descriptor(struct DumpInfo *info)
{
	int i;
	off_t max_paddr;
	struct pt_load_segment *pls;

	for (i = 0, max_paddr = 0; i < info->num_pt_load_segments; i++) {
		pls = &info->pt_load_segments[i];
		if (max_paddr < pls->phys_end)
			max_paddr = pls->phys_end;
	}
	return max_paddr / info->page_size;
}

int
readmem(struct DumpInfo *info, unsigned long vaddr, void *bufptr, size_t size)
{
	off_t offset;
	const off_t failed = (off_t)-1;

	/*
	 * Convert Virtual Address to File Offset.
	 */
	if (!(offset = vaddr_to_fileoffset(info, vaddr)))
		return FALSE;

	if (lseek(info->fd_memory, offset, SEEK_SET) == failed) {
		ERRMSG("Can't seek the dump memory(%s). %s\n",
		    info->name_memory, strerror(errno));
		return FALSE;
	}

	if (read(info->fd_memory, bufptr, size) != size) {
		ERRMSG("Can't read the dump memory(%s). %s\n",
		    info->name_memory, strerror(errno));
		return FALSE;
	}

	return size;
}

void
print_usage()
{
	MSG("\n");
	MSG("Usage:\n");
	MSG("  makedumpfile [-c] [-d dump_level] map_file dump_mem dump_file\n");
	MSG("\n");
	MSG("Available options:\n");
	MSG("  [-c]:\n");
	MSG("      This option enables the compression function of each page.\n");
	MSG("\n");
	MSG("  [-d dump_level]:\n");
	MSG("      This is a value of skipping page.\n");
	MSG("      The page type marked in the following table is skipped.\n");
	MSG("\n");
	MSG("      dump  |  zero   cache   cache    user\n");
	MSG("      level |  page   page    private  data\n");
	MSG("     ---------------------------------------\n");
	MSG("         0  |\n");
	MSG("         1  |  X\n");
	MSG("         2  |         X\n");
//	MSG("         3  |  X      X\n");
	MSG("         4  |         X       X\n");
//	MSG("         5  |  X      X       X\n");
//	MSG("         6  |         X       X\n");
//	MSG("         7  |  X      X       X\n");
	MSG("         8  |                          X\n");
//	MSG("         9  |  X                       X\n");
//	MSG("        10  |         X                X\n");
//	MSG("        11  |  X      X                X\n");
//	MSG("        12  |         X       X        X\n");
//	MSG("        13  |  X      X       X        X\n");
//	MSG("        14  |         X       X        X\n");
	MSG("        15  |  X      X       X        X\n");
	MSG("\n");
	MSG("  map_file:\n");
	MSG("      This is a pathname to a system.map file of first-kernel.\n");
	MSG("\n");
	MSG("  dump_mem:\n");
	MSG("      This is a pathname to a first-kernel memory core image.\n");
	MSG("      This argument is generally /proc/vmcore.\n");
	MSG("\n");
	MSG("  dump_file:\n");
	MSG("      This is a pathname to a filename created by this command.\n");
	MSG("\n");
}

int
open_system_map(struct DumpInfo *info)
{
	FILE *map;

	if ((map = fopen(info->name_sysmap, "r")) == NULL) {
		ERRMSG("Can't open the map file(%s). %s\n",
		    info->name_sysmap, strerror(errno));
		return FALSE;
	}

	info->map = map;
	return TRUE;
}

int
open_dump_memory(struct DumpInfo *info)
{
	int fd;

	if ((fd = open(info->name_memory, O_RDONLY)) < 0) {
		ERRMSG("Can't open the dump memory(%s). %s\n",
		    info->name_memory, strerror(errno));
		return FALSE;
	}

	info->fd_memory = fd;
	return TRUE;
}

int
open_dump_file(struct DumpInfo *info)
{
	int fd;

	if ((fd = open(info->name_dumpfile, O_RDWR|O_CREAT|O_EXCL,
	    S_IRUSR|S_IWUSR)) < 0) {
		ERRMSG("Can't open the dump file(%s). %s\n",
		    info->name_dumpfile, strerror(errno));
		return FALSE;
	}

	info->fd_dumpfile = fd;

	return TRUE;
}

int
dump_Elf32_pt_load(struct DumpInfo *info, Elf32_Phdr *prog, int num_pt_load)
{
	struct pt_load_segment *pls;

	if (prog->p_type != PT_LOAD) {
		ERRMSG("%s isn't the dump memory.\n", info->name_memory);
		return FALSE;
	}

	pls = &info->pt_load_segments[num_pt_load];
	pls->phys_start = prog->p_paddr;
	pls->phys_end = pls->phys_start + prog->p_filesz;
	pls->virt_start = prog->p_vaddr;
	pls->virt_end = pls->virt_start + prog->p_filesz;
	pls->file_offset = prog->p_offset;

	return TRUE;
}

int
dump_Elf64_pt_load(struct DumpInfo *info, Elf64_Phdr *prog, int num_pt_load)
{
	struct pt_load_segment *pls;

	if (prog->p_type != PT_LOAD) {
		ERRMSG("%s isn't the dump memory.\n", info->name_memory);
		return FALSE;
	}

	pls = &info->pt_load_segments[num_pt_load];
	pls->phys_start = prog->p_paddr;
	pls->phys_end = pls->phys_start + prog->p_filesz;
	pls->virt_start = prog->p_vaddr;
	pls->virt_end = pls->virt_start + prog->p_filesz;
	pls->file_offset = prog->p_offset;

	return TRUE;
}

int
get_elf_info(struct DumpInfo *info)
{
	int i, flag_elf;
	const off_t failed = (off_t)-1;

	Elf32_Ehdr *elf32;
	Elf64_Ehdr *elf64;
	Elf32_Phdr *load32;
	Elf64_Phdr *load64;
	char header[MIN_ELF_HEADER_SIZE];
	char *tmp_elf_header = NULL;

	/*
	 * Get minimam ELF header info.
	 */
	if (read(info->fd_memory, header, MIN_ELF_HEADER_SIZE)
	    != MIN_ELF_HEADER_SIZE) {
		ERRMSG("Can't read the dump memory(%s). %s\n",
		    info->name_memory, strerror(errno));
		goto out;
	}

	elf32 = (Elf32_Ehdr *)&header[0];
	elf64 = (Elf64_Ehdr *)&header[0];

	if (STRNEQ(elf32->e_ident, ELFMAG)
	    && (elf32->e_ident[EI_CLASS] == ELFCLASS32)) {
		flag_elf = ELF32;
		load32 = (Elf32_Phdr *)
		       &header[sizeof(Elf32_Ehdr)+sizeof(Elf32_Phdr)];
		info->num_pt_load_segments = elf32->e_phnum - 1;
		info->page_size = (uint32_t)load32->p_align;
		info->header_size = (Elf32_Off)load32->p_offset;
	} else if (STRNEQ(elf64->e_ident, ELFMAG)
	    && (elf64->e_ident[EI_CLASS] == ELFCLASS64)) {
		flag_elf = ELF64;
		load64 = (Elf64_Phdr *)
		       &header[sizeof(Elf64_Ehdr)+sizeof(Elf64_Phdr)];
		info->num_pt_load_segments = elf64->e_phnum - 1;
		info->page_size = (uint64_t)load64->p_align;
		info->header_size = (Elf64_Off)load64->p_offset;
	} else {
		ERRMSG("%s isn't the dump memory.\n", info->name_memory);
		goto out;
	}

	/*
	 * FIXME
	 *   If the page_size of 1st-kernel is different from the one of
	 *   capture(2nd)-kernel, the problem will happen.
	 */
	if (!info->page_size)
		info->page_size = (size_t)sysconf(_SC_PAGE_SIZE);
	if (!info->page_size) {
		ERRMSG("Can't get the size of page.\n");
		goto out;
	}
	if (!info->num_pt_load_segments) {
		ERRMSG("Can't get the number of PT_LOAD.\n");
		goto out;
	}
	if (!info->header_size) {
		ERRMSG("Can't get the size of the dump header.\n");
		goto out;
	}

	/*
	 * Get full ELF header info.
	 */
	if ((tmp_elf_header = (char *)g_malloc0(info->header_size)) == NULL) {
		ERRMSG("Can't allocate memory for the ELF header. %s\n",
		    strerror(errno));
		goto out;
	}
	if (lseek(info->fd_memory, 0, SEEK_SET) == failed) {
		ERRMSG("Can't seek the dump memory(%s). %s\n",
		    info->name_memory, strerror(errno));
		goto out;
	}
	if (read(info->fd_memory, tmp_elf_header, info->header_size)
	    != info->header_size) {
		ERRMSG("Can't read the dump memory(%s). %s\n",
		    info->name_memory, strerror(errno));
		goto out;
	}
	if (flag_elf & ELF32) {
		load32 = (Elf32_Phdr *)
		       &tmp_elf_header[sizeof(Elf32_Ehdr)+sizeof(Elf32_Phdr)];
		if ((info->pt_load_segments = (struct pt_load_segment *)
		    g_malloc0(sizeof(struct pt_load_segment) *
		    info->num_pt_load_segments)) == NULL) {
			ERRMSG("Can't allocate memory for the PT_LOAD. %s\n",
			    strerror(errno));
			goto out;
		}
		for (i = 0; i < info->num_pt_load_segments; i++)
			if(!dump_Elf32_pt_load(info, load32 + i, i))
				goto out;
	} else { /* ELF64 */
		load64 = (Elf64_Phdr *)
		       &tmp_elf_header[sizeof(Elf64_Ehdr)+sizeof(Elf64_Phdr)];
		if ((info->pt_load_segments = (struct pt_load_segment *)
		    g_malloc0(sizeof(struct pt_load_segment) *
		    info->num_pt_load_segments)) == NULL) {
			ERRMSG("Can't allocate memory for the PT_LOAD. %s\n",
			    strerror(errno));
			goto out;
		}
		for (i = 0; i < info->num_pt_load_segments; i++)
			if(!dump_Elf64_pt_load(info, load64 + i, i))
				goto out;
	}
	g_free(tmp_elf_header);
	return TRUE;
out:
	if (tmp_elf_header != NULL)
		g_free(tmp_elf_header);
	return FALSE;
}

int
get_symbol_info(struct DumpInfo *info)
{
	unsigned int max_mapnr;
	unsigned long sym_mem_map, sym_num_physpages, sym_smp_num_siblings;
	unsigned long addr_mem_map;
	char buf[BUFSIZE_FGETS], *mapitems[MAXARGS], *endp;

	/*
	 * Get a symbol info from the System.map.
	 */
	sym_mem_map = sym_num_physpages = sym_smp_num_siblings = 0;

	while (fgets(buf, BUFSIZE_FGETS, info->map)) {
		if (parse_line(buf, mapitems) != 3)
			continue;

		if (strcmp(mapitems[2], "mem_map") == 0) {
			sym_mem_map = strtoul(mapitems[0], &endp, 16);
        		if ((!sym_mem_map || sym_mem_map == ULONG_MAX)
			    || strlen(endp) != 0) {
				ERRMSG("The map file has invalid address of %s",
				    mapitems[2]);
				return FALSE;
			}
		}
		if (strcmp(mapitems[2], "smp_num_siblings") == 0) {
			sym_smp_num_siblings = strtoul(mapitems[0], &endp, 16);
        		if ((!sym_smp_num_siblings
			    || sym_smp_num_siblings == ULONG_MAX)
			    || strlen(endp) != 0) {
				ERRMSG("The map file has invalid address of %s",
				    mapitems[2]);
				return FALSE;
			}
		}
		if (strcmp(mapitems[2], "num_physpages") == 0) {
			sym_num_physpages = strtoul(mapitems[0], &endp, 16);
        		if ((!sym_num_physpages
			    || sym_num_physpages == ULONG_MAX)
			    || strlen(endp) != 0) {
				ERRMSG("The map file has invalid address of %s",
				    mapitems[2]);
				return FALSE;
			}
		}
		if ((sym_mem_map != 0) && (sym_num_physpages != 0)
		    && (sym_smp_num_siblings != 0))
			break;
	}

	/*
	 * Get the address of the symbol "mem_map".
	 * FIXME
	 *   If this command get the feature of analysing a struct,
	 *   the info of the memory management will be gotten from "pgdat_list".
	 */
	if (!sym_mem_map) {
		ERRMSG("Can't get the symbol of mem_map.\n");
		return FALSE;
	}
	if (!readmem(info, sym_mem_map, &addr_mem_map, sizeof addr_mem_map)) {
		ERRMSG("Can't get the address of mem_map.\n");
		return FALSE;
	}

	if (!addr_mem_map) {
		ERRMSG("Can't get the address of mem_map.\n");
		return FALSE;
	}
	info->addr_mem_map = addr_mem_map;

	/*
	 * SMP or not.
	 * FIXME
	 *   If this command get the feature of analysing a struct,
	 *   this code will be trash.
	 */
	if (sym_smp_num_siblings)
		info->flag_smp = 1;
	else
		info->flag_smp = 0;
	
	/*
	 * Get the number of the page descriptors.
	 */
	if (sym_num_physpages) {
		if (!readmem(info, sym_num_physpages, &max_mapnr,
		    sizeof max_mapnr) || !max_mapnr)
			max_mapnr = num_page_descriptor(info);
	} else
		max_mapnr = num_page_descriptor(info);

	if (!max_mapnr) {
		ERRMSG("Can't get the number of the page descriptors.\n");
		return FALSE;
	}
	info->max_mapnr = max_mapnr;
	info->len_bitmap = 2*divideup(info->max_mapnr, BITPERBYTE);

	return TRUE;
}

int
get_pagedesc_offset(struct DumpInfo *info)
{
	int size_pagedesc, offset_page_flags, offset_page__count;
	int offset_page_mapping;

	/*
	 * Get a offset of the page_discriptor's member.
	 * FIXME by the feature of GDB.
	 */
	if (info->flag_smp) {
		size_pagedesc = SIZE_PAGEDESC_SMP;
		offset_page_flags = OFFSET_PAGE_FLAGS_SMP;
		offset_page__count = OFFSET_PAGE__COUNT_SMP;
		offset_page_mapping = OFFSET_PAGE_MAPPING_SMP;
	} else {
		size_pagedesc = SIZE_PAGEDESC_UP;
		offset_page_flags = OFFSET_PAGE_FLAGS_UP;
		offset_page__count = OFFSET_PAGE__COUNT_UP;
		offset_page_mapping = OFFSET_PAGE_MAPPING_UP;
	}

	info->size_pagedesc = size_pagedesc;
	info->offset_page_flags  = offset_page_flags;
	info->offset_page__count = offset_page__count;
	info->offset_page_mapping = offset_page_mapping;

	return TRUE;
}

int
initial(struct DumpInfo *info)
{
	if (!get_elf_info(info))
		return FALSE;

	if (!get_symbol_info(info))
		return FALSE;

	if (!get_pagedesc_offset(info))
		return FALSE;

	return TRUE;
}

int
create_dump_header(struct DumpInfo *info)
{
	unsigned int hdr_size;
	struct disk_dump_header *dump_header;

	dump_header = info->dump_header;

	strcpy(dump_header->signature, DISK_DUMP_SIGNATURE);
	dump_header->block_size   = info->page_size;
	dump_header->sub_hdr_size = 1;
	dump_header->max_mapnr    = info->max_mapnr;
	dump_header->nr_cpus      = 1;
	dump_header->bitmap_blocks
	    = divideup(info->len_bitmap, dump_header->block_size);

	hdr_size = 1 +  dump_header->sub_hdr_size;
	info->offset_bitmap1
	    = (off_t)(hdr_size * dump_header->block_size);
	return TRUE;
}

int
write_dump_header(struct DumpInfo *info)
{
	int size;
	const off_t failed = (off_t)-1;

	if (lseek(info->fd_dumpfile, 0, SEEK_SET) == failed) {
		ERRMSG("Can't seek the dump file(%s). %s\n",
		    info->name_dumpfile, strerror(errno));
		return FALSE;
	}
	size = sizeof(struct disk_dump_header);
	if (write(info->fd_dumpfile, info->dump_header, size) != size) {
		ERRMSG("Can't write the dump file(%s). %s\n",
		    info->name_dumpfile, strerror(errno));
		return FALSE;
	}
	return TRUE;
}

void
set_bitmap(char *bitmap, unsigned int pfn, int val)
{
	int byte, bit;

	byte = pfn>>3;
	bit  = pfn & 7;

	if (val)
		bitmap[byte] |= 1<<bit;
	else
		bitmap[byte] &= ~(1<<bit);
}

static inline int
is_on(char *bitmap, int i)
{
	return bitmap[i>>3] & (1 << (i & 7));
}

static inline int
is_memory_hole(char *bitmap, int i)
{
	return !is_on(bitmap, i);
}

static inline int
is_dumpable(char *bitmap, int i)
{
	return is_on(bitmap, i);
}

static inline int
is_in_segs(struct DumpInfo *info, unsigned long long paddr)
{
	if (paddr_to_fileoffset(info, paddr))
		return TRUE;
	else
		return FALSE;
}

static inline int
is_zero_page(unsigned char *buf, size_t page_size)
{
	size_t i;

	for (i = 0; i < page_size; i++)
		if (buf[i])
			return FALSE;
	return TRUE;
}

int
read_cache(struct cache_data *cd)
{
	const off_t failed = (off_t)-1;

	if (lseek(cd->fd, cd->offset, SEEK_SET) == failed) {
		ERRMSG("Can't seek the dump file(%s). %s\n",
		    cd->file_name, strerror(errno));
		return FALSE;
	}
	if (read(cd->fd, cd->buf, cd->cache_size) != cd->cache_size) {
		ERRMSG("Can't read the dump file(%s). %s\n",
		    cd->file_name, strerror(errno));
		return FALSE;
	}
	cd->offset += cd->cache_size;
	return TRUE;
}

int
write_cache(struct cache_data *cd, void *buf, size_t size)
{
	const off_t failed = (off_t)-1;

	memcpy(cd->buf + cd->buf_size, buf, size);
	cd->buf_size += size;

	if (cd->buf_size < cd->cache_size)
		return TRUE;

	if (lseek(cd->fd, cd->offset, SEEK_SET) == failed) {
		ERRMSG("Can't seek the dump file(%s). %s\n",
		    cd->file_name, strerror(errno));
		return FALSE;
	}
	if (write(cd->fd, cd->buf, cd->cache_size) != cd->cache_size) {
		ERRMSG("Can't write the dump file(%s). %s\n",
		    cd->file_name, strerror(errno));
		return FALSE;
	}

	cd->buf_size -= cd->cache_size;
	memcpy(cd->buf, cd->buf + cd->cache_size, cd->buf_size);
	cd->offset += cd->cache_size;
	return TRUE;
}

int
write_cache_bufsz(struct cache_data *cd)
{
	const off_t failed = (off_t)-1;

	if (!cd->buf_size)
		return TRUE;

	if (lseek(cd->fd, cd->offset, SEEK_SET) == failed) {
		ERRMSG("Can't seek the dump file(%s). %s\n",
		    cd->file_name, strerror(errno));
		return FALSE;
	}
	if (write(cd->fd, cd->buf, cd->buf_size) != cd->buf_size) {
		ERRMSG("Can't write the dump file(%s). %s\n",
		    cd->file_name, strerror(errno));
		return FALSE;
	}
	cd->offset  += cd->buf_size;
	cd->buf_size = 0;
	return TRUE;
}

int
create_dump_bitmap(struct DumpInfo *info)
{
	int val;
	unsigned int i, pfn, remain_size;
	unsigned long addr_mem_map, paddr;
	unsigned char *page_cache = NULL, *pcache;
	unsigned int *_count;
	unsigned long *flags, *mapping;
	struct cache_data bm1, bm2;

	bm1.fd         = info->fd_dumpfile;
	bm1.file_name  = info->name_dumpfile;
	bm1.cache_size = BUFSIZE_BITMAP;
	bm1.buf_size   = 0;
	bm1.offset     = info->offset_bitmap1;
	bm1.buf        = NULL;

	bm2.fd         = info->fd_dumpfile;
	bm2.file_name  = info->name_dumpfile;
	bm2.cache_size = BUFSIZE_BITMAP;
	bm2.buf_size   = 0;
	bm2.offset     = bm1.offset + info->len_bitmap/2;
	bm2.buf        = NULL;
	if ((bm1.buf = g_malloc0(BUFSIZE_BITMAP)) == NULL) {
		ERRMSG("Can't allocate memory for 1st-bitmap buffer. %s\n",
		    strerror(errno));
		goto out;
	}
	if ((bm2.buf = g_malloc0(BUFSIZE_BITMAP)) == NULL) {
		ERRMSG("Can't allocate memory for 2nd-bitmap buffer. %s\n",
		    strerror(errno));
		goto out;
	}
	if ((page_cache = g_malloc(info->size_pagedesc*PGMM_CACHED)) == NULL) {
		ERRMSG("Can't allocate memory for the pagedesc cache. %s\n",
		    strerror(errno));
		goto out;
	}

	addr_mem_map = info->addr_mem_map;

	/*
	 * Loop by each pages.
	 */
	for (pfn = 0, paddr = 0; pfn < info->max_mapnr;
	    pfn++, addr_mem_map += info->size_pagedesc,
	    paddr += info->page_size) {

		if ((pfn % PGMM_CACHED) == 0) {
			if (!readmem(info, addr_mem_map, page_cache,
			    info->size_pagedesc * PGMM_CACHED)) {
				g_free(page_cache);
				goto out;
			}
		}
		if ((pfn != 0) && (pfn%PFN_BUFBITMAP) == 0) {
			/*
			 * Write the 1st-bitmap and the 2nd-bitmap.
			 */
			bm1.buf_size = BUFSIZE_BITMAP;
			bm2.buf_size = BUFSIZE_BITMAP;
			if (!write_cache_bufsz(&bm1))
				goto out;
			if (!write_cache_bufsz(&bm2))
				goto out;

			/*
			 * Clear the remainder of the bitmap.
			 */
			if ((info->max_mapnr - pfn) <= PFN_BUFBITMAP) {
				for (i = 0; i < PFN_BUFBITMAP; i++) {
					set_bitmap(bm1.buf, i, 0);
					set_bitmap(bm2.buf, i, 0);
				}
			}
		}
		pcache = page_cache + ((pfn%PGMM_CACHED) * info->size_pagedesc);
		flags = (unsigned long *)(pcache + info->offset_page_flags);
		_count = (unsigned int *)(pcache + info->offset_page__count);
		mapping = (unsigned long *)(pcache + info->offset_page_mapping);

		/*
		 * val  1: dump Page
		 *      0: not dump Page
		 */
		val = 1;

		/*
		 * Exclude the memory hole.
		 */
		if (!is_in_segs(info, paddr))
			val = 0;

		/*
		 * Set the 1st-bitmap.
		 *  val  1: not memory hole
		 *       0: memory hole
		 */
		set_bitmap(bm1.buf, pfn%PFN_BUFBITMAP, val);

		/*
		 * Exclude the cache page without the private page.
		 */
		if ((info->dump_level & DL_EXCLUDE_CACHE)
		    && (isLRU(*flags) || isSwapCache(*flags))
		    && !isPrivate(*flags) && !isAnon(*mapping))
			val = 0;

		/*
		 * Exclude the cache page with the private page.
		 */
		else if ((info->dump_level & DL_EXCLUDE_CACHE_PRI)
		    && (isLRU(*flags) || isSwapCache(*flags))
		    && !isAnon(*mapping))
			val = 0;

		/*
		 * Exclude the data page of the user process.
		 */
		else if ((info->dump_level & DL_EXCLUDE_USER_DATA)
		    && isAnon(*mapping))
			val = 0;

		/*
		 * Set the 2nd-bitmap.
		 *  val  1: dump page
		 *       0: not dump page(the memory hole, or the excluded page)
		 */
		set_bitmap(bm2.buf, pfn%PFN_BUFBITMAP, val);
	}

	/*
	 * Write the remainder of the bitmap.
	 */
	remain_size = (info->offset_bitmap1 + info->len_bitmap/2) - bm1.offset;
	bm1.buf_size = remain_size;
	bm2.buf_size = remain_size;
	if (!write_cache_bufsz(&bm1))
		goto out;
	if (!write_cache_bufsz(&bm2))
		goto out;

	g_free(page_cache);
	g_free(bm1.buf);
	g_free(bm2.buf);
	return TRUE;
out:
	if (page_cache != NULL)
		g_free(page_cache);
	if (bm1.buf != NULL)
		g_free(bm1.buf);
	if (bm2.buf != NULL)
		g_free(bm2.buf);
	return FALSE;
}

static inline void
print_progress(int current, int end)
{
	int progress;
	time_t tm;
	static time_t last_time = 0;

	if (current < end) {
		tm = time(NULL);
		if (tm - last_time < 1)
			return;
		last_time = tm;
		progress = current * 100 / end;
	} else
		progress = 100;

	ERRMSG("\r");
	ERRMSG("[%3d %%]", progress);
}

int
write_pages(struct DumpInfo *info)
{
	unsigned int pfn, num_dump, per = info->max_mapnr/100;
	unsigned int flag_change_bitmap = 0;
	unsigned long size_out;
	struct page_desc pd;
	off_t offset_data;
	struct disk_dump_header *dh = info->dump_header;
	unsigned char *buf = NULL, *buf_out = NULL;
	struct cache_data bm1, bm2, pdesc, pdata;
	const off_t failed = (off_t)-1;

	bm1.fd         = info->fd_dumpfile;
	bm1.file_name  = info->name_dumpfile;
	bm1.cache_size = BUFSIZE_BITMAP;
	bm1.buf_size   = 0;
	bm1.offset     = info->offset_bitmap1;
	bm1.buf        = NULL;

	bm2.fd         = info->fd_dumpfile;
	bm2.file_name  = info->name_dumpfile;
	bm2.cache_size = BUFSIZE_BITMAP;
	bm2.buf_size   = 0;
	bm2.offset     = bm1.offset + info->len_bitmap/2;
	bm2.buf        = NULL;

	pdesc.fd         = info->fd_dumpfile;
	pdesc.file_name  = info->name_dumpfile;
	pdesc.cache_size = info->page_size<<info->block_order;
	pdesc.buf_size   = 0;
	pdesc.buf        = NULL;

	pdata.fd         = info->fd_dumpfile;
	pdata.file_name  = info->name_dumpfile;
	pdata.cache_size = info->page_size<<info->block_order;
	pdata.buf_size   = 0;
	pdata.buf        = NULL;
	if ((buf = g_malloc(info->page_size)) == NULL) {
		ERRMSG("Can't allocate memory for the page. %s\n",
		    strerror(errno));
		goto out;
	}
	if ((buf_out = g_malloc(info->page_size)) == NULL) {
		ERRMSG("Can't allocate memory for the compression buffer. %s\n",
		    strerror(errno));
		goto out;
	}
	if ((bm1.buf = g_malloc0(BUFSIZE_BITMAP)) == NULL) {
		ERRMSG("Can't allocate memory for 1st-bitmap buffer. %s\n",
		    strerror(errno));
		goto out;
	}
	if ((bm2.buf = g_malloc0(BUFSIZE_BITMAP)) == NULL) {
		ERRMSG("Can't allocate memory for 2nd-bitmap buffer. %s\n",
		    strerror(errno));
		goto out;
	}
	if ((pdesc.buf = g_malloc(pdesc.cache_size + info->page_size))
	    == NULL) {
		ERRMSG("Can't allocate memory for the page desc buffer. %s\n",
		    strerror(errno));
		goto out;
	}
	if ((pdata.buf = g_malloc(pdata.cache_size + info->page_size))
	    == NULL) {
		ERRMSG("Can't allocate memory for the page data buffer. %s\n",
		    strerror(errno));
		goto out;
	}

	if (lseek(info->fd_dumpfile, info->offset_bitmap1 + info->len_bitmap/2,
	    SEEK_SET) == failed) {
		ERRMSG("Can't seek the dump file(%s). %s\n",
		    info->name_dumpfile, strerror(errno));
		goto out;
	}

	/*
	 * Calculate the offset of the page data.
	 */
	for (pfn = 0, num_dump = 0; pfn < dh->max_mapnr; pfn++) {
		if ((pfn%PFN_BUFBITMAP) == 0) {
			if (!read_cache(&bm2))
				goto out;
		}
		if (is_dumpable(bm2.buf, pfn%PFN_BUFBITMAP))
			num_dump++;
	}
	bm2.offset = bm1.offset + info->len_bitmap/2;
	pdesc.offset
	    = (1 + dh->sub_hdr_size + dh->bitmap_blocks) * dh->block_size;
	pdata.offset = pdesc.offset + sizeof(page_desc_t) * num_dump;

	/*
	 * Set a fileoffset of Physical Address 0x0.
	 */
	if (lseek(info->fd_memory, info->header_size, SEEK_SET) == failed) {
		ERRMSG("Can't seek the dump memory(%s). %s\n",
		    info->name_memory, strerror(errno));
		goto out;
	}
	offset_data = pdata.offset;

	/*
	 * Loop by each pages.
	 */
	for (pfn = 0; pfn < info->max_mapnr; pfn++) {

		if ((pfn % per) == 0)
			print_progress(pfn, info->max_mapnr);

		if ((pfn % PFN_BUFBITMAP) == 0) {
			if (flag_change_bitmap) {
				bm2.buf_size = BUFSIZE_BITMAP;
				bm2.offset  -= BUFSIZE_BITMAP;
				if (!write_cache_bufsz(&bm2))
					goto out;
			}
			if (!read_cache(&bm1) || !read_cache(&bm2))
				goto out;
			flag_change_bitmap = 0;
		}

		/*
		 * Check the memory hole.
		 */
		if (is_memory_hole(bm1.buf, pfn%PFN_BUFBITMAP))
			continue;
		/*
		 * Check the excluded page.
		 */
		if (!is_dumpable(bm2.buf, pfn%PFN_BUFBITMAP)) {
			if (lseek(info->fd_memory, info->page_size, SEEK_CUR)
			    == failed) {
				ERRMSG("Can't seek the dump memory(%s). %s\n",
				    info->name_memory, strerror(errno));
				goto out;
			}
			continue;
		}

		if (read(info->fd_memory, buf, info->page_size)
		    != info->page_size) {
			ERRMSG("Can't read the dump memory(%s). %s\n",
			    info->name_memory, strerror(errno));
			goto out;
		}

		/*
		 * Exclude the page filled with zeros.
		 */
		if ((info->dump_level & DL_EXCLUDE_ZERO)
		    && is_zero_page(buf, info->page_size)) {
			set_bitmap(bm2.buf, pfn%PFN_BUFBITMAP, 0);
			flag_change_bitmap = 1;
			continue;
		}

		/*
		 * Compress the page data.
		 */
		size_out = info->page_size;
		if (info->flag_compress
		    && (compress2(buf_out, &size_out, buf, info->page_size,
		    Z_BEST_SPEED) == Z_OK) && (size_out < info->page_size)) {
			pd.flags      = 1;
			pd.size       = size_out;
			memcpy(buf, buf_out, pd.size);
		} else {
			pd.flags      = 0;
			pd.size       = info->page_size;
		}
		pd.page_flags = 0;
		pd.offset     = offset_data;

		/*
		 * Write the page header.
		 */
		if (!write_cache(&pdesc, &pd, sizeof(page_desc_t)))
			goto out;

		/*
		 * Write the page data.
		 */
		if (!write_cache(&pdata, buf, pd.size))
			goto out;

		offset_data  += pd.size;
	}

	/*
	 * Write the remainder.
	 */
	if (flag_change_bitmap) {
		bm2.buf_size = BUFSIZE_BITMAP;
		bm2.offset  -= BUFSIZE_BITMAP;
		if (!write_cache_bufsz(&bm2))
			goto out;
	}
	if (!write_cache_bufsz(&pdesc))
		goto out;
	if (!write_cache_bufsz(&pdata))
		goto out;

	/*
	 * Print the progress of the end.
	 */
	print_progress(info->max_mapnr, info->max_mapnr);

	g_free(buf);
	g_free(buf_out);
	g_free(bm1.buf);
	g_free(bm2.buf);
	g_free(pdesc.buf);
	g_free(pdata.buf);
	return TRUE;
out:
	if (buf != NULL)
		g_free(buf);
	if (buf_out != NULL)
		g_free(buf_out);
	if (bm1.buf != NULL)
		g_free(bm1.buf);
	if (bm2.buf != NULL)
		g_free(bm2.buf);
	if (pdesc.buf != NULL)
		g_free(pdesc.buf);
	if (pdata.buf != NULL)
		g_free(pdata.buf);
	return FALSE;
}

void
close_system_map(struct DumpInfo *info)
{
	if (fclose(info->map) < 0)
		ERRMSG("Can't close the map file(%s). %s\n",
		    info->name_sysmap, strerror(errno));
}

void
close_dump_memory(struct DumpInfo *info)
{
	if (close(info->fd_memory) < 0)
		ERRMSG("Can't close the dump memory(%s). %s\n",
		    info->name_memory, strerror(errno));
}

void
close_dump_file(struct DumpInfo *info)
{
	if (close(info->fd_dumpfile) < 0)
		ERRMSG("Can't close the dump file(%s). %s\n",
		    info->name_dumpfile, strerror(errno));
}

int
main(int argc, char *argv[])
{
	int opt;
	struct DumpInfo *info = NULL;

	if ((info = g_malloc0(sizeof(struct DumpInfo))) == NULL) {
		ERRMSG("Can't allocate memory for the pagedesc cache. %s.\n",
		    strerror(errno));
		goto out;
	}
	if ((info->dump_header = g_malloc0(sizeof(struct disk_dump_header)))
	    == NULL) {
		ERRMSG("Can't allocate memory for the dump header. %s\n",
		    strerror(errno));
		goto out;
	}

	while ((opt = getopt(argc, argv, "b:cd:")) != -1) {
		switch (opt) {
		case 'b':
			info->block_order = atoi(optarg);
			break;
		case 'c':
			info->flag_compress = 1;
			break;
		case 'd':
			info->dump_level = atoi(optarg);
			break;
		case '?':
			ERRMSG("Commandline parameter is invalid.\n");
			print_usage();
			goto out;
		}
	}
	if (optind + 3 != argc) {
		ERRMSG("Commandline parameter is invalid.\n");
		print_usage();
		goto out;
	}
	info->name_sysmap   = argv[optind];
	info->name_memory   = argv[optind+1];
	info->name_dumpfile = argv[optind+2];

	if ((info->dump_level < MIN_DUMP_LEVEL)
	    || (MAX_DUMP_LEVEL < info->dump_level)) {
		ERRMSG("Dump_level is invalid.\n");
		print_usage();
		goto out;
	}

	if (!open_system_map(info))
		goto out;

	if (!open_dump_memory(info))
		goto out;

	if (!open_dump_file(info))
		goto out;

	if (!initial(info))
		goto out;

	if (!create_dump_header(info))
		goto out;

	if (!write_dump_header(info))
		goto out;

	if (!create_dump_bitmap(info))
		goto out;

	if (!write_pages(info))
		goto out;

	close_system_map(info);

	close_dump_memory(info);

	close_dump_file(info);

	MSG("\n");
	MSG("The dumpfile is saved to %s.\n", info->name_dumpfile);
	MSG("makedumpfile Completed.\n");
	g_free(info->pt_load_segments);
	g_free(info->dump_header);
	g_free(info);
	return COMPLETED;

out:
	ERRMSG("\n");
	ERRMSG("makedumpfile Failed.\n");
	if (info->map)
		fclose(info->map);
	if (info->fd_memory)
		close(info->fd_memory);
	if (info->fd_dumpfile)
		close(info->fd_dumpfile);
	if (info->pt_load_segments != NULL)
		g_free(info->pt_load_segments);
	if (info->dump_header != NULL)
		g_free(info->dump_header);
	if (info != NULL)
		g_free(info);
	return FAILED;
}
