#!/usr/bin/perl -w
# fdsssl.pl - script to setup and manage ssl and ssl keys with fds 1.1+
# Ryan Braun ryan.braun@ec.gc.ca

# deps: libsys-hostname-long-perl
#	libnet-ldap-perl

use strict;
use Sys::Hostname::Long;
use Sys::Hostname;
use Getopt::Long;
use File::Copy;
use Net::LDAP;

#use vars qw($opt_h $opt_n $opt_8 $opt_c $opt_i $opt_l $opt_d $opt_x $opt_q $opt_w $opt_m $opt_p $opt_a $opt_t);

#&GetOptions('h','n=s','8=s', 'c', 'i', 'l', 'd=s', 'x:s', 'q', 'w', 'm=s', 'p=s', 'a=s', 't') || exit 1;

# TODO puke if not root

my %opt;
GetOptions(
    \%opt,
    	'h',		# display help
	'q',		# initialize cert database files
	'w',		# create CAcert certificate
	'c',		# create Server-Cert ceritificate
	'i',		# init mode,  same as -qwc
	'n=s',		# hostname to use,  generally want to pass the FQDN hostname
	'8=s',		# use altDNS extensions when creating Server Cert,  comma separated ex -8 "fds,fds.ldap.com,192.168.0.10"
	'l',		# list certs in the database
	'd=s',		# CERTNAME to list detailed info about,  eg -l -d "CACert" or -l -d "Server-Cert"
	'x=s',		# extract Server-Cert to file, must pass cert name from a -l listing eg -x "Server-Cert-fds"
	'm=s',		# import a cert into the database.  must pass the cert filename eg -m /tmp/cacert.asc or -m /tmp/Server-Cert.p12
	'p=s',		# use pwd file from the CACert during a -m import,  otherwise you'll be prompted for the password used to create the exported cert
	'a=s',		# use an alternate config file rather then /etc/fdstools/ssl.conf
	't',		# turn on ssl on fds running on the localhost,  will make perl ldap calls to enable	
	'e'		# show fdssl examples

);
# option defs
# h - display help TODO
# n - hostname
# -8 - alt dns extentions comma separated and quoted
# c - create Server-Cert,  use with -n and -8, and -x to extract the cert
# i - init CA cert database.  will create db, CAcert and a Server-Cert for the localhost
# l - list certs in db,  use -d for detail on a certain cert
# d - string to lookup cert info,  get it from -l
# x - extract the Server-Cert to file
# q - init cert database
# w - create CAcert
# k - 
# a - use an alternate conf file rather then $conf
# m - import a key into the db
# p - pwdfile from the CA used to gen the p12 file being imported with -m

my $conf = "/etc/fdstools/ssl.conf";
my $certutil = "/usr/bin/certutil";
my $pk12util = "/usr/bin/pk12util";
my %config;
my (@key_serial, @key_hostname, @key_description);

# run initial subs to setup script
load_config();
getserial();
gethostname();

# TODO verifry both certutil and pk12util exist

if (defined($opt{h})) {
	showHelp();
	exit 0;
}

if (defined($opt{e})) {
	showExamples();
	exit 0;
}

# import keys into db
if (defined($opt{m})) {
	importKey();
	exit 0;
}
# list certs if -l
if (defined($opt{l})) {
	listCerts();
	exit 0;
}

if (defined($opt{t})) {
	enable_ssl();
	exit 0;
}

# extract cert
# don't extract if -c is given aswell, we need to create it first then extract.
if ( (defined($opt{x})) && (!defined($opt{c})) ) {
	extractCert($opt{x});
	exit 0;
}

# init mode,  create cb, cacert and server cert for the LOCALHOST
if (defined($opt{i})) {

	# several puke factors,  
	if (defined($opt{n})) {
		print "You can't run init mode with any -n HOSTNAME args,  quitting\n";
		exit 1;
	}
	if ((defined($opt{q})) || (defined($opt{w})) || (defined($opt{c}))) {
		print "You can't run init mode any of -c -w -q,  quitting\n";
		exit 1;
	}
	check_noise();
	check_pwdfile();
	createDB();
	createCA();
	createCert();
	exit 0;
	
}
# init db if -q is passed
if (defined($opt{q})) {
	check_noise();
	check_pwdfile();
	createDB();
	exit 0;
}

# create CACert if -w is passed
if (defined($opt{w})) {
	check_noise();
	check_pwdfile();
	createCA();
	exit 0;
}

# create a Server-Cert
if (defined($opt{c})) {
	check_noise();
	check_pwdfile();
	createCert();
	exit 0;
}

# no options,  display help

showHelp();
exit 0;


##################### subs ##############################3

# 
 sub showHelp {

	# display help
	print "fdsssl.pl - setup and manage certificates and certificate databases for Fedora Directory Server\n";
	print "It's essentially a glorifed wrapper for certutil and pk12util.\n\n";
	print "OPTIONS:

		-q	-  initialize the database files at location $config{SEC_DIR}  
				Probably a good idea to move old files to a backup locations, and this will generate new ones.
				
		-w	-  Create a CACert in the database located at $config{SEC_DIR}
				
		-c	-  Create a Server-Cert in the database located at $config{SEC_DIR}
				Additional switches used
					-n -- FQDN of the host you're creating the cert for. REQUIRED if the cert isn't for localhost
					-8 -- Comma separated list of altdns extension hostnames to add to the cert with. [optional]
		-i	-  Init mode.  Will run -q -w -c all in one shot.  Good for setting up the initial db on your CACert generating host
				but don't run on any other servers as it will create additional CACerts.
				
		-l	-  List all the certs in the database located at $config{SEC_DIR}
				Additional switches used
					-d -- Cert name to get detailed info about.  
						use -l to get the list of names, then run again with -l -d CERT_NAME

		-x	-  Extract a Server-Cert to a $config{SEC_DIR}/Server-Cert.p12 takes the Server-Cert name as the argument
					eq -x Server-Cert-fds
					Cert names are listed by running a fdsssl -l

		-m	-  Import a cert into an existing database,  will take either a .asc cert or .p12 as the argument
				ie -m /tmp/cacert.asc or -m /tmp/Server-Cert-fds.p12
				Additional switches used
					-p /path/to/pwdfile used to encrypt p12 Server Cert.  This is the pwdfile on the machine that
						created the cert.  If omitted,  you will be prompted for the password.
		
		-a	-  Specify an alternate config file. ie  -a /path/to/newssl.conf
		-t	-  Turn on ssl on localhost for instance $config{INSTANCE} with perl ldap calls
				Will bind as cn=Directory Manager
		-e	-  Show example commands

		\n\n\n";

 }

sub showExamples {


	print "\n\n\t\tEXAMPLES

		
		initial setup for your CACert signing instance of fds (you only want one of these)
		Remove/backup your existing $config{SEC_DIR}/key3.db cert8.db secmod.db file first
		
			fdsssl -i
	
		Then turn on ssl for your server
		
			fdsssl -t

		If you're only using one server you should be done here

		Create a Server-Cert for another Server

			fdsssl -c -n \"fds.test.com\"
			fdsssl -c -n \"fds.test.com\" -8 \"fds,192.168.0.100\"

		EXtract the cert you just created 
			
			fdsssl -l	(to list the available certs)
			fdsssl -x \"Server-Cert-test\"
			NOTE:  This is assuming you have custom cert names turned on in the config file.
				otherwise you'll end up with a bunch of certs all named Server-Cert
			This will create $config{SEC_DIR}/Server-Cert-test.p12

		Rinse and repeat for any other FDS servers you want to use SSL on.

		Importing keys into other servers

		Once you have your CACert signing server up and configured,  you need to import the keys/CAs generated on that machine.
		scp (or however you want to transfer the cacert.asc and Server-Cert.p12 file to the new machine.  You can also transfer 
		over the CACert machine's $config{SEC_DIR}/pwdfile for importing the cert into the new machine.  
			* you don't need to copy the pwdfile,  but you will be prompted for its contents when importing the key *

		On the new server,  remove the old cert8.db, key3.db and secmod.db files in $config{SEC_DIR} (doesn't hurt to back those up I guess)

		Now init the database on this machine

			fdsssl -q
	
		When done,  import the CACert

			fdsssl -m /tmp/cacert.asc

		When done,  import the Server-Cert
			
			fdsssl -m /tmp/Server-Cert-test.p12	(you will be prompted for the password used to create the Cert)
			or
			fdsssl -m /tmp/Server-Cert-test.p12 -p /path/to/CACerts/pwdfile.  
				* Remember,  this is not the pwdfile on the current machine (unless you have specified the same password of course ) *

		Then turn on ssl 

			fdsssl -t
	
		You *should* now be running fds with ssl mode enabled. Yay.\n\n\n";
		

		
}
	

sub importKey {

	# check to make sure the file passed is valid
	if ( ! -r $opt{m} ) {
		print "\n\nunable to read file $opt{m}, quitting\n\n";
		exit 1;
	}
	my @importkey;
	# check to see if we're importing a CA or a Server cert
	if ( $opt{m} =~ m/asc/ ) {
		#import the cacert in asc form
		print "passed an asc file\n";
		@importkey = ("$certutil", "-A",
				"-d $config{SEC_DIR}",
				"-P slapd-$config{INSTANCE}-",
				"-n \"CA certificate\"",
				"-t \"CT,,\"",
				"-a",
				"-i $opt{m}");
	} 

	if ( $opt{m} =~ m/p12/ ) {
		# import the p12 key
		print "passed the p12 file\n";
		#pk12util -i /tmp/Server-Cert-twisted/Server-Cert-twisted.p12 -d /etc/dirsrv/slapd-fds/ -P slapd-fds- -w /etc/dirsrv/slapd-fds/pwdfile-ca -k /etc/dirsrv/slapd-fds/pwdfile
		@importkey = ("$pk12util",
				"-i $opt{m}",
				"-d $config{SEC_DIR}",
				"-P slapd-$config{INSTANCE}-",
				"-k $config{SEC_DIR}/pwdfile");
		# if we passed the CA's pwdfile then use it
		if (defined($opt{p})) {
			# verify file exists
			if ( -r $opt{p} ) {
				# push argument @importkey
				push @importkey, "-w $opt{p}";
			} else {
				print "Unable to locate $opt{p}\n";
				print "You will be prompted below for the password used to encrypt the key you're importing\n";
				print "This is the contents of the pwdfile on the CAcert machine.\n\n";
			}
		} else {
			print "You will be prompted below for the password used to encrypt the key you're importing\n";
			print "This is the contents of the pwd file on the CAcert machine.\n\n";
		}

	}

	print "Importing key now\n\n";
	my $ret = system("@importkey");
}

sub extractCert {
	
	# will extract a cert from the db for import in another db, generally another fds cert database
	# only argument is the Server-Cert string to extract,  use -l to find it
	# TODO
	# the passed variable will need to have any whitespace escaped prior to it recieving it most likely
	my $name = shift;
	
	my @extractcert = ("$pk12util", "-o $config{SEC_DIR}/$name.p12",
				"-d $config{SEC_DIR}",
				"-P slapd-$config{INSTANCE}-",
				"-n $name",
				"-k $config{SEC_DIR}/pwdfile",
				"-w $config{SEC_DIR}/pwdfile");

	my $ret = system("@extractcert");
	if ( $ret > 0 ) {
		print "\n\nError extracting cert for $name\n";
		print "Command executed - @extractcert\n";
		print "Quitting\n\n";
		exit 1;
	} else {
		print "\t\tSuccesfully wrote cert file to $config{SEC_DIR}/$name.p12\n";
	}


}


sub listCerts {
	
	# list the certs in the database
	# will list all the cert names unless -d is specified with the specific cert to show its details.

	my @listcert = ("$certutil", "-L",
			"-P slapd-$config{INSTANCE}-",
			"-d $config{SEC_DIR}");

	if (defined($opt{d})) {
		# the the arg contains a space,  we need to escape it
		$opt{d} =~ s/ /\\ /;
		push @listcert, "-n $opt{d}";
	}

	my $ret = system("@listcert");
	if ( $ret > 0 ) {
		print "\n\nError listing certs,  perhaps you passed an invalid -d string? Try just -l\nQuitting\n\n";
		exit 1;
	}

}


sub createDB {
	
	print "\t\tChecking status of Database files\n";
	my $gendb = 0;
	if ( ! -f "$config{SEC_DIR}/slapd-$config{INSTANCE}-cert8.db" ) {
		$gendb = 1;
	}
	if ( ! -f "$config{SEC_DIR}/slapd-$config{INSTANCE}-key3.db" ) {
		$gendb = 1;
	}
	if ( ! -f "$config{SEC_DIR}/secmod.db" ) {
		$gendb = 1;
	}
	# create db if needed
	if ( $gendb ) {
		print "\tGenerating new Database files";
		my $ret = system("$certutil -N -d $config{SEC_DIR} -f $config{SEC_DIR}/pwdfile -P slapd-$config{INSTANCE}-");
		if ( $ret > 0 ) {
			print "\n\nThere was an error creating database files, quitting\n\n";
			exit 1;
		} else {
			print "\t--->Succesfully generated Database files\n";
		}
	} else {
		print "\tDatabase files seem intact, continuing\n";
	}

	# create symlinks too
	symlink("$config{SEC_DIR}/slapd-$config{INSTANCE}-key3.db","$config{SEC_DIR}/key3.db");
	symlink("$config{SEC_DIR}/slapd-$config{INSTANCE}-cert8.db","$config{SEC_DIR}/cert8.db");

	# find the user that fds runs at by looking at the ownership of the SEC_DIR
	my @stat = stat($config{SEC_DIR});
	chown $stat[4], $stat[5], "$config{SEC_DIR}/slapd-$config{INSTANCE}-key3.db";
	chown $stat[4], $stat[5], "$config{SEC_DIR}/slapd-$config{INSTANCE}-cert8.db";
	chown $stat[4], $stat[5], "$config{SEC_DIR}/secmod.db";
	chmod 0600, "$config{SEC_DIR}/slapd-$config{INSTANCE}-key3.db";
	chmod 0600, "$config{SEC_DIR}/slapd-$config{INSTANCE}-cert8.db";
	chmod 0600, "$config{SEC_DIR}/secmod.db";
	
}


sub createCert {

 #   certutil -S $prefixarg -n "Server-Cert" -s "cn=$myhost,ou=Fedora Directory Server" -c "CA certificate" -t "u,u,u" -m 1001 -v 120 -d $secdir -z $secdir/noise.txt -f $secdir/pwdfile.txt
	#create the Server-Cert
	# Creating certs using the name Server-Cert-$hostname to keep track of easier
	my $cert_name;

	# check to make sure we have a FQDN
	if ( $config{HOSTNAME_FQDN} eq "NOFQDN" ) {
		print "\n\nYou must specify a FQDN type hostname with the -n switch to create a cert,  quitting\n\n";
		exit 1;
	}


	if ( $config{SERV_CERT} ) {
		#define custom Server-Cert name here
			$cert_name = "Server-Cert-$config{HOSTNAME}";
		} else {
			$cert_name = "Server-Cert";
	}
	my @createcert = ("$certutil", "-S",
			"-P slapd-$config{INSTANCE}-",
			"-n $cert_name",
			"-s \"cn=$config{HOSTNAME_FQDN},ou=Fedora Directory Server\"",
			"-c \"CA certificate\"",
			"-t \"u,u,u\"",
			"-m ". scalar(@key_serial),
			"-v $config{EXPIRE}",
			"-d $config{SEC_DIR}",
			"-z $config{SEC_DIR}/noise",
			"-f $config{SEC_DIR}/pwdfile");

	if (defined($opt{8})) {
		# push the alt-dns entries onto the command array if arg was given
		push @createcert, "-8 $opt{8}";
	}

	my $ret = system("@createcert");
	if ( $ret > 0 ) {
		print "\n\nThere was an error creating $cert_name for $config{HOSTNAME_FQDN}\n\n";
			exit 1;
		} else {
			print "\t--->Succesfully generated $cert_name for $config{HOSTNAME_FQDN}\n";
			push @key_serial, scalar(@key_serial);
			push @key_hostname, $config{HOSTNAME_FQDN};
			push @key_description, "$cert_name\n";
			update_serial();
		}

	if (defined($opt{x})) {
		# export the cert we just created
		extractCert("$cert_name");
	}
}

sub createCA {

	
	# check to make sure there isn't a CACert already
	if ( checkCA() ) {
		print "Already found a ca cert,  quitting\n";
		exit 1;
	} 

	print "\t\tTrying to create encryption key\n";

	# create encryption key first
	my @createkey = ("$certutil","-G",
			"-P slapd-$config{INSTANCE}-",
			"-d $config{SEC_DIR}",
			"-z $config{SEC_DIR}/noise",
			"-f $config{SEC_DIR}/pwdfile",
			"1>","/dev/null");
	my $ret = system("@createkey");

	if ( $ret > 0 ) {
		print "\n\nError generating encryption key,  quitting\n\n";
		exit 1;
	} else {
		print "\t--->Succesfully created encryption key<---\n";
	}

	print "\t\tTrying to create CA cert\n";
	# next create CA cert
	my @createca = ("$certutil", "-S ", 
			"-P  slapd-$config{INSTANCE}-",
			"-n \"CA certificate\"", 
			"-s \"cn=CAcert\"",
			"-x", "-t \"CT,,\"", 
			"-m " . scalar(@key_serial),
			"-v $config{EXPIRE}", 
			"-d $config{SEC_DIR}", 
			"-z $config{SEC_DIR}/noise", 
			"-f $config{SEC_DIR}/pwdfile",
			"1>","/dev/null");

	$ret = system("@createca");

	if ( $ret > 0 ) {
		print "\n\nError creating CAcert,  quitting\n";
		exit;
	} else {
		print "\t--->Succesfully created CAcert<---\n";
		# add new serial data to arrays and update serial file
		push @key_serial, scalar(@key_serial);
		push @key_hostname, hostname_long;
		push @key_description, "CACert\n";
		update_serial();
	}

	print "\t\tExporting CAcert key is asc form\n";
	# export the cacert to asc
	my @exportca = ("$certutil", "-L", 
			"-P slapd-$config{INSTANCE}-",
			"-d $config{SEC_DIR}",
			"-n \"CA certificate\"",
			"-a",
			">", "$config{SEC_DIR}/cacert.asc");
	$ret = system("@exportca");
	if ( $ret > 0 ) {
		print "\n\nError generating $config{SEC_DIR}/cacert.asc,  quitting\n\n";
		exit 1;
	} else {
		print "Succesfully created $config{SEC_DIR}/cacert.asc\n";
	}


#    pk12util -d $secdir $prefixarg -o $secdir/cacert.p12 -n "CA certificate" -w $secdir/pwdfile.txt -k $secdir/pwdfile.txt
	print "\t\tExporting the CA key/cert to cacert.p12\t--->";
	my @keycert = ("$pk12util",
			"-d $config{SEC_DIR}",
			"-P slapd-$config{INSTANCE}-",
			"-o $config{SEC_DIR}/cacert.p12",
			"-n \"CA certificate\"",
			"-w $config{SEC_DIR}/pwdfile",
			"-k $config{SEC_DIR}/pwdfile");
	$ret = system("@keycert");

	if ( $ret > 0 ) {
		print "\n\nError generating $config{SEC_DIR}/cacert.p12,  quitting\n\n";
		exit 1;
	} 

}

sub checkCA {
	#certutil -L -d $secdir -n "CA certificate"
	#print "
	my $ret = system("$certutil -L -P slapd-$config{INSTANCE}- -d $config{SEC_DIR} -n \"CA certificate\" >/dev/null 2>&1");
	if ( $ret > 0 ) {
		# no cacert exists
		return 0;
	} else {
		# cacert exists
		return 1;
	}
}

sub check_pwdfile {

	print "\t\tChecking status of pwdfile\n";
	if ( ! -f "$config{SEC_DIR}/pwdfile" ) {
		print "No pwdfile detected, enter password or hit enter for random [RANDOM]: ";
		chomp(my $pwd = <STDIN>);
		if ( $pwd eq "" ) {
			my $ret = system("(ps -ef ; w ) | sha1sum | awk \'{print \$1}\' > $config{SEC_DIR}/pwdfile");
			if ($ret > 0) {
			print "Error occured during pwdfile generation,  quitting\n";
			exit 1;
			}
		} else {
			# change this to a perl open then print statments eventually
			open PWD, ">$config{SEC_DIR}/pwdfile";
			#my $ret = system("echo $pwd > $config{SEC_DIR}/pwdfile");
			print PWD $pwd;
			close PWD;
		}

	} else {
		print "\tUsing existing pwdfile\t\t--->$config{SEC_DIR}/pwdfile<---\n";
	}
	# must make sure the pwd file is owned and readable by the user fds runs as ONLY
	# determine user as the one who owns $config{SEC_DIR}
	my @stat = stat($config{SEC_DIR});
	chown $stat[4], $stat[5], "$config{SEC_DIR}/pwdfile";
	chown $stat[4], $stat[5], "$config{SEC_DIR}/pin.txt";
	chmod 0400, "$config{SEC_DIR}/pwdfile";
	chmod 0600, "$config{SEC_DIR}/pin.txt";

}


sub check_noise {

	print "\t\tChecking status of noise file\n";
	if ( ! -f "$config{SEC_DIR}/noise"	 ) {
		print "\tGenerating noise file $config{SEC_DIR}/noise\n";
		# generate noise file by piping the results of find /tmp to sha1sum
		my $ret = system("(find /tmp)|sha1sum > $config{SEC_DIR}/noise");
		if ($ret > 0) {
			print "Error occured during noise file generation,  quitting\n\n";
			exit 1;
		}
	} else {
		print "\tUsing existing noise file\t--->$config{SEC_DIR}/noise<---\n";
	}
}

sub gethostname {

	# get the hostname info,  either from the passed -n arg or the local info
	# then load them up into the config hash
	if (defined($opt{n})) {
		if ( $opt{n} =~ m/\./ ) {
			#hostname contains a period,  likely the FQDN
			$config{HOSTNAME_FQDN} = $opt{n};
			my @hn = split(/\./,$config{HOSTNAME_FQDN});
			$config{HOSTNAME} = $hn[0];
		} else {
			# passed arg was just a hostname, no domain info
			$config{HOSTNAME} = $opt{n};
			# set FQDN to 0
			$config{HOSTNAME_FQDN} = "NOFQDN";
		}
	} else {
		# grab local hostname info
		$config{HOSTNAME} = hostname;
		$config{HOSTNAME_FQDN} = hostname_long;
	}
	#print "FQDN = $config{HOSTNAME_FQDN}\n";
	#print "hostname = $config{HOSTNAME}\n";
}

sub enable_ssl {
	
	# check if $config{SERV_CERT} is enabled, and -d wasn't specified,  list certs then prompt to enter the certname to use.
	#
	my $cert_name;

	if (($config{SERV_CERT}) && (!defined($opt{d}))) {
		print "You have custom Server-Cert names enabled,  but didn't specify the cert name with -d\n";
		print "Listing available certs, enter cert name when prompted\n";
		listCerts();
		print "\n\nEnter name: ";
		chomp($cert_name = <STDIN>);
	} else {
		$cert_name = "Server-Cert";
	}
		
	
	# connect to localhost
	my $ldap = Net::LDAP->new( 'localhost' ) or die "$@";

	# add a prompt in here for the directory manager's password,  hard coded for now
	
	# bind up
	my $mesg = $ldap->bind( 'cn=Directory Manager', password => 'abc123ABC' );
	# create 2 entry objects,  one for cn=config,  the other for cn=encryption,cn=config
	$mesg = $ldap->search(  filter=>"(cn=encryption)",
				base=> "cn=encryption,cn=config" );
	my $cn_encryption = $mesg->entry;
	$mesg = $ldap->search(  filter=>"(objectclass=nsslapdConfig)",
				base=> "cn=config" );
	my $cn_config = $mesg->entry;
	
	
	# enable the various settings in cn=encryption,cn=config
	if ($cn_encryption->get_value ( 'nsSSL3' )) {
		print "Found existing attribute nsSSL3,  turning it on\n";
		$mesg = $ldap->modify( "cn=encryption,cn=config", replace => {  'nsSSL3' => 'on' });
	} else {
		#ldapadd but shouldnt need to 
	}
	if ($cn_encryption->get_value ( 'nsSSLClientAuth' )) {
		print "Found existing attribute nsSSLClientAuth, setting it to allowed\n";
		$mesg = $ldap->modify( "cn=encryption,cn=config", replace => { 'nsSSLClientAuth' => 'allowed' }  );
	} else {
		# ldapadd but shouldnt need to
	}
	
	if ($cn_encryption->get_value ( 'nsSSL3Ciphers' )) {
		print "nsSSLCiphers already configured,  skipping\n";
		
	} else {
		print "Adding nsSSLCiphers\n";
		$mesg = $ldap->modify( "cn=encryption,cn=config", add => { 'nsSSL3Ciphers' => '-rsa_null_md5,+rsa_rc4_128_md5,+rsa_rc4_40_md5,+rsa_rc2_40_md5,+rsa_des_sha,+rsa_fips_des_sha,+rsa_3des_sha,+rsa_fips_3des_sha,+fortezza,+fortezza_rc4_128_sha,+fortezza_null,+tls_rsa_export1024_with_rc4_56_sha,+tls_rsa_export1024_with_des_cbc_sha,-rc4,-rc4export,-rc2,-rc2export,-des,-desede3' } );
	}
	# create the new RSA item
	$mesg = $ldap->add ( "cn=RSA,cn=encryption,cn=config", attrs => [ 	'objectClass' => [ 'top' , 'nsEncryptionModule' ],
										'cn' => 'RSA' ,
										'nsSSLPersonalitySSL' => "$cert_name",
										'nsSSLToken' => 'internal (software)',
										'nsSSLActivation' => 'on' ]);
	
	
	#$cn_config->dump;
	# check to see if nsslapd-security exists,  if it does modify it,  if not create it
	if (defined($cn_config->get_value ('nsslapd-security'))) {
		print "Turning on nsslapd-descurity\n";
		$mesg = $ldap->modify( "cn=config", replace => { 'nsslapd-security' => 'on'});
	} else {
		#ldapadd but shouldnt need to.
	}
	# check to see if nsslapd-ssl-check-hostname exists,  if it does modify it,  if not create it
	if (defined($cn_config->get_value ('nsslapd-ssl-check-hostname'))) {
		print "Turning off nsslapd-ssl-check-hostname\n";
		$mesg = $ldap->modify( "cn=config", replace => { 'nsslapd-ssl-check-hostname' => 'off' });
	}

	# create pin file while we're at it.
	open PIN, ">$config{SEC_DIR}/pin.txt";
	open PWD, "$config{SEC_DIR}/pwdfile";
	while (<PWD>) {
		chomp;
		print PIN "Internal (Software) Token:$_";
	}
	close PIN;
	close PWD;

	my @stat = stat($config{SEC_DIR});
	chown $stat[4], $stat[5], "$config{SEC_DIR}/pin.txt";
	chmod 0600, "$config{SEC_DIR}/pin.txt";
}


sub getserial {
	
		# check if the serial file exists,  if not create one
		if ( ! -r $config{SERIAL} ) {
			print "No serial file found at $config{SERIAL},  creating one now\n";
			open SERIAL, ">$config{SERIAL}";
			print SERIAL "0:HOSTNAME:CERT TYPE\n";
			close SERIAL;
		}
		# suck in the serial file data
		open SERIAL, $config{SERIAL};
		while (<SERIAL>) {
			my @tmp = split(/:/);
			#print "@tmp\n";
			push @key_serial, $tmp[0];
			push @key_hostname, $tmp[1];
			push @key_description, $tmp[2];
		}
		close SERIAL;
}

sub update_serial {
	# update the serials file.
	open SERIAL, ">$config{SERIAL}";
	my $total = scalar(@key_serial);
	my $loop = 0;
	while ($loop < $total) {
		print SERIAL "$key_serial[$loop]:$key_hostname[$loop]:$key_description[$loop]";
		$loop+=1;
	}

	close SERIAL;
	
}


sub load_config
{

	# check which config file to use
	if (defined($opt{a})) {
		# check to make sure the passed file exists
		if ( -r $opt{a} ) {
			open CONFIG, $opt{a};
		} else {
			print "config file $opt{a} doesn't exist! quitting\n";
			exit 1;
		}
	} else {
		# open the default file
        	open CONFIG, $conf;
	}

        while (<CONFIG>) {
                chomp;                  # no newline
                s/#.*//;                # no comments
                s/^\s+//;               # no leading white
                s/\s+$//;               # no trailing white
                next unless length;     # anything left?
                my ($var, $value) = split(/\s*=\s*/, $_, 2);
                # strip quotes off of value
                $value =~ s/"//;
                $value =~ s/"$//;
                $config{$var} = $value;

        }
        close CONFIG;


}
