#!/usr/bin/perl -w
# fdstool.pl
# commandline util for fds to setup/manage rep agreements
# 
#   Ryan Braun	-  Ryan.Braun@ec.gc.ca
# 
# there are currently 4 switches
# -z enables TLS encryption across ALL connections, no ssl currently (make sure ldapsearch -ZZZ works and this should)
# -c /path/to/config.file specify an other then default config file (for example to manage 2 different BASE_DN's )
# -p will prompt you for the root_dn password
# -d will dump all releven replication data to the screen then exit.

# This script requires a config file.  
# Look at save_config sub for config file contents.
# 

#	09/10/2008 - made some fixes so the script would work with fds 1.1+
#	09/10/2008 - changed app name to repmanager.pl and created fdstools package

#  TODO list
#	1. implement replication agreement removal
#	2. some sort of complete server removal routine that will blow away reps/referrals
#		across all servers if you are perm removing a server
#	3. add/remove servers right from the menu rather then editing the config file
#	4. proper perl doc config
#	5. add Single Master option for creating replication objects


# requires libldap-perl package in debian


# based on concepts/code from

# mmr.pl - configure multimaster replication between two fedora-ds servers
#
# Mike Jackson <mj@sci.fi> 19.11.2005
# Federico Roman added the --port option. 10-08-2007
#
# Professional LDAP consulting for large and small projects
#
#       http://www.netauth.com
#
# GPLv2 License
#


use strict;
use warnings;
use Net::LDAP qw(LDAP_ALREADY_EXISTS LDAP_TYPE_OR_VALUE_EXISTS);
use Switch;
use Getopt::Long;
use vars qw($opt_z $opt_p $opt_c $opt_d);

&GetOptions('z','p','c=s', 'd') || exit 1;

# default config file location,  edit me
my $fds_tool_config = "/etc/fdstools/repman.conf";

#  option to point to a diff config file 
if ($opt_c) {
	# TODO -r the file to make sure it exists
	$fds_tool_config = $opt_c;
} 

my %config;
my @sup_server;
my @sup_server_ports;
my @con_server;
my @con_server_ports;
my $answer;


# TODO check to make sure config file exists else puke inside of load_config sub
# TODO add check to make sure that root is running this proggy!!!

load_config();

# puke if any of the required attributes aren't set.  BASE_DN, REP_DN, REPMAN_PW
if ( ($config{BASE_DN} eq "") || ($config{REP_DN} eq "") || ($config{REPMAN_PW} eq "") || ($config{ROOT_DN} eq "") ) {
	print "You need to fix you're config file at $fds_tool_config\n";
	print "One of BASE_DN, REP_DN, REPMAN_PW or ROOT_DN are not getting imported properly\n";
	print "Quitting\n";
	exit;
}


my $passwd = find_root_pw();
if ( ! $passwd ) {
	print "couldn't locate password in either $config{PAM_SECRET} or $config{NSS_SECRET}\n";
	print "try running with -p to specify a password\n";
	exit;
}
	


my $TLS;
# enable tls support if specified by switch or if enabled in the config file
if ($opt_z) {
	$TLS=1;
} else {
	$TLS = $config{TLS};
}



if ($opt_d) {
	# dump all info and quit
	dump_all();
}


while (1) {

	print "*****************************************************\n";
	print "******** FDS replication setup tool  ****************\n";
	print "*****************************************************\n\n\n";

	if ( ! $TLS ) {
		print "SESSION NOT ENCRYPTED!!!!!\n\n";
	}
	
	print "\t(L)ist current server status\n";
	print "\t(S)how all agreements/referrals\n";
	print "\t(I)initialize a consumer\n";
	print "\t(N)ew replication Agreement\n";
	print "\t(D)elete replication Agreement\n";
	print "\t(M)odify referrals\n";
	print "\t(Q)uit\n\n";
	print "\t\t:";
	
	
	chomp($answer = lc(<STDIN>));

	switch ($answer) {
		case 'l' { &server_status; }
		case 'i' { &init_consumer; }
		case 's' { &show_reps; }
		case 'n' { &add_replication; }
		case 'm' { &modify_referral;}
		case 'd' { &remove_rep_agreement; }
		case 'q' { print "quitting\n";exit; }
		
		
		}

}

#################################################
######## initialization subs
#################################################


sub init_consumer
{
	# initialize any consumer from any supplier
	# TODO perform ldap search to verify the proper rep agreement exists.
	# maybe ask which consumer you want to init first,  then search for any 
	# rep agreements on suppliers to that consumer, then provide those as options

	my ($ldap, $msg, $supplier,  $supplier_port, $consumer);
	
	print "\n\n\t\tPlease select the number of the supplier server:\n";
	print "\t** This is the server that will be sending the updates **\n\n";

	my $count = 0;
	foreach (@sup_server) {
		print "\t\t$count. $_\n";
		$count = $count + 1;
	}
	print "\n\t--->";
	chomp($answer = <STDIN>);
	
	
	# set variables
	$supplier = $sup_server[$answer];
	$supplier_port = $sup_server_ports[$answer];

	# create a big list of all conumsers (which is every server)
	my @all_servers = (@sup_server, @con_server);
	$count = 0;
	print "\t\tSelect the consumer from the list below:\n\n";
	foreach (@all_servers)	{

		print "\t\t$count: $_\n";
		$count = $count + 1;
	}
	print "\n\t--->";
	chomp($answer = <STDIN>);

	$consumer = $all_servers[$answer];
	
	if ($supplier eq $consumer) {
		# you're trying to have a server init itself!!
		print "\tYou trying to initialize $consumer off of $supplier!!\n";
		print "\tHow well do you think that will work?\n";
		print "\tCancelling operation\n";
		print "\t\tPress Enter to Continue\n\n";
		chomp($answer = <STDIN>);
		return;
	}

	init_target($supplier, $supplier_port, $consumer, $passwd);

}

sub init_target
{
	# this will set the initialize attribute in the rep agreement on the supplier
	# the supplier should then begin initializing the target
	my ($supplier, $supplier_port, $consumer, $bind_pw) = @_;
	my ($ldap, $msg);

	if ( ($ldap = conn_bind($supplier,$supplier_port,$passwd)) eq 0 ) {
		print "\t*********** bind/connect failed to $supplier on port $supplier_port ***************\n";
		return 0;
	}
	my $dn = "cn=\"Replication to $consumer\",cn=replica,cn=\"$config{BASE_DN}\",cn=mapping tree,cn=config";

	print "\ninitializing replication $supplier -> $consumer\n";
	
	$msg = $ldap->modify($dn, add => { 'nsDS5BeginReplicaRefresh' => 'start' });
	$msg->code && die "failed to add initialization attribute: " . $msg->error;

	disconnect($ldap);
}



#################################################
######### Replication Agreement subs
#################################################

sub add_replication
{
	# nsDS5ReplicaType
	# 0 - reserved for internal use
	# 1 - Dedicated Supplier
	# 2 - Dedicated Consumer	<---- consumer only
	# 3 - Supplier/Consumer		<---- MMR server

	my ($rep_type, $ssl, $supplier, $supplier_port, $consumer, $consumer_port, $init_consumer);
	my $count = 0;

	print "\tSetting up what type of replication?\n\n";
	print "\t(1) Supplier - Supplier (MMR)\n";
	print "\t(2) Supplier - Consumer (Master to Read Only)\n";
	print "\t--->";

	chomp($rep_type = <STDIN>);
	# TODO verify input

	# get source server
	print "\n\n\tSelect Supplier server:\n\n";

	foreach my $srvr (@sup_server) {
		print "\t($count) $srvr\n";
		$count = $count + 1;
	}
	print "\n\t--->";
	chomp($answer = <STDIN>);
	# set supplier server value from the server list
	$supplier = $sup_server[$answer];
	$supplier_port = $sup_server_ports[$answer];

	# get server for MMR replication
	$count = 0;
	if ( $rep_type eq 1) {
		
		my $selection;
		print "\n\tSelect target Supplier server for MMR setup\n";
		# TODO make sure the same server is not selected
		foreach my $srvr (@sup_server) {
			print "\t($count) $srvr\n";
			$count = $count + 1;
		}

		print "\n\t--->";
		chomp($selection = <STDIN>);
		$consumer = $sup_server[$selection];

		print "\n\tEnter port [$sup_server_ports[$selection]]\n";
		print "\t--->";
		chomp($answer = <STDIN> );
		if ($answer) {
			$consumer_port = $answer;
		} else {
			$consumer_port = $sup_server_ports[$selection];
		}

	}

	# get server for Master to read only
	$count = 0;
	if ( $rep_type eq 2) {
		
		my $selection;
		print "\n\tSelect Consumer server\n";
		
		foreach my $srvr (@con_server) {
			print "\t($count) $srvr\n";
			$count = $count + 1;
		}
		
		print "\n\t--->";
		# TODO check to make sure something was entered,  stuff will break if not.
		chomp($selection = <STDIN>);
		$consumer = $con_server[$selection];

		print "\n\tEnter port [$con_server_ports[$selection]]\n";
		print "\t--->";
		chomp($answer = <STDIN> );
		if ($answer) {
			$consumer_port = $answer;
		} else {
			$consumer_port = $con_server_ports[$selection];
		}

	}



	print "\n\n\tUse SSL for the replication? [no]\n";
	print "\t--->";
	chomp($ssl = <STDIN>);
	$ssl = lc($ssl);
	
	# check to make sure you're not trying to connect SSL to 389
	if ( ($ssl eq "yes") && ($consumer_port eq 389) ) {
		print "\tDo you really want to connect to $consumer on port $consumer_port\n";
		print "\tSSL connections over 389 generally break things\n";
		print "\tChange target port? [no]";
		chomp($answer = <STDIN> );
		if ($answer) {
			print "\tEnter new target port for SSL connections to $consumer [636]\n";
			print "\t--->";
			chomp($answer = <STDIN> );
			if ($answer) {
				$consumer_port = $answer;
			} else {
				$consumer_port = 636;
			}
		}
	}

	print "\n\n\tDo you want to initialize the consumer $consumer [no]\n";
	print "\t-->";
	chomp($init_consumer = <STDIN>);
	$init_consumer = lc($init_consumer);
	
	# verify you want to initialize the target if setting up MMR
	# in case you blow away
	if (($init_consumer eq "yes") && ($rep_type eq 1) ) {
		print "\t\ttDo you really want to init the MMR consumer $consumer (all data on that box will be destroyed? [yes]";
		#finish me
	}

	print "\n\n\t\t\t************** SUMMARY ***************\n\n";
	print "\t\tReplication Type: ";
	if ($rep_type eq 1) {
		print "\tMulti-Master Replication\n";
	} elsif ( $rep_type eq 2 ) {
		print "\tMaster-Consumer Replication\n";
	}
	print "\t\tSupplier server:\t\t$supplier\n";
	print "\t\tSupplier port:\t\t$supplier_port\n";
	print "\t\tConsumer server:\t\t$consumer\n";
	print "\t\tConsumer port:\t\t$consumer_port\n";
	if ($ssl) {
		print "\t\tUsing SSL?:\t\tYES\n";
	} else {
		print "\t\tUsing SSL?\t\tNO\n";
	}
	if ($init_consumer eq "yes") {
		print "\t\tInitialize consumer:\tYES\n";
	} else {
		print "\t\tInitialize consumer:\tNO\n";
	}

	print "\n\n\tAre you sure you want to create the replication between the 2 servers? [no]\n";
	print "\t--->";
	chomp($answer = <STDIN>);
	
	if (($answer eq "") || ($answer eq "no")) {
		# user entered no, bail
		exit;
	}

	# setup MMR between 2 boxes.
	if ( $rep_type eq 1 ) {
		
		# create user on supplier
		add_rep_user($supplier,$supplier_port,$passwd);
		# create user on consumer
		add_rep_user($consumer,$consumer_port,$passwd);
		# create changlog object on supplier
		add_changelog($supplier,$supplier_port,$passwd);
		# create changlog object on consumer
		add_changelog($consumer,$consumer_port,$passwd);
		# create replica object on supplier
		# my ($source, $source_port, $target, $target_port, $bind_pw,$rep_type)
		add_rep_object($supplier,$supplier_port, $consumer, $consumer_port, $passwd,1);
		# create replica object on consumer
		add_rep_object($consumer,$consumer_port, $supplier, $supplier_port, $passwd,1);
		# create rep agreement on supplier
		add_rep_agreement($supplier, $supplier_port, $consumer, $consumer_port, $ssl, $passwd);
		# create rep agreement on consumer
		add_rep_agreement($consumer, $consumer_port, $supplier, $supplier_port, $ssl, $passwd);
		# init consumer if required	
		# my ($source, $source_port, $target, $bind_pw) = @_;
		if ($init_consumer eq "yes") {
			init_target($supplier, $supplier_port, $consumer, $passwd);
		}

	}


	# setup Supplier -> read only consumer 
	if ( $rep_type eq 2 ) {

		# create user on source
		add_rep_user($supplier, $supplier_port, $passwd);
		# create user on target
		add_rep_user($consumer, $consumer_port, $passwd);
		# create changlog object on source
		add_changelog($supplier, $supplier_port, $passwd);
		# create replica object on source
		# my ($source, $source_port, $target, $target_port, $bind_pw,$rep_type)
		add_rep_object($supplier, $supplier_port, $consumer, $consumer_port, $passwd,1);
		# create replica object on target
		add_rep_object($consumer, $consumer_port, $supplier, $supplier_port, $passwd,2);
		# create rep agreement on source
		# my ($source, $source_port, $target, $target_port,$with_ssl, $bind_pw)
		add_rep_agreement($supplier, $supplier_port, $consumer, $consumer_port, $ssl, $passwd);
		#  Consumer -> modify dn: cn=$config{BASE_DN},cn=mapping tree,cn=config attribute nsslapd-state to "referral on update"
		# add referral to the master on target
		add_replica_referral($consumer, $consumer_port, $supplier, $supplier_port, $passwd);
		# init target if required	
		# my ($source, $source_port, $target, $bind_pw) = @_;
		if ($init_consumer eq "yes") {
			init_target($supplier, $supplier_port, $consumer, $passwd);
		}

	}

}

sub show_reps
{
	# display menu and show appropriate rep agreements.
	
	my $count;
	
	print "\n\t\tShow reps and referrals for which server?\n\n";
	print "\t(1) All Servers\n";
	print "\t(2) All Supplier servers\n";
	print "\t(3) All Consumer Servers\n";
	print "\r\r--->";
	chomp($answer = <STDIN>);
	print "\n\n";
	switch ($answer) {
		case '1' {
				$count = 0;
				print "\n\n\t\t\tSupplier Servers Replication Agreements:\n";
				foreach my $server (@sup_server) {
					print "\nhost $server\n";
					print "    |\ttarget\t\t\tPort\tSSL\tStatus\n";
					print "    |\t-----------------------------------------------------\n";
					get_reps($server,$sup_server_ports[$count],$passwd);
					$count = $count + 1;
					print "\n";
				}

# 				$count = 0;
# 				print "\n\n\t\t\tSupplier Servers Referrals:\n";
# 				foreach my $server (@sup_server) {
# 					print "\nhost $server\n";
# 					print "    |\ttarget\t\t\tPort\tSSL\tStatus\n";
# 					print "    |\t-----------------------------------------------------\n";
# 					get_referrals($server,$sup_server_ports[$count],$passwd);
# 					$count = $count + 1;
# 					print "\n";
# 				}

				$count = 0;

				print "\n\n\t\t\tConsumer Servers Referrals:\n";
				foreach my $server (@con_server) {
					print "\nhost $server\n";
					print "    |\ttarget\n";
					print "    |\t-----------------------------------------------------\n";
					get_referrals($server,$con_server_ports[$count],$passwd);
					$count = $count + 1;
					print "\n";
				}
			}
		case '2'
			{
				$count = 0;
				print "\n\n\t\t\tSupplier Servers:\n";
				foreach my $server (@sup_server) {
					print "\nhost $server\n";
					print "    |\ttarget\t\t\tPort\tSSL\tStatus\n";
					print "    |\t-----------------------------------------------------\n";
					get_reps($server,$sup_server_ports[$count],$passwd);
					$count = $count + 1;
					print "\n";
				}

# 				$count = 0;
# 				print "\n\n\t\t\tSupplier Servers Referrals:\n";
# 				foreach my $server (@sup_server) {
# 					print "\nhost $server\n";
# 					print "    |\ttarget\t\t\tPort\tSSL\tStatus\n";
# 					print "    |\t-----------------------------------------------------\n";
# 					get_referrals($server,$sup_server_ports[$count],$passwd);
# 					$count = $count + 1;
# 					print "\n";
# 				}

			}
		case '3'
			{
				$count = 0;
				print "\n\n\t\t\tConsumer Server Referrals:\n";
				foreach my $server (@con_server) {
					print "\nhost $server\n";
					print "    |\ttarget\n";
					print "    |\t-----------------------------------------------------\n";
					get_referrals($server,$con_server_ports[$count],$passwd);
					$count = $count + 1;
					print "\n";
				}
			}
		}


	print "\n\nPress Enter to continue";
	chomp($answer = <STDIN>);


}


sub add_rep_agreement
{
	my ($supplier, $supplier_port, $consumer, $consumer_port,$with_ssl, $bind_pw) = @_;
	my ($ldap, $msg);

	#$ldap = conn_bind($source, $source_port, $bind_pw);
	if ( ($ldap = conn_bind($supplier,$supplier_port,$passwd)) eq 0 ) {
		print "\t*********** bind/connect failed to $supplier on port $supplier_port ***************\n";
		return 0;
	}
	print "adding to Replication Agreement $supplier -> $consumer\n";
	$msg = $ldap->add(
		"cn=Replication to $consumer,cn=replica,cn=\"$config{BASE_DN}\",cn=mapping tree,cn=config",
			attr => [
			objectclass                  => [qw (top nsDS5ReplicationAgreement)],
			cn                           => "Replication to $consumer",
			Description		     => "Replication to $consumer",
			nsDS5ReplicaHost             => $consumer,
			nsDS5ReplicaRoot             => "$config{BASE_DN}",
			nsDS5ReplicaPort             => $consumer_port,
			#nsDS5ReplicaTransportInfo    => "SSL",
			nsDS5ReplicaBindDN           => $config{REP_DN},
			nsDS5ReplicaBindMethod       => "simple",
			nsDS5ReplicaCredentials      => $config{REPMAN_PW},
			nsDS5ReplicaUpdateSchedule   => "0000-2359 0123456",
			nsDS5ReplicaTimeOut          => 120,
		]
	);

	if ($msg->code == LDAP_ALREADY_EXISTS) {
		print "\t -> already exists\n\n";
		# disconnect here and return if exists
		disconnect($ldap);
		return;

	} else {
		$msg->code && die "failed to add replication agreement entry: " . $msg->error;
        }

	if ($with_ssl eq 1) {
		# TODO ldapmodify and add #nsDS5ReplicaTransportInfo    => "SSL",
		$msg  = $ldap->modify(
			"cn=Replication to $consumer,cn=replica,cn=\"$config{BASE_DN}\",cn=mapping tree,cn=config",	
				add	=> { 'nsDS5ReplicaTransportInfo' => "SSL" } );
	}
	
	disconnect($ldap);
	
}

sub remove_rep_agreement
{
	my ($server, $port);
	my ($ldap, $msg);
	
	print "\t\tFrom which supplier server do you want to remove the rep agreement?\n\n";
	my $count = 0;
	foreach (@sup_server) {
		print "\t$count. $_\n";
		$count = $count + 1;
	}
	chomp($answer = <STDIN>);

	$server = $sup_server[$answer];
	$port = $sup_server_ports[$answer];

	# get agreements off of server
	if ( ($ldap = conn_bind($server,$port,$passwd)) eq 0 ) {
		print "\t*********** bind/connect failed to $server on port $port ***************\n";
		return 0;
	}
	# grab any rep objects from the server
	$msg = $ldap->search(
                         base    => "cn=config",
                         filter  => "(objectClass=nsDS5ReplicationAgreement)",
                        );
    	$msg->code && die $msg->error;
	
	$count = 0;
	
	if ($msg->count == 0) {
		print "No rep Agrrements found on $server\n";
		print "\n\nPress Enter to continue";
		chomp($answer = <STDIN>);
		return;
	}
	
	print "\t\tSelect rep agreement to remove:\n\n";
	
	if ($msg->count) {
		for ($msg->entries) {
			print "$count. " . $_->get_value("cn") . "\n";
			$count = $count + 1;
		}
	}
	print "\n\n--->";
	chomp($answer = <STDIN>);

 	print "dn selected was  cn=" . $msg->entry($answer)->get_value("cn") . ",cn=replica,cn=\"$config{BASE_DN}\",cn=mapping tree,cn=config\n\n";
	
	my $dn = "cn=" . $msg->entry($answer)->get_value("cn") . ",cn=replica,cn=\"$config{BASE_DN}\",cn=mapping tree,cn=config";
	
	#$msg = $ldap->delete( $dn );


}

sub get_reps
{
	
	my ($server, $port, $bind_pw) = @_;
	my ($msg, $ldap);

	if ( ($ldap = conn_bind($server,$port,$passwd)) eq 0 ) {
		print "\t*********** bind/connect failed to $server on port $port ***************\n";
		return 0;
	}
	# grab any rep objects from the server
	$msg = $ldap->search(
                         base    => "cn=config",
                         filter  => "(objectClass=nsDS5ReplicationAgreement)",
                        );
    	$msg->code && die $msg->error;

	# if any were returned,  cycle through each object with a for loop
	if ($msg->count) {
		for ($msg->entries) 
	{
		# kind of ugly below, but basically one big string concant while pulling ldap values
		print "    |-->" . $_->get_value("nsDS5ReplicaHost");
		print "\t" . $_->get_value("nsDS5ReplicaPort");
		# show SSL option
		if ($_->get_value("nsDS5ReplicaTransportInfo")) {
			print "\tYES\t";
		} else {
			print "\tNO\t";
		}
		# output replication status
		if ($_->get_value("nsds5replicaLastUpdateStatus")) {
			print $_->get_value("nsds5replicaLastUpdateStatus") . "\n";
		} else {
			print "no status object in ldap!!\n";
		}
		
	}
	}
	else {
		print "\t -> none found\n";
	}

	disconnect($ldap);
}


#################################################
########  referral subs
#################################################

sub add_replica_referral
{
	# adds referral to the multivalued attribute  nsDS5ReplicaReferral in dn: cn=replica,cn="$config{BASE_DN}",cn=mapping tree,cn=config
	# should only need to add this to a read only consumer!!
	# the first entry will be created automatically by the add_rep_object,  this will add more referrals

	# TODO add check to make sure the replica object exists,  otherwise it will fail silently and throw and err=32 no such object 
	# in the servers log.
	my ($server, $server_port, $referral, $referral_port, $bind_pw) = @_;
	my ($ldap, $msg);

	#$ldap = conn_bind($srvr, $srvr_port, $bind_pw);
	if ( ($ldap = conn_bind($server,$server_port,$passwd)) eq 0 ) {
		print "\t*********** bind/connect failed to $server on port $server_port ***************\n";
		return 0;
	}
	print "Adding referral on $server back to $referral\n";

	# dn: cn=replica,cn="$config{BASE_DN}",cn=mapping tree,cn=config
	# nsDS5ReplicaReferral: ldap://xxxx:389/dc=xxx,dc=ec,dc=gc,dc=ca
	$msg = $ldap->modify("cn=replica,cn=\"$config{BASE_DN}\",cn=mapping tree,cn=config",
			add	=> { 'nsDS5ReplicaReferral' => "ldap://$referral:$referral_port/$config{BASE_DN}" });
	
	if ($msg->code == LDAP_ALREADY_EXISTS)
	{
		print "\t -> already exists\n\n";
	}


	disconnect($ldap);
}

sub remove_replica_referral
{
	# removes referral attribute items 
	my ($server, $port, $referral) = @_;
	my ($ldap, $msg);
	
	if ( ($ldap = conn_bind($server, $port, $passwd)) eq 0 ) {
		# connect has failed,  return
		print "\n\nPress Enter to continue\n";
		chomp($answer = <STDIN>);
		return;
 	} 
	
	my $dn = "cn=replica,cn=\"$config{BASE_DN}\",cn=mapping tree,cn=config";
	$ldap->modify($dn,
			delete => { 'nsDS5ReplicaReferral' => $referral } );
	
	disconnect($ldap);
	
}


# sub add_base_referral
# {
# 	## SHOULDNT NEED THIS,  AFTER ADDING TO THE REPLICA OBJECT THIS GETS UPDATED BY ITSELF
# 	
# 	# adds referral to the multivalued attribute nsslapd-referral in dn: cn="$config{BASE_DN}",cn=mapping tree,cn=config
# 	my ($srvr, $srvr_port, $referral, $referral_port, $bind_pw) = @_;
# 	my ($ldap, $msg);
# 
# 	$ldap = conn_bind($srvr, $srvr_port, $bind_pw);
# 
# 	# dn: cn="$config{BASE_DN}",cn=mapping tree,cn=config
# 	# nsslapd-referral: ldap://xxx.xxx.ec.gc.ca:389/dc%3Dxxx%2Cdc%3Dec%2Cdc%3Dgc%2Cdc%3Dca
# 	$msg = $ldap->modify("cn=\"$config{BASE_DN}\",cn=mapping tree,cn=config",
# 			add	=> { 'nsslapd-referral' => "ldap://$referral:$referral_port/$config{BASE_DN}" });
# 	
# 	if ($msg->code == LDAP_ALREADY_EXISTS)
# 	{
# 		print "\t -> already exists\n\n";
# 	}
# 
# 	print "msg is " . $msg->code . "\n";
# 
# 
# 	disconnect($ldap);
# 
# }


sub get_referrals
{
	my ($server, $port, $bind_pw, $return_list) = @_;
	my ($msg, $ldap);
	
	# get connected or return and output error
	
	if ( ($ldap = conn_bind($server,$port,$passwd)) eq 0 ) {
		print "\t*********** bind/connect failed to $server on port $port ***************\n";
		return 0;
	}

	# get refs from the replica object
	$msg = $ldap->search(filter=>"objectClass=nsDS5Replica", 
			 base=>"cn=config",
			 attrs=> ['nsDS5ReplicaReferral']);

	# TODO if a server has been added to the config file,  but no rep or referral entries exist
	# the following will cause the script to puke.

	# there should only be one replication object so assume its the 0 entry in the entry list
	my @refs = $msg->entry(0)->get_value("nsDS5ReplicaReferral");
	
	if ($return_list) {
		# return the list to the caller and exit before displaying
		disconnect($ldap);
		return @refs;
	}

	# otherwise display referrals
	foreach (@refs) {
		print "    |--> $_ \n";
	}
	
	disconnect($ldap);
		
}

sub modify_referral
{

	my ($server, $port, @servers, @ports);
	my ($msg, $ldap);

	print "\n\n\tModify referrals on what type of server:\n\n";
	print "\t\t1. Supplier\n";
	print "\t\t2. Consumer\n";	
	print "\n\t\t--->";
	chomp($answer = <STDIN>);

	# reassign lists to whatever type of server we want
	if ( $answer eq 1 ) {
		@servers = @sup_server;
		@ports = @sup_server_ports;
	} else {
		@servers = @con_server;
		@ports = @con_server_ports;
	}
	
	my $count = 0;
	print "\n\n\tEnter the server number:\n\n";
	foreach (@servers) {
		print "\t\t$count. $_:$ports[$count]\n";
		$count = $count + 1;
	}
	
	print "\n\t---> ";
	chomp($answer = <STDIN>);

	# set variables
	$server = $servers[$answer];
	$port = $ports[$answer];


	print "\n\n\t\tReferrals on $server\n\n";
	get_referrals($server, $port, $passwd);

	print "\n\n\t\tDo you want to :\n\n";
	print "\t1. Add referral\n";
	print "\t2. Remove referral\n";
	print "\n\t---> ";
	chomp($answer = <STDIN>);
	
	if ($answer eq 1) {
		# Add new referral
		print "\n\n\t\tAdd referral to which supplier (enter number)?\n\n";
		$count =0;
		foreach (@sup_server) {
			print "\t$count. $_:$sup_server_ports[$count]\n";
			$count = $count +1;
		}
		print "\t--->";
		chomp($answer = <STDIN>);
		add_replica_referral($server, $port, $sup_server[$answer], $sup_server_ports[$answer], $passwd);

	} elsif ( $answer eq 2 ) {

		# remove a referral
		print "\n\n\t\tRemove which referral (enter number)?\n\n";
		$count = 0;
		my @refs = get_referrals($server, $port, $passwd, 1);
		# check to make sure @refs has elements?
		foreach (@refs) {
			print "\t$count. $_\n";
			$count = $count + 1;
		}
		print "\n\t--->";
		chomp($answer = <STDIN>);
		remove_replica_referral($server, $port, $refs[$answer]);


	}
	

}


################################################
########## Misc Rep object subs
################################################


sub add_rep_object
{
	
	# nsDS5ReplicaType
	# 0 - reserved for internal use
	# 1 - Dedicated Supplier
	# 2 - Dedicated Consumer	<---- consumer only
	# 3 - Supplier/Consumer		<---- MMR server
	
	# passed variable $rep_type is not the same as the above!!

	# TODO add SMR options.

	my ($server, $server_port, $referral, $referral_port, $bind_pw, $rep_type) = @_;
	my ($ldap, $msg);
	
	# server is the server the object is getting created on
	# referral is the server to send referrals to if setting up a consumer read only
	#$ldap = conn_bind($server, $server_port, $bind_pw);
	if ( ($ldap = conn_bind($server,$server_port,$passwd)) eq 0 ) {
		print "\t*********** bind/connect failed to $server on port $server_port ***************\n";
		return 0;
	}
	##############################
	# add replica object
	##############################

	# type 1 is MMR
	# type 2 is consumer read only

	print "adding to Replica object to $server -> cn=replica,cn=\"$config{BASE_DN}\",cn=mapping tree,cn=config\n";
	
	# add consumer type rep
	if ( $rep_type eq 2 ) 	{
		$msg = $ldap->add(
			"cn=replica,cn=\"$config{BASE_DN}\",cn=mapping tree,cn=config",
			attr => [
				objectclass                  => [qw (top nsDS5Replica extensibleObject)],
				cn                           => "replica",
				nsDS5ReplicaId               => 65535,
				nsDS5ReplicaReferral	     => "ldap://$referral:$referral_port/$config{BASE_DN}",
				nsDS5ReplicaRoot             => $config{BASE_DN},
				nsDS5Flags                   => 0,
				nsDS5ReplicaBindDN           => $config{REP_DN},
				nsds5ReplicaPurgeDelay       => 604800,
				nsds5ReplicaLegacyConsumer   => "off",
				nsDS5ReplicaType             => 2,
			]
		);
	}

	# add MMR type rep
	if ($rep_type eq 1) {
		print "Trying to create cn=replica,cn=\"$config{BASE_DN}\",cn=mapping tree,cn=config\n\n";
		$msg = $ldap->add(
			"cn=replica,cn=\"$config{BASE_DN}\",cn=mapping tree,cn=config",
			attr => [
				objectclass                  => [qw (top nsDS5Replica extensibleObject)],
				cn                           => "replica",
				nsDS5ReplicaId               => $config{REP_ID},
				nsDS5ReplicaRoot             => $config{BASE_DN},
				nsDS5Flags                   => 1,
				nsDS5ReplicaBindDN           => $config{REP_DN},
				nsds5ReplicaPurgeDelay       => 604800,
				nsds5ReplicaLegacyConsumer   => "off",
				nsDS5ReplicaType             => 3,
			]
		);

		
	}

	if ($msg->code == LDAP_ALREADY_EXISTS)
	{
		print "\t -> already exists\n\n";
		# disconnect and return here before incrementing the rep_id if the object already exists
		disconnect($ldap);
		return;
		
	} else {
		$msg->code && die "failed to add replica entry: " . $msg->error;
	}

	# if MMR reptype,  and we created an object,  we need to increment the rep_id in the config file
	if ( $rep_type eq 1 ) {
		
		$config{REP_ID} = $config{REP_ID} + 1;
		save_config();
		load_config();
	}
	
	disconnect($ldap);

}



sub add_changelog
{
	# adds the changelog entry if required
	my ($server, $port, $bind_pw) = @_;
	my ($ldap, $msg);

	#$ldap = conn_bind($srvr, $port, $bind_pw);
	if ( ($ldap = conn_bind($server,$port,$passwd)) eq 0 ) {
		print "\t*********** bind/connect failed to $server on port $port ***************\n";
		return 0;
	}
	##############################
	# find the instance-dir
	##############################
	$msg = $ldap->search (
                          base    => "cn=config",
                          scope   => "base",
                          filter  => "(objectClass=*)",
                         );

	#my $instance_dir = $msg->entry(0)->get_value("nsslapd-instancedir");
	my $instance_dir = $msg->entry(0)->get_value("nsslapd-ldifdir");
	# check to make sure we got something back
	if (!defined($instance_dir)) {
		print "Unable to determine the instancedir,  pretty big issue,  puking\n";
		exit 1;
	}	
	# get the actual instancedir by removing /ldif
	$instance_dir =~ s/\/ldif//;
	#print "Instance dir found was $instance_dir!!\n";

	

	# default to /opt/fedora-ds/slapd if not found (I like symlinking slapd-XXXX to just slapd anyways)
	#$instance_dir ||= "/opt/fedora-ds/slapd";
	
	
	##############################
	# add changelog
	##############################
	print "adding to changelog object to $server -> cn=changelog5,cn=config\n";
	$msg = $ldap->add(
		"cn=changelog5,cn=config",
		attr => [
		objectclass                  => [qw (top extensibleObject)],
		cn                           => "changelog5",
		"nsslapd-changelogdir"       => "$instance_dir/changelogdb",
		]
	);

	if ($msg->code == LDAP_ALREADY_EXISTS)
	{
		print "\t -> already exists\n\n";
	} else {
		$msg->code && die "failed to add changelog entry: " . $msg->error;
	}


	disconnect($ldap);

}

sub add_rep_user
{
	my ($server, $port, $bind_pw) = @_;
	my ($ldap, $msg);
	
#	$ldap = conn_bind($srvr, $port, $bind_pw);
	if ( ($ldap = conn_bind($server,$port,$passwd)) eq 0 ) {
		print "\t*********** bind/connect failed to $server on port $port ***************\n";
		return 0;
	}

	print "adding replication user to $server -> uid=RManager,cn=config\n";
	
	$msg = $ldap->add(
          "uid=RManager,cn=config",
          attr => [
	# needed to add organizationalPerson inetorgperson or ldap would throw err=65 when trying to add
            objectclass    => [qw (top person organizationalPerson inetorgperson)],
            cn             => "RManager",
            sn             => "RManager",
            userPassword   => $config{REPMAN_PW},
          ]
    	);
	if ($msg->code == LDAP_ALREADY_EXISTS)
	{
		print "\t -> already exists\n\n";
	} else {
        	$msg->code && die "failed to add RManager entry: " . $msg->error;
	}
	
	disconnect($ldap);

}



#################################################
########## Connection subs
#################################################

sub disconnect
{
	
	my ($ldap) = $_[0];
	$ldap->unbind;
	$ldap->disconnect;
}

sub conn_bind
{
	# sub to connect and bind to the passed server
	my ($server, $port, $bind_pw, $status) = @_;
	my ($msg,$ldap);
	
	$ldap = Net::LDAP->new($server, port => $port, timeout=>10);
	
	if ( ! $ldap )
	{
			
			print "\t$server\t$port\t\tFAILURE Could not connect to server\n";
			return 0;
	}
	

	# encrypt if specified on commandline or in configs
	if ( ($TLS) || ($config{TLS}) ) {
		$msg = $ldap->start_tls(verify => 'require',capath => $config{CA_DIR});	 
 	 		if ( $msg->is_error() ) {
 	 			print "Bailed on server $server return code was " . $msg->code() . "\n";
 	 			print "error message is " . $msg->error(). "\n";
 	 			print "error_name - " . $msg->error_name . "\n\n" . "error_text is " . $msg->error_text . "\n";
				print "\n\n\nTLS HAS FAILED, QUITTING AND EITHER FIX OR TURN OFF TLS!!!!\n\n\n";
				exit;
 	 		}
 	}

	# attempt to bind
	$msg = $ldap->bind ( $config{ROOT_DN},password =>$bind_pw ,version => 3 );
	#$mesg->code && die $mesg->error;

	if ($msg->code) {
		# print error message here because we have access to the $mesg object
		print "\t$server\t$port\t\tFAILURE " . $msg->code . " error text is " . $msg->error_name . "\n";
		
		return 0;
	}

	# if only checking status of the servers,  disconnect,  and return 1,  else return ldap object
	if ($status) {
		disconnect($ldap);
		return 1;
	} else {
		return $ldap;
	}
}

sub server_status {
	
	my $count;

		# cycle through suppliers to check root bind dn status
		$count = 0;
		print "\t\t\tChecking supplier servers\n";
		print "\tservername\t\tport\t\tstatus\n";
		print "-------------------------------------------------------\n";
		
		foreach my $srvr (@sup_server) {
			#print "about to check $srvr on port $sup_server_ports[$count]\n";
			if (conn_bind($srvr,$sup_server_ports[$count],$passwd,1)) {
				print "\t$srvr\t$sup_server_ports[$count]\t\tSUCCESS\n";
			} 
			$count = $count +1;
		}
		
		# cycle through consumers to check root bind dn status
		$count = 0;
		
		print "\n\n\t\t\tChecking consumer servers\n";
		print "\tservername\t\tport\t\tstatus\n";
		print "-------------------------------------------------------\n";
		
		foreach my $srvr (@con_server) {
			#print "about to check $srvr on port $sup_server_ports[$count]\n";
			if (conn_bind($srvr,$con_server_ports[$count],$passwd,1)) {
				print "\t$srvr\t$con_server_ports[$count]\t\tSUCCESS\n";
			} 
			$count = $count +1;
		}

		print "\n\nPress Enter to continue";
		chomp($answer = <STDIN>);



}


sub find_root_pw
{
	# sub to figure out where the root dn password is located.
	# this whole program assumes the same root dn password accross all servers!!
	# the file checks are configurable in the .conf file
	
	if ($opt_p) {
		print "Root DN password:";
		chomp($answer = <STDIN>);
		return $answer;
	}

	# check if the libpam.secret file exists and is not zero sized
	if ( ( -e $config{PAM_SECRET}) && (! -z $config{PAM_SECRET}) ) {
		open PW, $config{PAM_SECRET};
		chomp($answer =(<PW>));
		close PW;
		return $answer;
	}
	# check if the libnss-ldap.secret  file exists and is not zero sized
	if ( ( -e $config{NSS_SECRET}) && (! -z $config{NSS_SECRET}) ) {
		open PW, $config{NSS_SECRET};
		chomp($answer =(<PW>));
		close PW;
		return $answer;
	}

}
	
#################################################
########## Config file subs
#################################################

sub save_config
{
	open CONFIG, ">$fds_tool_config";

#	doing this dynamically means random order in config file. oh well
# 	foreach(keys %config) {
# 		print "$_=\"$config{$_}\"\n";
# 	}
	print CONFIG "########################\n";
	print CONFIG "#### Config file for fdstool.pl\n";
	print CONFIG "#### any changes to the comments will be blown away!!\n";
	print CONFIG "#### but you can change any of the values and they will be saved\n";
	print CONFIG "########################\n";
	print CONFIG "# list all suppliers here in format server.com[:389] delimmited with a ;\n";
	print CONFIG "SUPPLIERS=\"$config{SUPPLIERS}\"\n";
	print CONFIG "# list all consumers here in format server.com[:389] delimmited with a ;\n";
	print CONFIG "CONSUMERS=\"$config{CONSUMERS}\"\n";
	print CONFIG "# Root bind dn (or whatever bind dn you want to use)\n";
	print CONFIG "ROOT_DN=\"$config{ROOT_DN}\"\n";
	print CONFIG "# dn of you're replication user\n";
	print CONFIG "REP_DN=\"$config{REP_DN}\"\n";
	print CONFIG "# base_dn for the replication,  this must match exactly, include any whitespace\n";
	print CONFIG "BASE_DN=\"$config{BASE_DN}\"\n";
	print CONFIG "# replication ID value,  script will auto-increment when needed\n";
	print CONFIG "REP_ID=\"$config{REP_ID}\"\n";
	print CONFIG "# location of libpam-ldap's secret file (or any file containing the ROOT_DN password)\n";
	print CONFIG "PAM_SECRET=\"$config{PAM_SECRET}\"\n";
	print CONFIG "# location of libnss-ldap's secret file.  libpam gets checked first\n";
	print CONFIG "NSS_SECRET=\"$config{NSS_SECRET}\"\n";
	print CONFIG "# REP_DN user's password\n";
	print CONFIG "REPMAN_PW=\"$config{REPMAN_PW}\"\n";
	print CONFIG "# location of ca certs for TLS enc\n";
	print CONFIG "CA_DIR=\"$config{CA_DIR}\"\n";
	print CONFIG "# set TLS=1 to always encrypt session with TLS, otherwise set to 0\n";
	print CONFIG "# you can still encrypt by using -z switch\n";
	print CONFIG "TLS=\"$config{TLS}\"\n";

	close CONFIG;
}

sub load_config
{

	open CONFIG, $fds_tool_config;
	
	while (<CONFIG>) {
		chomp;                  # no newline
        	s/#.*//;                # no comments
		s/^\s+//;               # no leading white
		s/\s+$//;               # no trailing white
		next unless length;     # anything left?
		my ($var, $value) = split(/\s*=\s*/, $_, 2);
		# strip quotes off of value
		$value =~ s/"//;
		$value =~ s/"$//;
		$config{$var} = $value;
		
	} 
	close CONFIG;

	
	# reset lists in case we're reloading config
	@sup_server = ();
	@sup_server_ports = ();
	@con_server = ();
	@con_server_ports = ();
	
	# create server lists for ease of use.
	my $count=0;
	my @temp;
	foreach(split(/;/,$config{SUPPLIERS})) {
		# split the string to check for ports
 		@temp = split(/:/,$_);
 		$sup_server[$count] = $temp[0];
		if ($temp[1]) {
			$sup_server_ports[$count] = $temp[1];
		} else {
			# default to 389
			$sup_server_ports[$count] = 389;
		}
	
		$count = $count +1;
	}
	$count=0;
	foreach(split(/;/,$config{CONSUMERS}))	 {
		# split the string to check for ports
		@temp = split(/:/,$_);
		$con_server[$count] = $temp[0];
		if ($temp[1]) {
			$con_server_ports[$count] = $temp[1];
		} else {
			# default to 389
			$con_server_ports[$count] = 389;
		}
		
		$count = $count +1;
	}


}

sub dump_all
{
	# this will dump all the replication info for each server to the screen for troubleshooting
	my @all_servers = (@sup_server, @con_server);
	my @all_ports = (@sup_server_ports, @con_server_ports);
	my $count = 0;
	
	foreach my $srvr (@all_servers) {
		dump_info($srvr,$all_ports[$count],$passwd);
		$count = $count + 1;
	}

	print "QUITTING\n\n\n";
	exit;

}

#################################################
######### Info dump subs
#################################################

sub dump_info
{
	my ($server, $port, $bind_pw) = @_;
	my ($ldap, $msg);
	
	
	if ( ($ldap = conn_bind($server,$port,$bind_pw)) eq 0 ) {
		print "\t*********** bind/connect failed to $server on port $port ***************\n";
		return 0;
	}

	$msg = $ldap->search(filter=>"objectClass=nsDS5Replica", 
			 base=>"cn=config");
	print "Dumping Replica agreement on $server\n\n";
	foreach my $entry ($msg->entries) { $entry->dump; print "\n\n"; }
	
	$msg = $ldap->search(filter=>"objectClass=nsDS5ReplicationAgreement", 
			 base=>"cn=config");
	print "\nDumping all Replication agreements on $server\n\n";
	foreach my $entry ($msg->entries) { $entry->dump; print "\n\n"; }

	
	$msg = $ldap->search(filter=>"cn=changelog5", 
			 base=>"cn=config");
	print "\nDumping changelog object on $server\n\n";
	foreach my $entry ($msg->entries) { $entry->dump; print "\n\n"; }

	$msg = $ldap->search(filter=>"objectClass=nsMappingTree", 
			 base=>"cn=config");
	print "\nDumping mapping tree object on $server\n\n";
	foreach my $entry ($msg->entries) { $entry->dump; print "\n\n"; }


}
