module Main where

import Foreign
import Foreign.C
import Foreign.C.String

-- When writing a library, one wouldn't want to export one of these
-- since it doesn't make sense to instanciate them.
data Base = Base ()
data Derived = Derived ()

data BasePtr    = BasePtr    (ForeignPtr Base)
data DerivedPtr = DerivedPtr (ForeignPtr Derived)

class BaseDerived a where
  toBase :: a -> BasePtr

instance BaseDerived BasePtr where
  toBase (BasePtr a) = BasePtr a

instance BaseDerived DerivedPtr where
  toBase (DerivedPtr a) = BasePtr $ castForeignPtr a


-- Standard compiant compilers would need the following two functions
-- foreign import ccall "wrapper" 
--   f_deleteBaseFunPtr :: ((Ptr Base) -> IO ())-> 
--     IO (FunPtr ((Ptr Base) -> IO()))
-- foreign import ccall "wrapper" 
--   f_deleteDerivedFunPtr :: ((Ptr Derived) -> IO ())-> 
--     IO (FunPtr ((Ptr Derived) -> IO()))

foreign import ccall f_newBase    :: IO (Ptr Base)
foreign import ccall f_deleteBase :: (Ptr Base) -> IO ()
foreign import ccall f_Base_memfun_printName :: (Ptr Base) -> IO ()

foreign import ccall f_newDerived :: IO (Ptr Derived)
foreign import ccall f_deleteDerived :: (Ptr Derived) -> IO ()


newBase :: IO BasePtr
newBase = do
  a <- f_newBase :: IO( Ptr Base)
  --c <- f_deleteBaseFunPtr f_deleteBase :: IO( FunPtr ((Ptr Base) -> IO () ) )
  --b <- newForeignPtr a c :: IO( ForeignPtr Base )
  --Standard compilers would use the above two lines instead of the following
  b <- newForeignPtr a (f_deleteBase a) :: IO( ForeignPtr Base )
  return (BasePtr b)

newDerived :: IO DerivedPtr
newDerived = do
  a <- f_newDerived :: IO( Ptr Derived)
  --c <- f_deleteDerivedFunPtr f_deleteDerived
  --b <- newForeignPtr a c
  --Standard compilers would use the above two lines instead of the following
  b <- newForeignPtr a (f_deleteDerived a) :: IO( ForeignPtr Derived )
  return (DerivedPtr b)

base_memfun_printName :: (BaseDerived a) => a -> IO ()
base_memfun_printName a = do
  let BasePtr b = toBase a
  f_Base_memfun_printName $ foreignPtrToPtr b
  touchForeignPtr b -- This is used to ensure the pointer isn't deleted before
                    -- the above function is called
  return ()

-- This is an alternative way to do things, but it may pollute your namespace
--
class CppPointer a where
  (->>) :: forall b . a -> ( a -> b ) -> b
  c ->> d = d c

printName :: (BaseDerived a) => a -> IO ()
printName = base_memfun_printName
--
-- End Alternitive way

instance CppPointer BasePtr

main = do
    b <- newBase
    d <- newDerived
    -- b->>printName -- The alternative, more c++ like way to 
                  -- interface to Haskell
    base_memfun_printName b
    base_memfun_printName d





