/*
 * QuickTime palette handling
 * Copyright (c) 2015 Mats Peterson
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <stdio.h>
#include <inttypes.h>

#include "libavcodec/avcodec.h"
#include "libavutil/intreadwrite.h"
#include "qtpalette.h"

int get_qtpalette(int codec_id, uint8_t *stsd, uint32_t *palette)
{
    int tmp, bit_depth, greyscale, i;

    /* Get the bit depth and greyscale state */
    tmp = AV_RB16(stsd + 66);
    bit_depth = tmp & 0x1F;
    greyscale = tmp & 0x20;

    /* Do not create a greyscale palette for Cinepak */
    if (greyscale && codec_id == AV_CODEC_ID_CINEPAK)
        return 0;

    /* If the depth is 2, 4, or 8 bpp, file is palettized. */
    if ((bit_depth == 2 || bit_depth == 4 || bit_depth == 8)) {
        int color_table_id, color_count;
        uint32_t a, r, g, b;

        color_table_id = AV_RB16(stsd + 68);

        if (greyscale) {
            int color_index, color_dec;
            /* compute the greyscale palette */
            color_count = 1 << bit_depth;
            color_index = 255;
            color_dec   = 256 / (color_count - 1);
            for (i = 0; i < color_count; i++) {
                r = g = b = color_index;
                palette[i] = (0xFFU << 24) | (r << 16) | (g << 8) | (b);
                color_index -= color_dec;
                if (color_index < 0)
                    color_index = 0;
            }
        } else if (color_table_id) {
            /* The color table ID is non-zero. Interpret this as
             * being -1, which means use the default Macintosh
             * color table */
            const uint8_t *color_table;
            color_count = 1 << bit_depth;
            if (bit_depth == 2)
                color_table = ff_qt_default_palette_4;
            else if (bit_depth == 4)
                color_table = ff_qt_default_palette_16;
            else
                color_table = ff_qt_default_palette_256;
            for (i = 0; i < color_count; i++) {
                r = color_table[i * 3 + 0];
                g = color_table[i * 3 + 1];
                b = color_table[i * 3 + 2];
                palette[i] = (0xFFU << 24) | (r << 16) | (g << 8) | (b);
            }
        } else {
            /* The color table ID is 0; the color table is in the sample
             * description */
            color_count = AV_RB16(stsd + 76) + 1;
            fprintf(stderr, "COLOR COUNT: %d\n\n", color_count);
            if (color_count <= 256) {
                uint8_t *p = stsd + 78;
                for (i = 0; i < color_count; i++) {
                    /* each A, R, G, or B component is 16 bits;
                       only use the top 8 bits */
                    a = *p++; p++;
                    r = *p++; p++;
                    g = *p++; p++;
                    b = *p++; p++;
                    palette[i] = (a << 24 ) | (r << 16) | (g << 8) | (b);
                }
            }
        }

        return 1;
    }

    return 0;
}
