/*-
 * Copyright (c) 2006 Iain Hibbert
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/audioio.h>
#include <sys/ioctl.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <poll.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "bthfp.h"

static void do_signal(int);
static int open_audio(const char *);
static int copy_data(int, int);

/*
 * We are the child process.
 *
 * Our job is to open the handsfree audio device, which will block until
 * the Audio Gateway opens the connection. Then, we open the real audio
 * device and copy sounds each way until the handsfree device drops the 
 * connection.
 *
 * Rinse and Repeat until the fat controller sends a SIGHUP.
 */
void
audio_baby(void)
{
	struct sigaction sa;
	struct pollfd fds[2];

	memset(&sa, 0, sizeof(sa));
	sa.sa_handler = do_signal;

	if (sigaction(SIGHUP, &sa, NULL) < 0)
		_exit(EXIT_FAILURE);

	for (;;) {
		fds[0].events = POLLIN | POLLERR | POLLHUP;
		fds[0].fd = open_audio(handsfree);	/* blocks */
		if (fds[0].fd < 0) {
			warn("%s", handsfree);
			break;
		}

		fds[1].events = POLLIN | POLLERR | POLLHUP;
		fds[1].fd = open_audio(speaker);
		if (fds[1].fd < 0) {
			warn("%s", speaker);
			break;
		}

		fprintf(stderr, "Audio Connection established\n");

		for (;;) {
			fds[0].revents = 0;
			fds[1].revents = 0;

			if (poll(fds, 2, INFTIM) < 0)
				break;

			if ((fds[0].revents && copy_data(fds[0].fd, fds[1].fd) < 0))
				break;

			if ((fds[1].revents && copy_data(fds[1].fd, fds[0].fd) < 0))
				break;
		}

		fprintf(stderr, "Audio Connection released\n");

		close(fds[0].fd);
		close(fds[1].fd);
	}

	_exit(EXIT_FAILURE);
}

static void
do_signal(int s)
{

	switch (s) {
	case SIGHUP:
	default:
		_exit(EXIT_SUCCESS);
	}
}

/*
 * Open Audio device 
 */
static int
open_audio(const char *name)
{
	audio_info_t	info;
	int		fd, n;

	fd = open(name, O_RDWR, 0);
	if (fd < 0)
		return -1;

	if (fcntl(fd, F_SETFL, O_NONBLOCK | O_ASYNC) < 0)
		return -1;

	n = 1;	/* set full duplex */
	if (ioctl(fd, AUDIO_SETFD, &n) < 0)
		return -1;

	/*
	 * set play and record modes of
	 *	8kHz 16-bit slinear mono
	 */
	if (ioctl(fd, AUDIO_GETINFO, &info) < 0)
		return -1;

	info.mode = (AUMODE_PLAY | AUMODE_RECORD);

	info.play.sample_rate = 8000;
	info.play.channels = 1;
	info.play.precision = 16;
	info.play.encoding = AUDIO_ENCODING_SLINEAR_LE;

	info.record.sample_rate = 8000;
	info.record.channels = 1;
	info.record.precision = 16;
	info.record.encoding = AUDIO_ENCODING_SLINEAR_LE;

	if (ioctl(fd, AUDIO_SETINFO, &info) < 0)
		return -1;

	return fd;
}

/*
 * Copy audio data from src to dst file descriptor.
 */
int
copy_data(int src, int dst)
{
	static char	buf[4096];
	ssize_t		len;

	for (;;) {
		len = read(src, buf, sizeof(buf));
		if (len == 0)
			return -1;

		if (len < 0) {
			if (errno != EAGAIN)
				return -1;

			break;
		}

		if (write(dst, buf, len) != len)
			return -1;
	}

	return 0;
}
