/*-
 * Copyright (c) 2006 Iain Hibbert
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <bluetooth.h>
#include <err.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <netbt/rfcomm.h>

#include "bthfp.h"

static void usage(void);
static void knife(void);
static void sighandler(int);

/* global options */
const char	*speaker;
const char	*handsfree;
const char	*mixer;
int		 verbose;
int		 server_channel;

static pid_t	 pid;	/* audio process */

int
main(int ac, char *av[])
{
	struct sigaction	sa;
	int			ch;

	speaker = getenv("BTHFP_AUDIO0");
	if (speaker == NULL)
		speaker = "/dev/audio0";

	handsfree = getenv("BTHFP_AUDIO1");
	if (handsfree == NULL)
		handsfree = "/dev/audio1";

	mixer = getenv("BTHFP_MIXER1");
	if (mixer == NULL)
		mixer = "/dev/mixer1";

	server_channel = 0;
	verbose = 0;

	while ((ch = getopt(ac, av, "c:h:m:s:v")) != -1) {
		switch (ch) {
		case 'c':	/* Server channel */
			server_channel = atoi(optarg);
			if (server_channel < RFCOMM_CHANNEL_MIN
			    || server_channel > RFCOMM_CHANNEL_MAX)
				errx(EXIT_FAILURE, "Bad channel number");;

			break;

		case 'h':	/* handsfree audio device */
			handsfree = optarg;
			break;

		case 'm':	/* handsfree mixer device */
			mixer = optarg;
			break;

		case 's':	/* speaker device */
			speaker = optarg;
			break;

		case 'v':	/* Verbose */
			verbose = 1;
			break;

		default:
			usage();
		}
	}

	pid = fork();
	if (pid == -1)
		err(EXIT_FAILURE, "fork");
		
	if (pid == 0) {
		audio_baby();
		_exit(EXIT_FAILURE);
	}

	atexit(knife);

	memset(&sa, 0, sizeof(sa));
	sa.sa_handler = sighandler;
	if (sigaction(SIGCHLD, &sa, NULL) < 0)
		err(EXIT_FAILURE, "sigaction");

	controller();
	exit(EXIT_FAILURE);
}

static void
sighandler(int s)
{

	switch (s) {
	case SIGCHLD:
		pid = -1;
		break;

	default:
		break;
	}
}

static void
knife(void)
{

	if (pid != -1)
		kill(pid, SIGHUP);
}

static void
usage(void)
{

	fprintf(stderr,
		"usage: %s [-v] [-c channel] [-h device] [-m device] [-s device]\n"
		"Where:\n"
		"\t-c channel   specify local channel (and register as Handsfree)\n"
		"\t-h device    btsco audio device (default: /dev/audio1)\n"
		"\t-m device    btsco mixer device (default: /dev/mixer1)\n"
		"\t-s device    system audio device (default: /dev/audio0)\n"
		"\t-v           verbose output\n"
		"", getprogname());

	exit(EXIT_FAILURE);
}
