/*-
 * Copyright (c) 2006 Iain Hibbert
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/audioio.h>
#include <sys/ioctl.h>

#include <bluetooth.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <poll.h>
#include <sdp.h>
#include <signal.h>
#include <stdarg.h>
#include <stdlib.h>
#include <termios.h>
#include <unistd.h>

#include <netbt/rfcomm.h>
#include <dev/bluetooth/btsco.h>

#include "bthfp.h"

static void sighandler(int);

static int init_kbd(void);
static void reset_kbd(void);
static int read_kbd(int);

static int init_mixer(void);

static int init_server(void);
static int accept_rfcomm(int);

static int open_rfcomm(void);
static void establish_rfcomm(void);
static int send_rfcomm(const char *, ...);
static int recv_rfcomm(void);

static void rc_error(char *);
static void rc_ok(char *);
static void rc_ring(char *);
static void rc_no_carrier(char *);
static void rc_busy(char *);
static void rc_no_answer(char *);
static void rc_delayed(char *);
static void rc_blacklisted(char *);
static void rc_brsf(char *);
static void rc_bsir(char *);
static void rc_btrh(char *);
static void rc_bvra(char *);
static void rc_ccwa(char *);
static void rc_ciev(char *);
static void rc_cind(char *);
static void rc_clcc(char *);
static void rc_cme_error(char *);
static void rc_clip(char *);
static void rc_cnum(char *);
static void rc_vgm(char *);
static void rc_vgs(char *);

/* stored termios */
static struct termios tio;

/* state of establishment */
static enum {	CLOSED,
	BRSF_SET,
	CIND_TEST,
	CIND_READ,
	CMER_SET,
	CLIP_SET,
	SERVICE
} state;

/* global store */
static int rf;		/* RFCOMM fd */
static bdaddr_t laddr;	/* local bdaddr */
static bdaddr_t raddr;	/* remote bdaddr */
static int channel;	/* remote channel */

static struct result_code {
	const char	*code;
	int		strlen;
	void		(*handler)(char *);
} result_codes[] = {
	{ "ERROR",		5,	rc_error	},
	{ "OK",			2,	rc_ok		},
	{ "RING",		4,	rc_ring		},
	{ "NO CARRIER",		10,	rc_no_carrier	},
	{ "BUSY",		4,	rc_busy		},
	{ "NO ANSWER",		9,	rc_no_answer	},
	{ "DELAYED",		7,	rc_delayed	},
	{ "BLACKLISTED",	11,	rc_blacklisted	},
	{ "+BRSF:",		6,	rc_brsf		},
	{ "+BSIR:",		6,	rc_bsir		},
	{ "+BTRH:",		6,	rc_btrh		},
	{ "+BVRA:",		6,	rc_bvra		},
	{ "+CCWA:",		6,	rc_ccwa		},
	{ "+CIEV:",		6,	rc_ciev		},
	{ "+CIND:",		6,	rc_cind		},
	{ "+CLCC:",		6,	rc_clcc		},
	{ "+CME ERROR:",	11,	rc_cme_error	},
	{ "+CLIP:",		6,	rc_clip		},
	{ "+CNUM:",		6,	rc_cnum		},
	{ "+VGM:",		5,	rc_vgm		},
	{ "+VGS:",		5,	rc_vgs		},
	{ NULL }
};

#define P_STDIN		0
#define P_MIXER		1
#define P_SERVER	2
#define P_RFCOMM	3
#define P_NUM		4

void
controller(void)
{
	struct sigaction sa;
	struct pollfd	 fds[P_NUM];
	int		 n, done;

	memset(&sa, 0, sizeof(sa));
	sa.sa_handler = sighandler;
	if (sigaction(SIGINT, &sa, NULL) < 0
	    || sigaction(SIGHUP, &sa, NULL) < 0
	    || sigaction(SIGTERM, &sa, NULL) < 0
	    || sigaction(SIGPIPE, &sa, NULL) < 0)
		err(EXIT_FAILURE, "sigaction");

	fds[P_STDIN].fd = init_kbd();
	fds[P_STDIN].events = POLLIN | POLLERR | POLLHUP;

	fds[P_MIXER].fd = init_mixer();
	fds[P_MIXER].events = POLLIN | POLLERR | POLLHUP;

	fds[P_SERVER].fd = init_server();
	fds[P_SERVER].events = POLLIN | POLLERR | POLLHUP;

	fds[P_RFCOMM].fd = -1;
	fds[P_RFCOMM].events = POLLIN | POLLERR | POLLHUP;

	state = CLOSED;
	done = 0;
	rf = -1;

	while (!done) {
		fds[P_STDIN].revents = 0;
		fds[P_MIXER].revents = 0;
		fds[P_SERVER].revents = 0;
		fds[P_RFCOMM].revents = 0;

		if (state == CLOSED)
			fds[P_RFCOMM].fd = open_rfcomm();

		n = poll(fds, P_NUM, INFTIM);
		if (n < 0)
			break;

		if (fds[P_STDIN].revents)
			done = read_kbd(fds[P_STDIN].fd);

		if (fds[P_MIXER].revents)
			/* handler dat mixer event */;

		if (fds[P_SERVER].revents)
			fds[P_RFCOMM].fd = accept_rfcomm(fds[P_SERVER].fd);

		if (fds[P_RFCOMM].revents)
			fds[P_RFCOMM].fd = recv_rfcomm();
	}
}

static void
sighandler(int s)
{

	switch (s) {
	case SIGPIPE:
		printf("caught SIGPIPE, exiting\n");
		
	case SIGTERM:
	case SIGHUP:
	case SIGINT:
	default:
		exit(EXIT_FAILURE);
	}
}

/*
 * Set up the keyboard handler
 */
static int
init_kbd(void)
{
	struct termios t;

	/*
	 * Take a copy of the termios structure so that we can
	 * re-instate it at exit. Turn off echo and canonical
	 * line handling for stdin.
	 */
	if (tcgetattr(STDIN_FILENO, &t) < 0)
		return -1;

	memcpy(&tio, &t, sizeof(tio));
	t.c_lflag &= ~(ECHO | ICANON);

	if (tcsetattr(STDIN_FILENO, TCSANOW, &t) < 0)
		return -1;

	atexit(reset_kbd);

	return STDIN_FILENO;
}

/*
 * Reset saved terminal state on stdin
 */
static void
reset_kbd(void)
{

	tcsetattr(STDIN_FILENO, TCSAFLUSH, &tio);
}

/*
 * Keyboard event. Return a value if the user wants to quit.
 */
static int
read_kbd(int fd)
{
	char key;

	if (read(fd, &key, 1) != 1)
		err(EXIT_FAILURE, "read_kbd");

	switch (key) {
	case 'a': case 'A':	/* Answer */
		send_rfcomm("A");
		break;

	case 'h': case 'H':	/* Hangup */
		send_rfcomm("+CHUP");
		break;

	case 'q': case 'Q':	/* Quit */
		return 1;

	default:
		break;
	}

	return 0;
}

/*
 * Initialise the mixer device, including getting information
 * about connections we want to make.
 */
static int
init_mixer(void)
{
	struct btsco_info info;
	int fd;

	fd = open(mixer, O_WRONLY, 0);
	if (fd < 0)
		err(EXIT_FAILURE, "%s", mixer);

	if (ioctl(fd, BTSCO_GETINFO, &info) < 0)
		err(EXIT_FAILURE, "BTSCO_GETINFO");

	bdaddr_copy(&laddr, &info.laddr);
	bdaddr_copy(&raddr, &info.raddr);
	channel = info.channel;

/*
	memset(&vgs, 0, sizeof(vgs));
	vgs.dev = info->vgs;
	if (ioctl(fd, AUDIO_MIXER_READ, &vgs) < 0)
		err(EXIT_FAILURE, "AUDIO_MIXER_READ");

	memset(&vgm, 0, sizeof(vgm));
	vgm.dev = info->vgm;
	if (ioctl(fd, AUDIO_MIXER_READ, &vgm) < 0)
		err(EXIT_FAILURE, "AUDIO_MIXER_READ");
*/

	/* enable mixer changed event */
	if (fcntl(fd, F_SETFL, O_ASYNC) < 0)
		err(EXIT_FAILURE, "F_SETFL");

	return fd;
}

/*
 * Initialise RFCOMM server socket and register as HANDSFREE
 * with local SDP server.
 */
static int
init_server(void)
{
	struct sockaddr_bt addr;
	sdp_hf_profile_t hf;
	void *ss;
	int fd;

	if (server_channel == 0)
		return -1;

	fd = socket(PF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM);
	if (fd < 0)
		err(EXIT_FAILURE, "socket");

	memset(&addr, 0, sizeof(addr));
	addr.bt_len = sizeof(addr);
	addr.bt_family = AF_BLUETOOTH;
	bdaddr_copy(&addr.bt_bdaddr, &laddr);
	addr.bt_channel = server_channel;

	if (bind(fd, (struct sockaddr *)&laddr, sizeof(laddr)) < 0)
		err(EXIT_FAILURE, "%s/%d", bt_ntoa(&laddr, NULL), server_channel);

	if (listen(fd, 1) < 0)
		err(EXIT_FAILURE, "listen");

	/* Register as Handsfree */
	memset(&hf, 0, sizeof(hf));
	hf.server_channel = server_channel;
	hf.supported_features = 0x0000;

	ss = sdp_open_local(NULL);
	if (ss == NULL || (errno = sdp_error(ss)) != 0)
		err(EXIT_FAILURE, "sdp_open_local");

	if (sdp_register_service(ss,
			SDP_SERVICE_CLASS_HANDSFREE,
			&laddr, (uint8_t *)&hf, sizeof(hf), NULL) != 0) {
		errno = sdp_error(ss);
		err(EXIT_FAILURE, "sdp_register_service");
	}

	return fd;
}

/*
 * Accept RFCOMM connection, and start establishment.
 */
static int
accept_rfcomm(int sv)
{
	struct sockaddr_bt addr;
	socklen_t len;
	int fd;

	memset(&addr, 0, sizeof(addr));
	len = sizeof(addr);
	fd = accept(sv, (struct sockaddr *)&addr, &len);
	if (fd < 0)
		return rf;

	if (state != CLOSED
	    || addr.bt_len != sizeof(addr)
	    || addr.bt_family != AF_BLUETOOTH
	    || !bdaddr_same(&addr.bt_bdaddr, &raddr)) {
		close(fd);
		return rf;
	}

	rf = fd;
	establish_rfcomm();
	return rf;
}

/*
 * Open RFCOMM connection, and start establishment.
 */
static int
open_rfcomm(void)
{
	struct sockaddr_bt addr;
	int fd;

	if (state != CLOSED)
		warnx("%s() but state != CLOSED!", __func__);

	fd = socket(PF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM);
	if (fd < 0)
		return -1;

	memset(&addr, 0, sizeof(addr));
	addr.bt_len = sizeof(addr);
	addr.bt_family = AF_BLUETOOTH;
	bdaddr_copy(&addr.bt_bdaddr, &laddr);

	if (bind(fd, (struct sockaddr *)&addr, sizeof(addr)) < 0)
		return -1;

	printf("Connecting.. ");
	fflush(stdout);

	bdaddr_copy(&addr.bt_bdaddr, &raddr);
	addr.bt_channel = channel;

	if (connect(fd, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
		close(fd);
		printf("failed\n");
		return -1;
	}
	printf("ok\n");

	rf = fd;
	establish_rfcomm();
	return fd;
}

/*
 * Establish Service Level connection on RFCOMM link, this will
 * be called after each received OK.
 */
static void
establish_rfcomm(void)
{

	switch (state) {
	case CLOSED:
		/*
	 	*	Bit	Feature
	 	*	0	EC and/or NR function
	 	*	1	Call waiting and 3-way calling
	 	*	2	CLI presentation capability
	 	*	3	Voice Recognition activation
	 	*	4	Remote Volume Control
	 	*	5	Enhanced call status
	 	*	6	Enhanced call control
	 	*	7-31	Reserved
	 	*/
		state = BRSF_SET;
		send_rfcomm("+BRSF=%u", (1 << 4) | (1 <<2));
		break;

	case BRSF_SET:
		state = CIND_TEST;
		send_rfcomm("+CIND=?");
		break;

	case CIND_TEST:
		state = CIND_READ;
		send_rfcomm("+CIND?");
		break;

	case CIND_READ:
		state = CMER_SET;
		send_rfcomm("+CMER=3,0,0,1");
		break;

	case CMER_SET:
		state = CLIP_SET;
		send_rfcomm("+CLIP=1");
		break;

	case CLIP_SET:
		state = SERVICE;
		printf("Service Level established\n");
		break;

	case SERVICE:
		break;

	default:
		break;
	}
}

/*
 * Send a message to the Audio Gateway in appropriate wrapping.
 */
static int
send_rfcomm(const char *msg, ...)
{
	char buf[256], fmt[256];
	va_list ap;
	int len;

	va_start(ap, msg);

	if (verbose) {
		snprintf(fmt, sizeof(fmt), "< AT%s\n", msg);
		vprintf(fmt, ap);
	}

	snprintf(fmt, sizeof(fmt), "AT%s\r", msg);
	vsnprintf(buf, sizeof(buf), fmt, ap);
	len = send(rf, buf, strlen(buf), 0);

	va_end(ap);
	return len;
}

/*
 * Receive a message from the Audio Gateway
 */
static int
recv_rfcomm(void)
{
	struct result_code *rc;
	char buf[1024], *code, *next;
	size_t len;

	/* XXX and what if buf is not large enough? */

	len = recv(rf, buf, sizeof(buf), 0);
	if (len < 0) {
		printf("Service Connection lost\n");
		state = CLOSED;
		close (rf);
		rf = -1;
		return -1;
	}

	/*
	 * Result Codes should be <cr><lf><code><cr><lf>, and we
	 * may get several stacked up at once.
	 */

	next = buf;

	for (;;) {
		code = next;
		if (len < 2 || (code[0] != '\r' && code[1] != '\n'))
			break;

		code += 2;
		len -= 2;

		next = code;
		while (len > 0 && next[0] != '\r')
			next++, len--;

		if (len < 2 || next[1] != '\n')
			break;

		next[0] = '\0';
		next += 2;
		len -= 2;

		if (verbose)
			printf("> %.*s\n", next - code, code);

		for (rc = result_codes ; rc->code != NULL ; rc++) {
			if (strncmp(code, rc->code, rc->strlen) == 0) {
				(*rc->handler)(code + rc->strlen);
				break;
			}
		}
	}

	return rf;
}

/*
 * Error
 */
static void
rc_error(char *arg)
{

	switch (state) {
	case BRSF_SET:
		/*
		 * Older devices might not know BRSF command.
		 * (For these we should do a SDP query)
		 */
		establish_rfcomm();
		break;

	case CIND_TEST:
	case CIND_READ:
	case CMER_SET:
	case CLIP_SET:
	case SERVICE:
	default:
		break;
	}
}

/*
 * Ok
 */
static void
rc_ok(char *arg)
{

	/*
	 * continue towards service level..
	 */
	if (state != SERVICE)
		establish_rfcomm();
}

/*
 * Ring
 */
static void
rc_ring(char *arg)
{

	printf("Ring\n");
}

/*
 * No Carrier
 */
static void
rc_no_carrier(char *arg)
{
}

/*
 * Busy
 */
static void
rc_busy(char *arg)
{
}

/*
 * No Answer
 */
static void
rc_no_answer(char *arg)
{
}

/*
 * Delayed
 */
static void
rc_delayed(char *arg)
{
}

/*
 * Blacklisted
 */
static void
rc_blacklisted(char *arg)
{
}

/*
 * Bluetooth Retrieve Supported Features
 *
 * +BRSF: <AG supported features bitmap>
 *
 *	Bit	Feature
 *	0	Three-way calling
 *	1	EC and/or NR function
 *	2	Voice recognition function
 *	3	In-band ring tone capability
 *	4	Attach a number to a voice tag
 *	5	Ability to reject a call
 *	6	Enhanced call status
 *	7	Enhanced call control
 *	8	Extended Error Result Codes
 *	9-31	Reserved
 */
static void
rc_brsf(char *arg)
{
	uint16_t features;

	features = atoi(arg);
	printf("Features: 0x%4.4x\n", features);
}

/*
 * Bluetooth Setting of In-band Ring tone
 */
static void
rc_bsir(char *arg)
{
}

/*
 * Bluetooth Response and Hold Feature
 */
static void
rc_btrh(char *arg)
{
}

/*
 * Bluetooth Voice Recognition Activation
 */
static void
rc_bvra(char *arg)
{
}

/*
 * Call Waiting notification
 */
static void
rc_ccwa(char *arg)
{
}

/*
 * Indicator Events reporting
 */
static void
rc_ciev(char *arg)
{
}

/*
 * Current phone indicators
 */
static void
rc_cind(char *arg)
{

	switch (state) {
	case CIND_TEST: /* (<descr>,(<ind-list>))[,(<descr>,(<ind-list>))[,...]] */
		break;

	case CIND_READ:	/* <ind>[,<ind>[,...]] */
		break;

	default:
		break;
	}
}

/*
 * List Current Calls
 */
static void
rc_clcc(char *arg)
{
}

/*
 * Communications Mobile Equipment Error
 */
static void
rc_cme_error(char *arg)
{
}

/*
 * Calling Line Identification
 */
static void
rc_clip(char *arg)
{
}

/*
 * Subscriber Number Information
 */
static void
rc_cnum(char *arg)
{
}

/*
 * Gain of Microphone
 */
static void
rc_vgm(char *arg)
{
}

/*
 * Gain of Speaker
 */
static void
rc_vgs(char *arg)
{
}
