////////////////////////////////////////////////////////
//
// GEM - Graphics Environment for Multimedia
//
// zmoelnig@iem.kug.ac.at
//
// Implementation file
//
//    Copyright (c) 1997-2000 Mark Danks.
//    Copyright (c) Gnther Geiger.
//    Copyright (c) 2001-2002 IOhannes m zmoelnig. forum::fr::umlute. IEM
//    Copyright (c) 2002 James Tittle & Chris Clepper
//    For information on usage and redistribution, and for a DISCLAIMER OF ALL
//    WARRANTIES, see the file, "GEM.LICENSE.TERMS" in this distribution.
//
/////////////////////////////////////////////////////////

#include "pix_freenect.h"
#include "Base/GemState.h"
#include "Base/GemCache.h"
#include "Base/GemConfig.h"

CPPEXTERN_NEW_WITH_TWO_ARGS(pix_freenect, t_float, A_DEFFLOAT, t_float, A_DEFFLOAT); // argument -> Kinect ID
//CPPEXTERN_NEW(pix_freenect)

/////////////////////////////////////////////////////////
//
// LIBFREENECT C++ WRAPPER STUFF
//
/////////////////////////////////////////////////////////

class Mutex {
public:
	Mutex() {
		pthread_mutex_init( &m_mutex, NULL );
	}
	void lock() {
		pthread_mutex_lock( &m_mutex );
	}
	void unlock() {
		pthread_mutex_unlock( &m_mutex );
	}
private:
	pthread_mutex_t m_mutex;
};

/* thanks to Yoda---- from IRC */
class MyFreenectDevice : public Freenect::FreenectDevice {
public:
	MyFreenectDevice(freenect_context *_ctx, int _index)
		: Freenect::FreenectDevice(_ctx, _index), m_buffer_depth(640*480*4),m_buffer_video(640*480*4), m_gamma(2048), m_new_rgb_frame(false), m_new_depth_frame(false) //I made the buffer bigger because of RGBA -> could be solved better
	{
		for( unsigned int i = 0 ; i < 2048 ; i++) {
			float v = i/2048.0;
			v = std::pow(v, 3)* 6;
			m_gamma[i] = v*6*256;
		}
	}
	//~MyFreenectDevice(){}
	// Do not call directly even in child
	void VideoCallback(void* _rgb, uint32_t timestamp) {
		m_rgb_mutex.lock();
		uint8_t* rgb = static_cast<uint8_t*>(_rgb);
		std::copy(rgb, rgb+getVideoBufferSize(), m_buffer_video.begin());
		m_new_rgb_frame = true;
		m_rgb_mutex.unlock();
	};
	// Do not call directly even in child
	void DepthCallback(void* _depth, uint32_t timestamp) {
		m_depth_mutex.lock();
		uint16_t* depth = static_cast<uint16_t*>(_depth);
		for( unsigned int i = 0 ; i < FREENECT_FRAME_PIX ; i++) {
			int pval = m_gamma[depth[i]];
			int lb = pval & 0xff;
			int form_mult = 4; // Changed for RGBA (4 instead of originally 3)
			switch (pval>>8) {
			case 0:																					
				m_buffer_depth[form_mult*i+0] = 255;
				m_buffer_depth[form_mult*i+1] = 255-lb;
				m_buffer_depth[form_mult*i+2] = 255-lb;
				break;
			case 1:
				m_buffer_depth[form_mult*i+0] = 255;
				m_buffer_depth[form_mult*i+1] = lb;
				m_buffer_depth[form_mult*i+2] = 0;
				break;
			case 2:
				m_buffer_depth[form_mult*i+0] = 255-lb;
				m_buffer_depth[form_mult*i+1] = 255;
				m_buffer_depth[form_mult*i+2] = 0;
				break;
			case 3:
				m_buffer_depth[form_mult*i+0] = 0;
				m_buffer_depth[form_mult*i+1] = 255;
				m_buffer_depth[form_mult*i+2] = lb;
				break;
			case 4:
				m_buffer_depth[form_mult*i+0] = 0;
				m_buffer_depth[form_mult*i+1] = 255-lb;
				m_buffer_depth[form_mult*i+2] = 255;
				break;
			case 5:
				m_buffer_depth[form_mult*i+0] = 0;
				m_buffer_depth[form_mult*i+1] = 0;
				m_buffer_depth[form_mult*i+2] = 255-lb;
				break;
			default:
				m_buffer_depth[form_mult*i+0] = 0;
				m_buffer_depth[form_mult*i+1] = 0;
				m_buffer_depth[form_mult*i+2] = 0;
				break;
			}
		}
		m_new_depth_frame = true;
		m_depth_mutex.unlock();
	}
	bool getRGB(std::vector<uint8_t> &buffer) {
		m_rgb_mutex.lock();
		if(m_new_rgb_frame) {
			buffer.swap(m_buffer_video);
			m_new_rgb_frame = false;
			m_rgb_mutex.unlock();
			return true;
		} else {
			m_rgb_mutex.unlock();
			return false;
		}
	}

	bool getDepth(std::vector<uint8_t> &buffer) {
		m_depth_mutex.lock();
		if(m_new_depth_frame) {
			buffer.swap(m_buffer_depth);
			m_new_depth_frame = false;
			m_depth_mutex.unlock();
			return true;
		} else {
			m_depth_mutex.unlock();
			return false;
		}
	}

private:
	std::vector<uint8_t> m_buffer_depth;
	std::vector<uint8_t> m_buffer_video;
	std::vector<uint16_t> m_gamma;
	Mutex m_rgb_mutex;
	Mutex m_depth_mutex;
	bool m_new_rgb_frame;
	bool m_new_depth_frame;
};

Freenect::Freenect freenect;
MyFreenectDevice* device;
freenect_video_format requested_format(FREENECT_VIDEO_RGB);

/////////////////////////////////////////////////////////
//
// pix_freenect
//
/////////////////////////////////////////////////////////
// Constructor
//
/////////////////////////////////////////////////////////

pix_freenect :: pix_freenect(t_float kinect_device_nr, t_float out_channel)
{ 
	post("pix_freenect 0.01 - experimental - 2011 by Matthias Kronlachner");
  m_width=640;
  m_height=480;
  x_angle = 0;
  x_led = 1;
  
  if ((int)out_channel==0)
  {
		x_channel=0; // Set Outputmode
		post("kinect output set to rgb");
	} else if ((int)out_channel==1) {
		x_channel=1; // Set Outputmode
		post("kinect output set to depth");
	} else {
		x_channel=0; // RGB Standard
		post("kinect default output rgb %i", (int)out_channel);
	}
  video_format=(freenect_video_format)0; //FREENECT_VIDEO_RGB
	depth_format=(freenect_depth_format)0; //FREENECT_DEPTH_11BIT

	post("opening Kinect device nr %i...", (int)kinect_device_nr);
	device = &freenect.createDevice<MyFreenectDevice>((int)kinect_device_nr); // CONNECT TO SPECIFIC KINECT WITH CREATION ARGUMENT
	// PD CRAHSES IF NO KINECT CONNECTED - THAT SHOULD BE CHANGED.....
  
}

/////////////////////////////////////////////////////////
// Destructor
//
/////////////////////////////////////////////////////////
pix_freenect :: ~pix_freenect()
{ 
	device->stopVideo();
	device->stopDepth();
}


/////////////////////////////////////////////////////////
// startRendering
//
/////////////////////////////////////////////////////////

uint16_t t_gamma[2048];

void pix_freenect :: startRendering(){
	
  m_image.image.xsize = m_width;
  m_image.image.ysize = m_height;
  m_image.image.csize=4; //RGBA
  //m_image.image.setCsizeByFormat(m_reqFormat);
  m_image.image.reallocate();
  
  m_rendering=true;


  device->startVideo();
	device->startDepth();

	device->setVideoFormat(video_format);
	device->setDepthFormat(depth_format);
	
  //return true;
}

/////////////////////////////////////////////////////////
// render
//
/////////////////////////////////////////////////////////
	static std::vector<uint8_t> depth(640*480*4);
	static std::vector<uint8_t> rgb(640*480*4);
	
void pix_freenect :: render(GemState *state)
{
  int size = m_image.image.xsize * m_image.image.ysize * m_image.image.csize;
  
  if (x_channel==0) //RGB OUTPUT -> Im sure there are better ways to convert from RGB to RGBA...
  {
		//post("render RGB");
		if (device->getRGB(rgb)) // True if new image
		{
			if (((int)video_format==0) || ((int)video_format==5))
			{
					int i=0;
					while (i<=size-1) {
						int num=(i%4)+(i/4)*3;
						if ((i % 4)==3)
							{
											m_image.image.data[i]=1.0; // set alpha to 1 - NOT WORKIN?!
							} else {
											m_image.image.data[i]=rgb[num];
							}
						i++;
					}
				} else if (((int)video_format==1) || ((int)video_format==2)) {
					int i=0;
					while (i<=size-1) {
						int num=i/4;
						if ((i % 4)==3)
							{
											m_image.image.data[i]=1.0; // set alpha to 1 - NOT WORKIN?!
							} else {
											m_image.image.data[i]=rgb[num];
							}
						i++;
					}
				} else if (((int)video_format==4)) {
					int i=0;
					while (i<=size-1) {
						int num=i;
						m_image.image.data[i]=rgb[num];
						i++;
					}
			} else if ((int)video_format==6) { //YUV RAW
					m_image.image.data=&rgb[0];
			}
			m_image.newimage = 1;
			m_image.image.notowned = true;
			m_image.image.upsidedown=true;
			state->set(GemState::_PIX, &m_image);
		} else {
			//post("got no new image");
			m_image.newimage = 0;
			state->set(GemState::_PIX, &m_image);
			return;
		}
	}
	
	if (x_channel==1) // DEPTH OUTPUT
	{
		//post("render DEPTH");
		if (device->getDepth(depth))
		{
			m_image.image.data = &depth[0];
			m_image.newimage = 1;
			m_image.image.notowned = true;
			m_image.image.upsidedown=true;
			state->set(GemState::_PIX, &m_image);
		} else {
			//post("got no new image");
			m_image.newimage = 0;
			state->set(GemState::_PIX, &m_image);
			return;
		}
	}


}

///////////////////////////////////////
// POSTRENDERING -> Clear
///////////////////////////////////////

void pix_freenect :: postrender(GemState *state)
{
  m_image.newimage = 0;
  state->set(GemState::_PIX, static_cast<pixBlock*>(NULL));

}

///////////////////////////////////////
// STOPRENDERING -> Stop Transfer
///////////////////////////////////////

void pix_freenect :: stopRendering(){
	device->stopVideo();
	device->stopDepth();
}

//////////////////////////////////////////
// Messages - Settings
//////////////////////////////////////////

void pix_freenect :: floatVideoModeMess (float videomode)
{
	if ( ( (int)videomode>=0 ) && ( (int)videomode<=6 ) ) 
	{
		if (((int)videomode!=3) && ((int)videomode!=4)) {
			
			video_format = (freenect_video_format)(int)videomode;
			device->setVideoFormat(video_format);
			post("Video mode set to %i", (int)videomode);
		} else {
			post("Video mode %i currently not supported", (int)videomode);
		}
	} else {
		post("Not Valid Number! (0-6)");
}
}

void pix_freenect :: floatDepthModeMess (float depthmode)
{
	if ( ( (int)depthmode>=0 ) && ( (int)depthmode<=3 ) ) 
	{
		if ( ((int)depthmode!=2) && ((int)depthmode!=3) )
		{
			depth_format = (freenect_depth_format)(int)depthmode;
			device->setDepthFormat(depth_format);
			post("Depth mode set to %i", (int)depthmode);
		} else {
			post("Depth mode %i currently not available", (int)depthmode);
		}
		
	} else {
		post("Not Valid Number! (0-3)");
	}
}

void pix_freenect :: floatChannelMess (float channel)
{
	if ((int)channel==0)
	{
		x_channel=(int)channel;
		post("Output changed to RGB (%d)", (int)channel);
	} else if ((int)channel==1) {
		x_channel=(int)channel;
		post("Output changed to DEPTH (%d)", (int)channel);
	} else {
		post("Not Valid Number!");
	}
}

void pix_freenect :: floatAngleMess (float angle)
{
  x_angle = (int)angle;
  if ( angle<-30.0 ) x_angle = -30;
  if ( angle>30.0 ) x_angle = 30;
  //post("Angle %d", x_angle);
  device->setTiltDegrees(x_angle);
}

void pix_freenect :: floatLedMess (float led)
{
	//post("LED %f", led);

  if ( ( (int)led>=0 ) && ( (int)led<=5 ) ) x_led = (int)led;
  	if (x_led == 1) {
		device->setLed(LED_GREEN);
	}
	if (x_led == 2) {
		device->setLed(LED_RED);
	}
	if (x_led == 3) {
		device->setLed(LED_YELLOW);
	}
	if (x_led == 4) {
		device->setLed(LED_BLINK_GREEN);
	}
	if (x_led == 5) {
		device->setLed(LED_BLINK_RED_YELLOW);
	}
	if (x_led == 0) {
		device->setLed(LED_OFF);
	}
}

void pix_freenect :: bangMess ()
{
	double dx,dy,dz;
	device->getState().getAccelerometers(&dx,&dy,&dz); //NOT WORKING NOW
	post("\r mks acceleration: %4f %4f %4f", dx, dy, dz);
}

/////////////////////////////////////////////////////////
// static member function
//
/////////////////////////////////////////////////////////
void pix_freenect :: obj_setupCallback(t_class *classPtr)
{
  class_addmethod(classPtr, (t_method)&pix_freenect::floatVideoModeMessCallback,
  		  gensym("video_mode"), A_FLOAT, A_NULL);
  class_addmethod(classPtr, (t_method)&pix_freenect::floatDepthModeMessCallback,
  		  gensym("depth_mode"), A_FLOAT, A_NULL);
  class_addmethod(classPtr, (t_method)&pix_freenect::floatChannelMessCallback,
  		  gensym("channel"), A_FLOAT, A_NULL);
  class_addmethod(classPtr, (t_method)&pix_freenect::floatAngleMessCallback,
  		  gensym("angle"), A_FLOAT, A_NULL);
  class_addmethod(classPtr, (t_method)&pix_freenect::floatLedMessCallback,
  		  gensym("led"), A_FLOAT, A_NULL);
  class_addmethod(classPtr, (t_method)&pix_freenect::bangMessCallback,
  		  gensym("bang"), A_NULL);
}

void pix_freenect :: floatVideoModeMessCallback(void *data, t_floatarg videomode)
{
  GetMyClass(data)->floatVideoModeMess((float)videomode);
}

void pix_freenect :: floatDepthModeMessCallback(void *data, t_floatarg depthmode)
{
  GetMyClass(data)->floatDepthModeMess((float)depthmode);
}

void pix_freenect :: floatChannelMessCallback(void *data, t_floatarg channel)
{
  GetMyClass(data)->floatChannelMess((float)channel);
}

void pix_freenect :: floatAngleMessCallback(void *data, t_floatarg angle)
{
  GetMyClass(data)->floatAngleMess((float)angle);
}

void pix_freenect :: floatLedMessCallback(void *data, t_floatarg led)
{
  GetMyClass(data)->floatLedMess((float)led);
}

void pix_freenect :: bangMessCallback(void *data)
{
  GetMyClass(data)->bangMess();
}

