/*******************************************************************************
 * FILE:        TestGrassRasterReader.java
 * DESCRIPTION:
 * NOTES:       ---
 * AUTHOR:      Antonello Andrea
 * EMAIL:       andrea.antonello@hydrologis.com
 * COMPANY:     HydroloGIS / Engineering, University of Trento / CUDAM
 * COPYRIGHT:   Copyright (C) 2005 HydroloGIS / University of Trento / CUDAM, ITALY, GPL
 *******************************************************************************
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 ******************************************************************************
 *
 * CHANGE LOG:
 *
 * version:
 * comments: changes
 * author:
 * created:
 *****************************************************************************/

package com.hydrologis.jgrass;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.FloatBuffer;

import javax.swing.JFileChooser;

import com.hydrologis.jgrass.io.GrassLocation;
import com.hydrologis.jgrass.io.GrassRasterByRowReader;
import com.hydrologis.jgrass.map.Window;

public class TestGrassRasterReader
{

  public TestGrassRasterReader(
      String cellFilePath, String outFilePath)
  {
    if (cellFilePath == null)
    {
      /*
       * ask for grass raster map path
       */
      JFileChooser fc = new JFileChooser();
      fc.showOpenDialog(null);

      File selFile = fc.getSelectedFile();
      cellFilePath = selFile.getAbsolutePath();
    }

    File outFile = null;
    if (outFilePath == null)
    {
      /*
       * ask for the ascii outputfile
       */
      JFileChooser fctoSave = new JFileChooser();
      fctoSave.showSaveDialog(null);

      outFile = fctoSave.getSelectedFile();
      outFilePath = outFile.getAbsolutePath();
    }
    BufferedWriter writeoutfile = null;
    try
    {
      writeoutfile = new BufferedWriter(new FileWriter(outFilePath));
    }
    catch (IOException e)
    {
      e.printStackTrace();
    }

    /*
     * create the grasslocation object. It holds the needed info 
     * about where everything lives in the GRASS database structure.
     */
    GrassLocation location = new GrassLocation(cellFilePath);
    /*
     * create the reader. It will be able to read the raster row, by row
     */
    GrassRasterByRowReader reader = new GrassRasterByRowReader(location, null, new Float(-9999.0),
        FloatBuffer.class);
    /*
     * open the reader. Everything is checked and the allocation table of the file is read
     */
    reader.open();

    /*
     * get the read region
     */
    Window dataWindow = reader.getDataWindow();
    try
    {
      writeoutfile.write("north: " + dataWindow.getNorth());
      writeoutfile.write("\nsouth: " + dataWindow.getSouth());
      writeoutfile.write("\nwest: " + dataWindow.getWest());
      writeoutfile.write("\neast: " + dataWindow.getEast());
      writeoutfile.write("\nns-res: " + dataWindow.getNSResolution());
      writeoutfile.write("\new-res: " + dataWindow.getWEResolution() + "\n");

      /*
       * read the map
       */
      FloatBuffer row = null;
      while (reader.hasMoreRows())
      {
        row = (FloatBuffer) reader.getNextRow();
        for (int i = 0; i < dataWindow.getCols(); i++)
        {
          writeoutfile.write(row.get() + " ");
        }
        writeoutfile.write("\n");

      }

      writeoutfile.close();
    }
    catch (IOException e)
    {
      e.printStackTrace();
    }

    /*
     * close the map handle
     */
    reader.close();
  }

  /**
   * The grass raster map is given throught the path to its cell file 
   * (even if just placeholder for fcell).
   * 
   * Output is a GRASS ascii file for demonstration purpose.
   * 
   * @param args - input file path and output file path
   */
  public static void main(String[] args)
  {
    if (args.length > 1)
      new TestGrassRasterReader(args[0], args[1]);
    else
      new TestGrassRasterReader(null, null);
  }
}
