/*******************************************************************************
 * FILE:        GrassRasterReaderByRow.java
 * DESCRIPTION:
 * NOTES:       ---
 * AUTHOR:      Antonello Andrea
 * EMAIL:       andrea.antonello@hydrologis.com
 * COMPANY:     HydroloGIS / Engineering, University of Trento / CUDAM
 * COPYRIGHT:   Copyright (C) 2005 HydroloGIS / University of Trento / CUDAM, ITALY, GPL
 *
 ******************************************************************************
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 ******************************************************************************
 *
 * CHANGE LOG:
 *
 * author:   John Preston (jpreston@uwimona.edu.jm)
 * comments: caching during reads to speed up I/O
 * created:  September 12, 2004
 *
 *****************************************************************************/
package com.hydrologis.jgrass.io;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.DoubleBuffer;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.nio.ShortBuffer;
import java.util.BitSet;
import java.util.LinkedHashMap;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.zip.DataFormatException;
import java.util.zip.Inflater;

import com.hydrologis.jgrass.io.utils.FileUtils;
import com.hydrologis.jgrass.io.utils.ShowPercent;
import com.hydrologis.jgrass.map.Window;

/**
 * This reads any native Raster map format. It supports integer,
 * float and double and the transformation of any of those
 * into int, float and double matrixes, as well as in the ByteBuffers of
 * the same tipes.
 */
public class GrassRasterByRowReader
{

  private static final String SOURCE_PROMPT = "d.rast.wizardsourceprompt.label";

  private static final String READER_ICON = "system.raster.mapbrowser.grass.icon";

  /* 0 for 1-byte integer, 1 for 2-byte integer and so on, -1 for float, -2 for double */
  private int rasterMapType = -9999;
  private int numberOfBytesPerValue = -9999;
  private int numberOfBytesPerValueInOutput = -9999;
  /* 1 for compressed, 0 for not compressed */
  private int compressed = -9999;

  private Window fileWindow = null;
  private Window dataWindow = null;

  private Object novalue = new Double(Double.NaN);

  /**
   * Comment for <code>theFilePath</code> this is the complete path including
   * the filename
   */
  //  private String theFilePath = null;
  //  private String theNullFilePath = null;
  private RandomAccessFile cellFile = null;
  private RandomAccessFile nullFile = null;
  private GrassLocation location = null;
  private String reclassPath = null;
  private int reclassFirstCategory = -1;
  private Vector reclassTable = null;

  /**
   * Comment for <code>addressesofrows</code> the pointers to access in the
   * map file the begin of every row
   */
  private long[] addressesofrows = null;

  private ByteBuffer rasterByteBuffer = null;

  private boolean moreData = false;
  private boolean hasChanged = true;
  private boolean isOldIntegerMap = false;

  //  private ByteBuffer rowCache = null;
  private int rowCacheRow = -1;
  private int firstDataRow = -1;
  private int rowindex = -1;
  /* Get a local reference to speed things up */
  private int filerows = -1;
  private double filenorth = -1;
  private double filensres = -1;
  private double datanorth = -1;
  private double datansres = -1;
  private int almightyrowgoingon = -1;

  private byte[] nullRow = null;
  /* The rowDataArray holds the unpacked row data */
  private ByteBuffer rowToReturn = null;

  private double[] range = new double[] { Double.MAX_VALUE, Double.MIN_VALUE}; // min, max

  private boolean isfloat = false;
  private boolean isdouble = false;
  private boolean isint = false;
  private boolean isshort = false;
  private double nv = 0.0;

  private Object dataObject = null;

  /** 
   * The GrassRasterByRowReader reads a GRASS binary file. At the time
   * of writing It can be an integer (old and new type), float or double map, 
   * whether compressed or non compressed.
   * 
   * Since there are many possibilities, some parameters are passed as objects
   * and then checked through the instanceof. 
   * 
   * Pay attention to the fact that if the output is of the kind double[][], the novalue
   * will have to be a Double.
   * 
   * @param location - the definition of everything needed and living in the GRASS database structure
   * @param dataWindow - the geographic window to read, that means that not the whole file 
   *                                             has to be read and also that a window bigger than the file can be read
   * @param novalue - the value to give to novalues found in the map
   * @param matrixtype - 
   * @param dataObject
   */
  public GrassRasterByRowReader(
      GrassLocation location, Window dataWindow, Object novalue, Object dataObject)
  {
    this.location = location;
    if (dataWindow != null) this.dataWindow = dataWindow;
    this.novalue = novalue;
    this.dataObject = dataObject;
  }

  /*
   * this opens checks for the file existence, sets the active and map
   * regions, defines the maptype, opens the map and extracts the header
   * to extract the rowaddresse and check if everything is alright
   */
  public boolean open()
  {
    cellFile = nullFile = null;

    if (hasChanged)
    {
      /* Test the type of the map. */
      if (rasterMapType == -9999)
      {
        if (!getRasterMapTypes())
        {
          System.out.println("cannot determine the type of map, map=" + location.getMapname());
          return false;
        }
        System.out.println("map opened successfully, datasize="
            + (rasterMapType == -2 ? "double" : rasterMapType == -1 ? "float"
                : rasterMapType == 1 ? "1 byte integer" : rasterMapType == 2 ? "2 byte integer"
                    : rasterMapType == 4 ? "4 byte integer" : "unknown size") + ", rasterMapType="
            + rasterMapType + ", numberOfBytesPerValue=" + numberOfBytesPerValue + ", compressed="
            + compressed);
      }

      if (compressed == 1)
      {
        /* Read the header of the map and the addresses of the compressed rows. */
        try
        {
          addressesofrows = getRowAddressesFromHeader(readHeader(cellFile));
        }
        catch (IOException e)
        {
          System.out.println("problems extracting the cell file header: ");
          e.printStackTrace();
          return false;
        }
      }

      /* Ok. Get ready to read data */
      moreData = true;
    }

    /*
     * was in hasmore
     */

    System.out.println("retrieving data in window: " + dataWindow.toString());

    /* Byte array that will hold a complete null row */
    nullRow = null;
    rowCacheRow = -1;
    firstDataRow = -1;
    rowindex = -1;
    /* Get a local reference to speed things up */
    filerows = fileWindow.getRows();
    filenorth = fileWindow.getNorth();
    filensres = fileWindow.getNSResolution();
    datanorth = dataWindow.getNorth();
    datansres = dataWindow.getNSResolution();
    almightyrowgoingon = 0;

    if (dataObject.equals(DoubleBuffer.class))
    {
      isdouble = true;
      numberOfBytesPerValueInOutput = 8;

    }
    /* Return data as type FloatBuffer */
    else if (dataObject.equals(FloatBuffer.class))
    {
      isfloat = true;
      numberOfBytesPerValueInOutput = 4;
    }
    /* Return data as type IntBuffer */
    else if (dataObject.equals(IntBuffer.class))
    {
      isint = true;
      numberOfBytesPerValueInOutput = 4;
    }
    else if (dataObject.equals(ShortBuffer.class))
    {
      isshort = true;
      numberOfBytesPerValueInOutput = 2;
    }

    // check the novalue
    if (novalue instanceof Double)
    {
      nv = ((Double) novalue).doubleValue();
    }
    /* Return data as type FloatBuffer */
    else if (novalue instanceof Float)
    {
      nv = ((Float) novalue).doubleValue();
    }
    /* Return data as type IntBuffer */
    else if (novalue instanceof Integer)
    {
      nv = ((Integer) novalue).doubleValue();
    }
    else if (novalue instanceof Short)
    {
      nv = ((Short) novalue).doubleValue();
    }

    return true;
  }

  public void close()
  {
    try
    {
      if (cellFile != null) cellFile.close();
      if (nullFile != null) nullFile.close();
    }
    catch (IOException e)
    {
      e.printStackTrace();
    }
  }

  /**
   * checks if there are more data to get
   */
  public boolean hasMoreRows()
  {
    if (dataWindow != null)
    {
      try
      {
        if (almightyrowgoingon >= dataWindow.getRows()) return false;
        moreData = true;

        /* allocate new rowdata */
        byte[] array = new byte[dataWindow.getCols() * numberOfBytesPerValueInOutput];
        rowToReturn = ByteBuffer.wrap(array);
        /*
         *  here we go with the progressbar
         */
        ShowPercent.getPercent((int) almightyrowgoingon, dataWindow.getRows() - 1, 1);

        /* If thread has been interrupted the return */
        //        if (currentThread.isInterrupted())
        //        {
        //          rowToReturn = null;
        //          rowCacheRow = -1;
        //          nullRow = null;
        //          moreData = false;
        //          System.gc();
        //          return moreData;
        //        }
        /* 
         * Calculate the map file row for the current data window row.
         */
        double filerow = (filenorth - (datanorth - (almightyrowgoingon * datansres))) / filensres;
        filerow = Math.floor(filerow);
        //          int filerow = (int) ((filenorth - (datanorth - (row * datansres + datansres2))) / filensres);
        //System.out.print("FILENORTH="+filenorth+", DATANORTH="+datanorth+", FILEROW="+row);
        if (filerow < 0 || filerow >= filerows)
        {
          //  System.out.println(", NULL ROW");
          /* 
           * If no data has been read yet, then increment first data row counter 
           */
          if (firstDataRow == -1) rowindex++;
          /* 
           * Write a null row to the raster buffer. To speed things up
           * the first time this is called it instantiates the buffer and
           * fills it with null values that are reused the other times. 
           */
          if (nullRow == null) nullRow = initNullRow();
          rowToReturn.put(nullRow);
          //          rasterByteBuffer.put(nullRow);
        }
        else
        {
          //  System.out.println(", DATA ROW");
          if (firstDataRow == -1) firstDataRow = rowindex + 1;
          /* Read row and put in raster buffer */
          if (filerow == rowCacheRow)
          {
            //            rasterByteBuffer.put(rowDataCache);
          }
          else
          {
            readRasterRow((int) filerow, rowToReturn);
            rowCacheRow = (int) filerow;
            //            rasterByteBuffer.put(rowDataCache);
          }
        }
        //          System.out.println("FILEROWS="+filerows+", FILEROW="+filerow+", ROWCACHEROW="+rowCacheRow+", ROW_COUNTER="+row);

        almightyrowgoingon++;

        rowCacheRow = -1;
        nullRow = null;
        rowToReturn.rewind();
        System.gc();
      }
      catch (IOException e)
      {
        moreData = false;
      }
      catch (DataFormatException e)
      {
        moreData = false;
      }
    }

    return moreData;
  }

  /**
   * returns the actual row
   */
  public Object getNextRow()
  {

    if (dataObject.equals(DoubleBuffer.class))
    {
      return rowToReturn.asDoubleBuffer();
    }
    /* Return data as type FloatBuffer */
    else if (dataObject.equals(FloatBuffer.class))
    {
      return rowToReturn.asFloatBuffer();
    }
    /* Return data as type IntBuffer */
    else if (dataObject.equals(IntBuffer.class))
    {
      return rowToReturn.asIntBuffer();
    }
    else
    {
      // Don't know what top do yet, thro some exception
      dataObject = null;
    }

    return null;
  }

  /**
   * returns the read window
   */
  public Window getDataWindow()
  {
    if (dataWindow != null) return dataWindow;
    return null;
  }

  /**
   * Reads the header part of the file into memory
   */
  private ByteBuffer readHeader(RandomAccessFile ds) throws IOException
  {

    /*
     * the first byte defines the number of bytes are used to describe 
     * the row addresses in the header (once it was sizeof(long) in grass
     * but then it was turned to an offset (that brought to reading problems 
     * in JGrass whenever the offset was != 4). 
     */
    int first = ds.read();

    ByteBuffer fileHeader = ByteBuffer.allocate(1 + first * fileWindow.getRows() + first);

    ds.seek(0);
    /* Read header */
    ds.read(fileHeader.array());

    return fileHeader;
  }

  /**
   * Extract the row addresses from the header information of the file
   */
  private long[] getRowAddressesFromHeader(ByteBuffer header)
  {
    /*
     *  Jump over the no more needed first byte (used in 
     *  readHeader to define the header size)
     */
    byte firstbyte = header.get();

    /* Read the data row addresses inside the file */
    long[] adrows = new long[fileWindow.getRows() + 1];
    if (firstbyte == 4)
    {
      for (int i = 0; i <= fileWindow.getRows(); i++)
      {
        adrows[i] = (long) header.getInt();
      }
    }
    else if (firstbyte == 8)
    {
      for (int i = 0; i <= fileWindow.getRows(); i++)
      {
        adrows[i] = header.getLong();
      }
    }
    else
    {
      // problems
    }
    return adrows;
  }

  /**
   * Determines the map type given a file and its mapset. It reads the
   * information from the header file in the cellhd directory and determines
   * the geographic limits, format of the data, etc from the file. A
   * Typical file header looks like:
   * proj:       1
   * zone:       13
   * north:      4928000
   * south:      4914000
   * east:       609000
   * west:       590000
   * cols:       950
   * rows:       700
   * e-w resol:  20
   * n-s resol:  20
   * format:     0
   * compressed: 1
   *
   * If the first line is 'reclass' then this file is a reclassified file
   * and the original data file is given by the next two lines:
   * reclass
   * name: soils
   * mapset: PERMANENT
   * #1
   * 5
   * 3
   * 8
   * ....
   * ....
   *
   * We also check the cell_misc directory to determine the raster map
   * binary format and type, the file f_format, which holds type, exists
   * only if the map is floating point map.
   */
  private boolean getRasterMapTypes()
  {
    LinkedHashMap fileMapHeader = new LinkedHashMap();
    /* Read contents of 'cellhd/name' file from the current mapset */
    String line;
    BufferedReader cellhead;
    String reclassFile = null;
    String reclassMapset = null;

    reclassTable = null;
    try
    {
      cellhead = new BufferedReader(new FileReader(location.getMapsetPath() + File.separator
          + GrassLocation.CELLHD + File.separator + location.getMapname()));
      cellhead.mark(128);
      /* Read first line to determine if file is a reclasses file.
       * If it is then open the data file and continue as per usual. */
      if ((line = cellhead.readLine()) == null) return false;
      if (line.trim().equalsIgnoreCase("reclass"))
      {
        /* The next two lines hold the orginal map file amd mapset */
        for (int i = 0; i < 2; i++)
        {
          if ((line = cellhead.readLine()) == null) return false;
          StringTokenizer tk = new StringTokenizer(line, ":");
          if (tk.countTokens() == 2)
          {
            String s = tk.nextToken();
            if (s.equalsIgnoreCase("name"))
              reclassFile = tk.nextToken().trim();
            else if (s.equalsIgnoreCase("mapset")) reclassMapset = tk.nextToken().trim();
          }
        }
        /* Instantiate the reclass table */
        reclassTable = new Vector();
        /* The next line holds the start value for categories */
        if ((line = cellhead.readLine()) == null) return false;
        if (line.charAt(0) == '#')
        {
          int reclassFirstCategory = Integer.parseInt(line.trim().substring(1));
          /* Pad reclass table until the first reclass category */
          for (int i = 0; i < reclassFirstCategory; i++)
          {
            reclassTable.addElement("");
          }
        }
        else
        {
          /* Add an empty element for the 0th category */
          reclassTable.addElement("");
        }
        /* Now read the reclass table */
        while ((line = cellhead.readLine()) != null)
        {
          reclassTable.addElement(new Integer(line));
        }
        /* Construct original data file path */
        reclassPath = location.getLocationPath() + File.separator + reclassMapset + File.separator;
        /* Test for its existence */
        cellhead = new BufferedReader(new FileReader(reclassPath + GrassLocation.CELLHD + File.separator
            + reclassFile));

        System.out.println("map is a reclassed map, original=" + reclassPath + reclassFile
            + ", reclassFirstCategory=" + reclassFirstCategory);
      }
      else
      {
        /* Push first line back onto buffered reader stack */
        cellhead.reset();
      }
      while ((line = cellhead.readLine()) != null)
      {
        StringTokenizer tok = new StringTokenizer(line, ":");
        if (tok.countTokens() == 2)
        {
          String key = tok.nextToken().trim();
          String value = tok.nextToken().trim();
          /* If key is 'ew resol' or 'ns resol' then store 'xx res' */
          if (key.indexOf("resol") != -1)
            fileMapHeader.put(key.replaceAll("resol", "res"), value);
          else
            fileMapHeader.put(key, value);
        }
      }

      /* Setup file window object that holds the geographic limits of the file data.  */
      fileWindow = null;
      if (fileMapHeader.containsKey("n-s res"))
      {
        fileWindow = new Window(Double.parseDouble((String) fileMapHeader.get("west")), Double
            .parseDouble((String) fileMapHeader.get("east")), Double.parseDouble((String) fileMapHeader
            .get("south")), Double.parseDouble((String) fileMapHeader.get("north")), Double
            .parseDouble((String) fileMapHeader.get("e-w res")), Double.parseDouble((String) fileMapHeader
            .get("n-s res")));
      }
      else if (fileMapHeader.containsKey("cols"))
      {
        fileWindow = new Window(Double.parseDouble((String) fileMapHeader.get("west")), Double
            .parseDouble((String) fileMapHeader.get("east")), Double.parseDouble((String) fileMapHeader
            .get("south")), Double.parseDouble((String) fileMapHeader.get("north")), Integer
            .parseInt((String) fileMapHeader.get("rows")), Integer.parseInt((String) fileMapHeader
            .get("cols")));
      }
      else
      {

      }
      System.out.println("map file window: " + fileWindow.toString());
      /*
       * if no datawindow was supplied, use the whole file window
       */
      if (dataWindow == null) dataWindow = fileWindow;

      if (!fileMapHeader.get("format").equals(""))
      {
        rasterMapType = new Integer((String) fileMapHeader.get("format")).intValue();
        if (rasterMapType > -1)
        {
          rasterMapType++;
          /* In Grass integers can be from 1 to 4 bytes. Jgrass will
           * convert them all directly into an intger (4-bytes) at reding and
           * decompressing time. Therefore the numberofbytespervalue
           * is always 4. */
          numberOfBytesPerValue = 4;
          /* Instantiate cell file object. */
          File ds = null;
          if (reclassPath == null)
            ds = new File(location.getMapsetPath() + File.separator + GrassLocation.CELL + File.separator
                + location.getMapname());
          else
            ds = new File(reclassPath + GrassLocation.CELL + File.separator + reclassFile);
          if (ds.exists())
          {
            cellFile = new RandomAccessFile(ds, "r");
            /* Check if null file exists. */
            nullFile = null;
            if (reclassPath == null)
              ds = new File(location.getMapsetPath() + File.separator + GrassLocation.CELL_MISC
                  + File.separator + location.getMapname() + File.separator + GrassLocation.CELLMISC_NULL);
            else
              ds = new File(reclassPath + GrassLocation.CELL_MISC + File.separator + reclassFile
                  + File.separator + GrassLocation.CELLMISC_NULL);
            if (ds.exists())
            {
              nullFile = new RandomAccessFile(ds, "r");
              if (nullFile == null)
              {
                isOldIntegerMap = false;
                System.out.println("cannot open null file: " + ds.toString());
              }
              else
              {
                isOldIntegerMap = true;
                System.out.println("opening null file: " + ds.toString());
              }
            }
          }
          else
          {
            System.out.println("integer map file doesn't exist, map=" + location.getMapname());
            return false;
          }
        }
        else if (rasterMapType < 0)
        {
          /* Read contents of 'cell_misc/name/f_format' file from the current mapset */
          File ds5 = null;
          if (reclassPath == null)
            ds5 = new File(location.getMapsetPath() + File.separator + GrassLocation.CELL_MISC
                + File.separator + location.getMapname() + File.separator + GrassLocation.CELLMISC_FORMAT);
          else
            ds5 = new File(reclassPath + GrassLocation.CELL_MISC + File.separator + reclassFile
                + File.separator + GrassLocation.CELLMISC_FORMAT);
          if (ds5.exists())
          {
            /* if the file f_format exists, then we are talking about floating maps */
            BufferedReader cellmiscformat = new BufferedReader(new FileReader(ds5));
            while ((line = cellmiscformat.readLine()) != null)
            {
              StringTokenizer tokk = new StringTokenizer(line, ":");
              if (tokk.countTokens() == 2)
              {
                String key = tokk.nextToken().trim();
                String value = tokk.nextToken().trim();
                fileMapHeader.put(key, value);
              }
            }
            //    assign the values
            if (!fileMapHeader.get("type").equals(""))
            {
              if (((String) fileMapHeader.get("type")).equalsIgnoreCase("double"))
              {
                rasterMapType = -2;
                numberOfBytesPerValue = 8;
              }
              else if (((String) fileMapHeader.get("type")).equalsIgnoreCase("float"))
              {
                rasterMapType = -1;
                numberOfBytesPerValue = 4;
              }
              else
              {
                System.out
                    .println("Type in the " + GrassLocation.CELLMISC_FORMAT + " file is not consistent");
                return false;
              }
            }
            else
            {
              System.out.println("FileSystem inconsistent. Floating type not defined");
              return false;
            }
            cellmiscformat.close();
          }
          else
          {
            System.out.println("floating point format file not found, file=" + ds5.toString());
            return false;
          }
          isOldIntegerMap = false;
          /* Instantiate cell file and null file objects */
          if (reclassPath == null)
            ds5 = new File(location.getMapsetPath() + File.separator + GrassLocation.FCELL + File.separator
                + location.getMapname());
          else
            ds5 = new File(reclassPath + GrassLocation.FCELL + File.separator + reclassFile);
          if (ds5.exists())
          {
            cellFile = new RandomAccessFile(ds5, "r");
            if (cellFile == null)
            {
              System.out.println("cannot open floating point map file, file: " + ds5.toString());
              return false;
            }
            else
            {
              System.out.println("opening floating point cell file: " + ds5.toString());
            }
            nullFile = null;
            if (reclassPath == null)
              ds5 = new File(location.getMapsetPath() + File.separator + GrassLocation.CELL_MISC
                  + File.separator + location.getMapname() + File.separator + GrassLocation.CELLMISC_NULL);
            else
              ds5 = new File(reclassPath + GrassLocation.CELL_MISC + File.separator + reclassFile
                  + File.separator + GrassLocation.CELLMISC_NULL);
            if (ds5.exists())
            {
              nullFile = new RandomAccessFile(ds5, "r");
              if (nullFile == null)
              {
                System.out.println("cannot open null file: " + ds5.toString());
              }
              else
              {
                System.out.println("opening null file: " + ds5.toString());
              }
            }
          }
          else
          {
            System.out.println("cannot open file , file=" + ds5.toString());
            return false;
          }
        }
      }
      else
      {
        System.out.println("FileSystem inconsistent. Fileformat not recognized");
        return false;
      }

      if (!fileMapHeader.get("compressed").equals(""))
      {
        compressed = new Integer((String) fileMapHeader.get("compressed")).intValue();
      }
      else
      {
        System.out.println("FileSystem inconsistent. Compression not defined");
        return false;
      }

      cellhead.close();
    }
    catch (FileNotFoundException e)
    {
      e.printStackTrace();
      return false;
    }
    catch (IOException e)
    {
      e.printStackTrace();
      return false;
    }
    return true;
  }

  /**
   * reads a row of data from the file into a byte array.
   *
   * @param currentfilerow the current row to be extracted from the file
   * @param rowToReturn the byte array to store the unpacked row data
   * @return boolean TRUE for success, FALSE for failure.
   * @throws IOException
   * @throws DataFormatException
   */
  private boolean readRasterRow(int currentfilerow, ByteBuffer rowBuffer) throws IOException,
      DataFormatException
  {
    /* Read the correct approximated row from the file. The row contents as
     * saved in a cache for along with the row number. If the row requested
     * is the row in the cache then we do not ned to read from the file. */

    //    int currentfilecol;
    boolean iscompressed = (compressed == 1 ? true : false);

    /* Data window geographic boundaries */
    double activeewres = dataWindow.getWEResolution();
    //    double activeewres2 = activeewres / 2;
    double activewest = dataWindow.getWest();

    /* Map file geographic limits */
    double filewest = fileWindow.getWest();
    double fileewres = fileWindow.getWEResolution();

    //System.out.println("currentfilerow="+currentfilerow+", fileWindow.getRows()="+fileWindow.getRows());

    /* 
     * Reset row cache and read new row data 
     * We need to have one more rowcache, because we have to clean the data from the novalues
     * 
     * Or could we check and write direct to the rowBuffer? Should be possible.
     */
    ByteBuffer rowCache = ByteBuffer.allocate(fileWindow.getCols() * ((rasterMapType == -2) ? 8 : 4));
    //    rowCache.rewind();
    getMapRow(currentfilerow, rowCache, iscompressed);
    //    rowCacheRow = currentfilerow;

    //  if the northing is inside the file boundaries, calculate the values
    //    for (double col = activewest; col < activeeast; col += activeewres)
    for (double col = 0; col < dataWindow.getCols(); col++)
    {
      /* Calculate the column value of the data to be extracted from the row */
      double x = (((activewest + (col * activeewres)) - filewest) / fileewres);
      x = Math.floor(x);

      //      currentfilecol = (int) (((activewest + (col * activeewres + activeewres2)) - filewest) / fileewres);
      if ((x < 0 || x >= fileWindow.getCols()) || readNullValueAtRowCol(currentfilerow, (int) x))
      {
        //System.out.println("COL="+col+", X="+x+", NULL VALUE(1)");
        /* Depending on the map type we store a different 'NO VALUE' value. */
        if (rasterMapType > 0)
        {
          /* For integers the NaN pattern doesn't seem to
           * so we we use the positive infinite value. */
          putValueInBuffer(rowBuffer, nv);
        }
        else if (rasterMapType == -1)
        {
          /* For floats we use the Not A Number (NAN) value. */
          putValueInBuffer(rowBuffer, nv);
        }
        else if (rasterMapType == -2)
        {
          /* For double values we use the NAN value. */
          putValueInBuffer(rowBuffer, nv);
        }
        else
        {
          /* Don't know what to do. Probably throw some exception? */
        }
      }
      else
      {
        //System.out.println("COL="+col+", X="+x+", DATA VALUE");
        rowCache.position((int) x * numberOfBytesPerValue);
        if (rasterMapType > 0)
        {
          /* Integers */
          int cell = rowCache.getInt();
          setRange((double) cell);

          /* File is an integer map file with 0 = novalue */
          if (cell == 0 && isOldIntegerMap)
          {
            putValueInBuffer(rowBuffer, nv);
          }
          else
          {
            /* If map is a reclass then get the reclassed value */
            if (reclassTable != null)
            {
              cell = ((Integer) reclassTable.elementAt(cell)).intValue();
            }
            putValueInBuffer(rowBuffer, cell);
          }
        }
        else if (rasterMapType == -1)
        {
          /* Floating point map with float values. */
          float cell = rowCache.getFloat();
          if (reclassTable != null)
          {
            cell = ((Integer) reclassTable.elementAt((int) cell)).floatValue();
          }
          setRange((double) cell);
          putValueInBuffer(rowBuffer, cell);
        }
        else if (rasterMapType == -2)
        {
          /* Floating point map with double values. */
          double cell = rowCache.getDouble();
          if (reclassTable != null)
          {
            cell = ((Integer) reclassTable.elementAt((int) cell)).doubleValue();
          }
          setRange(cell);
          putValueInBuffer(rowBuffer, cell);
        }
        else
        {
          /* Don't know what to do. Probably throw some exception? */
        }
      }
    }
    //System.out.println();

    return true;
  }

  /**
   * 
   */
  private byte[] initNullRow()
  {
    int len = dataWindow.getCols() * numberOfBytesPerValueInOutput;
    byte[] nrow = new byte[len];

    if (isint)
    {
      ByteBuffer src = ByteBuffer.allocate(4);
      src.putInt((int) nv);
      byte[] arr = src.array();
      for (int i = 0; i < len; i += 4)
        System.arraycopy(arr, 0, nrow, i, 4);
    }
    else if (isfloat)
    {
      ByteBuffer src = ByteBuffer.allocate(4);
      src.putFloat((float) nv);
      byte[] arr = src.array();
      for (int i = 0; i < len; i += 4)
        System.arraycopy(arr, 0, nrow, i, 4);
    }
    else if (isdouble)
    {
      ByteBuffer src = ByteBuffer.allocate(8);
      src.putDouble(nv);
      byte[] arr = src.array();
      for (int i = 0; i < len; i += 8)
        System.arraycopy(arr, 0, nrow, i, 8);
    }

    return nrow;
  }

  /**
   * read a row of the map from the active region
   *
   * @param currentrow
   * @param iscompressed
   * @return @throws
   *                IOException
   * @throws DataFormatException
   */
  private void getMapRow(int currentrow, ByteBuffer rowdata, boolean iscompressed) throws IOException,
      DataFormatException
  {
    //    if (logger.isDebugEnabled())
    //    {
    //      logger.debug("ACCESSING THE FILE at row: " + currentrow +
    //                   ", rasterMapType = " + rasterMapType +
    //                   ", numberOfBytesPerValue = " + numberOfBytesPerValue +
    //                   ", iscompressed = " + iscompressed);
    //    }

    if (iscompressed)
    {
      /* Compressed maps */
      if (rasterMapType == -2)
      {
        /* Compressed double map */
        readCompressedFPRowByNumber(rowdata, currentrow, addressesofrows, cellFile, numberOfBytesPerValue);
      }
      else if (rasterMapType == -1)
      {
        /* Compressed floating point map */
        readCompressedFPRowByNumber(rowdata, currentrow, addressesofrows, cellFile, numberOfBytesPerValue);
      }
      else if (rasterMapType > 0)
      {
        /* Compressed integer map */
        readCompressedIntegerRowByNumber(rowdata, currentrow, addressesofrows, cellFile);
      }
      else
      {
        System.out.println("format not double nor float");
      }
    }
    else
    {
      if (rasterMapType < 0)
      {
        /* Uncompressed floating point map */
        readUncompressedFPRowByNumber(rowdata, currentrow, cellFile, numberOfBytesPerValue);
      }
      else if (rasterMapType > 0)
      {
        /* Uncompressed integer map */
        readUncompressedIntegerRowByNumber(rowdata, currentrow, cellFile);
      }
      else
      {
        System.out.println("Unknown case, iscompressed=" + iscompressed + ", compressed=" + compressed
            + ", rasterMapType=" + rasterMapType);
      }
    }
    return;
  }

  /**
   * read a row of data from a compressed floating point map
   * 
   * @param rn
   * @param adrows
   * @param file
   * @param typeBytes
   * @return the ByteBuffer containing the data
   * @throws IOException
   * @throws DataFormatException
   */
  private void readCompressedFPRowByNumber(ByteBuffer rowdata, int rn, long[] adrows,
      RandomAccessFile thefile, int typeBytes) throws DataFormatException, IOException
  {
    int offset = (int) (adrows[rn + 1] - adrows[rn]);
    /* The fact that the file is compressed does not mean that the row is
     * compressed. If the first byte is 0 (49), then the row is compressed,
     * otherwise (first byte = 48) the row has to be read in simple XDR
     * uncompressed format. */
    byte[] tmp = new byte[offset - 1];
    thefile.seek(adrows[rn]);
    int firstbyte = (thefile.read() & 0xff);
    if (firstbyte == 49)
    {
      /* The row is compressed. */
      //      thefile.seek((long) adrows[rn] + 1);
      thefile.read(tmp, 0, offset - 1);
      Inflater decompresser = new Inflater();
      decompresser.setInput(tmp, 0, tmp.length);
      decompresser.inflate(rowdata.array());
      decompresser.end();
    }
    else if (firstbyte == 48)
    {
      /* The row is NOT compressed */
      //      thefile.seek((long) (adrows[rn]));
      //      if (thefile.read() == 48)
      //      {
      //        thefile.seek((long) (adrows[rn] + 1));
      thefile.read(rowdata.array(), 0, offset - 1);
      //      }
    }
  }

  /**
   * read a row of data from an uncompressed floating point map
   * 
   * @param rn
   * @param file
   * @param typeBytes
   * @return the ByteBuffer containing the data
   * @throws IOException
   * @throws DataFormatException
   */
  private void readUncompressedFPRowByNumber(ByteBuffer rowdata, int rn, RandomAccessFile thefile,
      int typeBytes) throws IOException, DataFormatException
  {
    int datanumber = fileWindow.getCols() * typeBytes;
    thefile.seek((long) (rn * datanumber));

    thefile.read(rowdata.array());
  }

  /**
   * read a row of data from a compressed integer point map
   * 
   * @param rn
   * @param adrows
   * @param file
   * @return the ByteBuffer containing the data
   * @throws IOException
   */
  private void readCompressedIntegerRowByNumber(ByteBuffer rowdata, int rn, long[] adrows,
      RandomAccessFile thefile) throws IOException, DataFormatException
  {
    int offset = (int) (adrows[rn + 1] - adrows[rn]);

    thefile.seek(adrows[rn]);
    /* Read how many bytes the values are
     * ex 1 => if encoded: 1 byte for the value and one byte for the count = 2
     *      2 => if encoded: 2 bytes for the value and one byte for the count = 3
     *      etc... etc */
    int bytespervalue = (thefile.read() & 0xff);
    ByteBuffer cell = ByteBuffer.allocate(bytespervalue);
    int cellValue = 0;

    /* Create the buffer in which read the compressed row  */
    byte[] tmp = new byte[offset - 1];
    thefile.read(tmp);
    ByteBuffer tmpBuffer = ByteBuffer.wrap(tmp);
    tmpBuffer.order(ByteOrder.nativeOrder());

    /* Create the buffer in which read the decompressed row. The final 
     * decompressed row will always contain 4-byte integer values */
    if ((offset - 1) == (bytespervalue * fileWindow.getCols()))
    {
      /* There is no compression in this row */
      for (int i = 0; i < offset - 1; i = i + bytespervalue)
      {
        /* Read the value */
        tmpBuffer.get(cell.array());

        /* Integers can be of 1, 2, or 4 bytes. As rasterBuffer expects
         * 4 byte integers we need to pad them with 0's. The order of
         * the padding is determined by the ByteOrder of the buffer. */
        if (bytespervalue == 1)
        {
          cellValue = (cell.get(0) & 0xff);
        }
        else if (bytespervalue == 2)
        {
          cellValue = cell.getShort(0);
        }
        else if (bytespervalue == 4)
        {
          cellValue = cell.getInt(0);
        }
        //                if (logger.isDebugEnabled()) logger.debug("tmpint=" + tmpint );
        rowdata.putInt(cellValue);
      }
    }
    else
    {
      /* If the row is compressed, then the values appear in pairs (like couples
       * a party). The couple is composed of the count and the value value
       * (WARNING: this can be more than one byte). Therefore, knowing the
       * length of the compressed row we can calculate the number of couples. */
      int couples = (offset - 1) / (1 + bytespervalue);

      for (int i = 0; i < couples; i++)
      {
        /* Read the count of values */
        int count = (int) (tmpBuffer.get() & 0xff);

        /* Read the value */
        tmpBuffer.get(cell.array());

        /* Integers can be of 1, 2, or 4 bytes. As rasterBuffer expects
         * 4 byte integers we need to pad them with 0's. The order of
         * the padding is determined by the ByteOrder of the buffer. */
        if (bytespervalue == 1)
        {
          cellValue = (cell.get(0) & 0xff);
        }
        else if (bytespervalue == 2)
        {
          cellValue = cell.getShort(0);
        }
        else if (bytespervalue == 4)
        {
          cellValue = cell.getInt(0);
        }
        /* Now write the cell value the required number
         * of times to the raster row data buffer. */
        for (int j = 0; j < count; j++)
        {
          ////                    if (logger.isDebugEnabled()) logger.debug(" " + tmpint);
          rowdata.putInt(cellValue);
        }
      }
    }
  }

  /**
   * read a row of data from an uncompressed integer map
   * @param rn
   * @param thefile
   * @return
   * @throws IOException
   * @throws DataFormatException
   */
  private void readUncompressedIntegerRowByNumber(ByteBuffer rowdata, int rn, RandomAccessFile thefile)
      throws IOException, DataFormatException
  {
    int cellValue = 0;
    ByteBuffer cell = ByteBuffer.allocate(rasterMapType);

    /* The number of bytes that are inside a row in the file. */
    int filerowsize = fileWindow.getCols() * rasterMapType;

    /* Position the file pointer to read the row */
    thefile.seek((long) (rn * filerowsize));

    /* Read the row of data from the file */
    ByteBuffer tmpBuffer = ByteBuffer.allocate(filerowsize);
    thefile.read(tmpBuffer.array());

    /* Transform the rasterMapType-size-values
     * to a standard 4 bytes integer value */
    while (tmpBuffer.hasRemaining())
    {
      // read the value
      tmpBuffer.get(cell.array());

      /* Integers can be of 1, 2, or 4 bytes. As rasterBuffer expects
       * 4 byte integers we need to pad them with 0's. The order of
       * the padding is determined by the ByteOrder of the buffer. */
      if (rasterMapType == 1)
      {
        cellValue = (cell.get(0) & 0xff);
      }
      else if (rasterMapType == 2)
      {
        cellValue = cell.getShort(0);
      }
      else if (rasterMapType == 4)
      {
        cellValue = cell.getInt(0);
      }
      //              if (logger.isDebugEnabled()) logger.debug("tmpint=" + cellValue );
      rowdata.putInt(cellValue);
    }
  }

  /**
   * read the null value from the null file (if it exists) and returns the 
   * information about the particular cell (true if it is novalue, false if 
   * it is not a novalue
   * 
   * @param mapset
   * @param currentfilerow
   * @param currentfilecol
   * @return
   */
  private boolean readNullValueAtRowCol(int currentfilerow, int currentfilecol) throws IOException
  {
    /* If the null file doesn't exist and the map is an integer, than it is 
     * an old integer-map format, where the novalues are the cells that
     * contain the values 0  */
    if (nullFile != null)
    {
      long byteperrow = (long) Math.ceil((double) fileWindow.getCols() / 8.0); // in the null map of cell_misc
      long currentByte = (long) Math.ceil((double) (currentfilecol + 1) / 8.0); // in the null map

      // currentfilerow starts from 0, so it is the row before the one we need
      long byteToRead = (byteperrow * currentfilerow) + currentByte;

      nullFile.seek(byteToRead - 1);

      int bitposition = (currentfilecol) % 8;

      byte[] thetmp = new byte[1];
      thetmp[0] = nullFile.readByte();
      BitSet tmp = FileUtils.fromByteArray(thetmp);

      boolean theBit = tmp.get(7 - bitposition);

      return theBit;

    }

    return false;

  }

  /**
   * @param tmp
   */
  private void setRange(double tmp)
  {
    //  set the range
    if (tmp < range[0] && tmp == tmp) range[0] = tmp;
    if (tmp > range[1] && tmp == tmp) range[1] = tmp;
  }

  /**
   * retrieve the range values of the map
   */
  public double[] getRange()
  {
    return range;
  }

  private void putValueInBuffer(ByteBuffer buffer, double value)
  {
    if (isfloat)
    {
      buffer.putFloat((float) value);
    }
    else if (isint)
    {
      buffer.putInt((int) value);
    }
    else if (isdouble)
    {
      buffer.putDouble(value);
    }
    else if (isshort)
    {
      buffer.putShort((short) value);
    }

  }
}
