#!/usr/bin/python

# fcn2shp
# Copyright (C) 2008 Christian Pellegrin <cripell@evolware.org> - Main author of the script
# Copyright (C) 2008 Stefano Salvador <stefano.salvador@gmail.com> - Simplified the original script
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import sys
import zipfile
import re
import operator
import os
import osgeo.osr
import osgeo.ogr
import shutil
import math
import cPickle

# il mapset contiene una chiave per ogni mappa che corrisponde
# ad un layer. Il nome della mappa/layer e' la classe di simbologia
#
# Ogni mappa a sua volta contiene:
# o -> una array di oggetti geografici
# x -> un array di liste (x, punto) ordinato per x crescente
# y -> un array di liste (y, punto) ordinato per y crescente
#
# ogni oggetto contiene:
# p -> un array di punti che lo costituiscono
# text [se testo] -> testo
# add [se testo o punto] -> eventuali informazioni addizionali
# skip -> se questo oggetto non va considerato
#
# ogni punto contiene:
# x ->
# y ->
# z ->
# t -> tipo (IR, BC)
# v -> visibile (V, I, T)
# o -> oggetto di appartenenza
# lat
# lon
# h
#

def dlist(l):
    d = dict()
    for i in l:
        d[i] = 1
    return d

def mappa_add(mappa, o):
    mappa['o'].append(o)
    for p in o['p']:
        mappa['x'].append([p['x'],p])
        mappa['y'].append([p['y'],p])

def process_dat(mapset, lines):
    o = None                            # oggetto corrente
    l = None                            # linea corrente
    nl = len(lines)
    i = 0
    while i < nl:
        if l == None:
            l = lines[i]
            i = i + 1
        #print "<%s>"%(l)

        if len(l) < 39:
            l = None
            continue

        try:
            tipo = l[0:2] + '000' + l[5:7]
            cc = l[34:36]
            p = { 'v' : l[7],
                  'x' : int(l[9:17]),
                  'y' : int(l[18:26]),
                  'z' : int(l[27:33]),
                  't' : l[37:39] }
            calculate_lat_lon(p)
        except:
            print "syntax error in line %d: <%s>"%(i, l)
            raise

        if not tipo in mapset:
            mapset[tipo] = { 'o' : [],
                             'x' : [],
                             'y' : [] }

        if cc == '11':
            t_o = { 'p' : [p] }
            p['o'] = t_o
            mappa_add(mapset[tipo], t_o)
            if i < nl:
                l = lines[i]
                i = i + 1
                if l[34:36] != '  ':
                    continue
                t_o['add'] = l.rstrip()
        elif cc == '21' or cc == '31' or cc == '41':
            if o != None:
                print "error at line %d, 21 without a 23 before"%(i)
                sys.exit(1)
            o = { 'p' : [p] }
            p['o'] = o
        elif cc == '22' or cc == '32' or cc == '42':
            if o == None:
                print "error at line %d, 22 without a 21 before"%(i)
                sys.exit(1)
            o['p'].append(p)
            p['o'] = o
        elif cc == '23' or cc == '33' or cc == '43':
            if o == None:
                print "error at line %d, 23 without a 21 before"%(i)
                sys.exit(1)
            o['p'].append(p)
            p['o'] = o
            mappa_add(mapset[tipo], o)
            o = None
        elif cc == '81':
            t_o = { 'p' : [p] }
            p['o'] = t_o
            t_o['add'] = lines[i].rstrip()
            i = i + 1
            t_o['text'] = lines[i].rstrip()
            i = i + 1
            mappa_add(mapset[tipo], t_o)
            # la seguente linea serve per certe entries errate nella 1:25000
            if i < nl:
                l = lines[i]
                i = i + 1
                if len(l) > 3 and l[0] != ' ' and l[3] != ' ':
                    continue
        else:
            print "error at line %d, unknown cc"%(i)
            sys.exit(1)

        l = None

def calculate_lat_lon(p):
      z = p['z']
      if z == 999999:
	  z = 0
      else:
	  z = z / 100.0
      p['lon'] = (p['x'] + 200000000) / 100.0
      p['lat'] = (p['y'] + 500000000) / 100.0
      p['h'] = z

def create_shapefiles(dir, mapset, cname):
    driver = osgeo.ogr.GetDriverByName("ESRI Shapefile")
    data_source = driver.CreateDataSource(dir)
    srs = osgeo.osr.SpatialReference()
    srs.ImportFromEPSG(4806) # Gauss Boaga geographic
    for mappa in mapset.keys():
        if (mappa[1] == 'A'):
            ltype = osgeo.ogr.wkbPolygon
        elif (mappa[1] == 'P' or mappa[1] == 'T'):
            ltype = osgeo.ogr.wkbPoint
        elif (mappa[1] == 'L'):
            ltype = osgeo.ogr.wkbLineString
        else:
            print "Unknow type for map %s"%(mappa)
            sys.exit(1)
        layer = data_source.CreateLayer(mappa, srs, ltype)
        field_name = osgeo.ogr.FieldDefn("Name", osgeo.ogr.OFTString)
        field_name.SetWidth(80)
        layer.CreateField(field_name)
        for o in mapset[mappa]['o']:
            feature = osgeo.ogr.Feature(layer.GetLayerDefn())
            if 'text' in o:
                feature.SetField("Name", o['text'])
            if mappa[1] == 'P' or mappa[1] == 'T':
                wkt = "POINT(%.11f %.11f)"%(o['p'][0]['lon'], o['p'][0]['lat'])
            elif mappa[1] == 'L' or mappa[1] == 'A':
                if mappa[1] == 'L':
                    hdr = "LINESTRING("
                    ftr = ")"
                else:
                    hdr = "POLYGON(("
                    ftr = "))"
                wkt = (hdr +
                       ", ".join(map(lambda p: "%.11f %.11f"%(p['lon'], p['lat']) , o['p'])) +
                       ftr)
            else:
                print "Unknow type for map %s"%(mappa)
                sys.exit(1)
            geo = osgeo.ogr.CreateGeometryFromWkt(wkt)
            feature.SetGeometry(geo)
            layer.CreateFeature(feature)
            feature.Destroy()
    data_source.Destroy()

def norm_name(ni, lm):
    # trailing and leading whitspace
    no = ni.lstrip().rstrip()
    # letters with space between
    if lm.match(no):
        no = "".join(no.split())
    return no

def adj_names(mapset):
    nome_altimetria = dlist(('1L000CD', '1L000CI', '1L000CA', '1L000DI',
                             '1L000II', '1P000PQ', '1P000AU', '9P000G1',
                             '9P000G2', '9P000G3', '9P000G4', '9P000RR',
                             '9P000CA', '9P000FI', '9P000FF', '9P000PA',
                             '9P000PN',
                             'AL000CD', 'AL000CI', 'AL000CA',
                             'IP000GM', 'IP000RR', 'IP000CA'))
    for mappa in mapset.keys():
        if mappa[1] == 'T':
            # normalizzazione semplice
            for o in mapset[mappa]['o']:
                if 'text' in o:
                    n = norm_name(o['text'],
                                  re.compile('^([^ ] +)+[^ ]$'))
                    if len(n) < 2:
                        o['skip'] = 1
        if mappa in nome_altimetria:
            for o in mapset[mappa]['o']:
                o['text'] = "%.0f"%(o['p'][0]['h'])

# main

if len(sys.argv) != 3:
    print "fcn2shp.py [input dat file] [outdir]"
    sys.exit(1)

ctr_fname = sys.argv[1]
out_dir = sys.argv[2]

z = zipfile.ZipFile(ctr_fname, "r")
filename = z.namelist()[0]
cname = filename[1:7]
out_dir = out_dir + "/" + cname + "/"
ctr_lines = z.read(filename).splitlines()

mapset = dict()

print "* Processing input file"
process_dat(mapset, ctr_lines)

print "* Adjusting names"
adj_names(mapset)

print "* Creating shapefiles"
os.mkdir(out_dir)
create_shapefiles(out_dir, mapset, cname)

print ', '.join(map(lambda i : "%s=%d"%(i, len(mapset[i]['o'])), sorted(mapset.keys())))
