-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   BaseFunctions Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------


module CHD.BaseFunctions
  (
    zipMaybe,
    foldM,
    Vector,
    vectorAdd,
    vectorSub,
    vectorLength,
    vectorDistance,
    vectorAngle,
    vectorDirection,
    vectorCreate,
    breakCut,
    uncurry3
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------


import Tcl
  (
    Coord
  )


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------


type Vector = Coord


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------


-- zipMaybe: special Function used by GuiState
--           merges two lists to list of tupels.
--           first element ist importet as type-maybe

zipMaybe :: [a] -> [b] -> [(Maybe a, b)]
zipMaybe _ [] = []
zipMaybe [] (b:bList) = (Nothing, b):(zipMaybe [] bList)
zipMaybe (a:aList) (b:bList) = (Just a, b):(zipMaybe aList bList)



-- foldM: apply a monadic function to every element of list.
--        the world maybe changed every time
--        the return is the total changed world 
--        used sometimes

foldM :: Monad m => (a -> b -> m a) -> a -> [b] -> m a
foldM _ neut [] = return neut
foldM function neut (element:list) = do
  newInit <- function neut element
  foldM function newInit list


-------------------------------------------------------------------------------
-- Vector FUNCTIONS
--        used by GuiState to calculate vector-arrows for canvas
--
-- Vector (= Tcl.Coord) is (Int, Int)
--
-- vectorAdd:       adds two vectors (v1+v2)
-- vectorSub:       subtracts a vector from another (v1-v2)
-- vectorLength:    returns length of vector
-- vectorDistance:  returns distance from p1 to p2 
-- vectorAngle:     returns angle of vector
-- vectorDirection: returns angle from p1 to p2
-- vectorCreate:    returns vector by length and angle
-- correctAngle:    reformats angle to [ 0 , 2*pi [
-------------------------------------------------------------------------------

vectorAdd	  :: Vector -> Vector -> Vector
vectorSub	  :: Vector -> Vector -> Vector
vectorLength	  :: Vector -> Float
vectorDistance	  :: Coord -> Coord -> Float
vectorAngle	  :: Vector -> Float
vectorDirection	  :: Coord -> Coord -> Float
vectorCreate	  :: Float -> Float -> Vector


vectorAdd (ax,ay) (bx,by) = (ax+bx,ay+by)

vectorSub (ax,ay) (bx,by) = (ax-bx,ay-by)

vectorLength (x,y) = sqrt ((toEnum x)^2 + (toEnum y)^2)

vectorDistance a b = vectorLength (vectorSub b a)

vectorAngle (x,y) = correctAngle $
  let xx = (toEnum x)
      yy = (toEnum y) in
  if (y < 0)
    then atan(-xx/yy)
    else if (y > 0)
      then atan(-xx/yy)+pi
      else if (x >= 0)
        then pi/2
	else pi*3/2

vectorDirection a b = vectorAngle (vectorSub b a)

vectorCreate length angle = 
  (fromEnum(length*sin(angle)),
   fromEnum(length*cos(angle+pi)))

correctAngle :: Float -> Float
correctAngle alpha =
  if (alpha < 0) 
    then correctAngle $ alpha+2*pi
    else if (alpha >= 2*pi)
      then correctAngle $ alpha-2*pi
      else alpha


breakCut :: (Char -> Bool) -> String -> (String,String)
breakCut testFct string =  
  let (arg1,arg2) = break testFct string in
  if arg2==[]
     then (arg1,arg2)
     else (arg1,(tail arg2))


uncurry3 :: (a -> b -> c -> d) -> ((a, b, c) -> d)
uncurry3 f (fst,snd,thi) = f fst snd thi
