-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   BaseTypes Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------


module CHD.BaseTypes 
  ( 
    Step(..),
    CodePosition(..),
    showShortcut,
    colorFct,
    ColorFct, 
    ThreadNo(..),
    MVarNo(..),
    ChanNo(..),
    QSemNo(..),
    QSemNNo(..),
    SampleVarNo(..),
    Object(..),
    ObjContent(..),
    ThreadStatus(..),
    ThreadAction(..),
    BreakCounter(..),
    ReturnMVar(..),
    ReturnThread(..),
    parseObject
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import qualified Control.Concurrent as C
import Char

import CHD.BaseFunctions


-------------------------------------------------------------------------------
-- CLASSES
-------------------------------------------------------------------------------

class ColorFct a
  where
  colorFct :: a -> String
  colorFct _ = "black"


-- ShowShortcut expands Show:
--   showShortcut is another show-function for shorter Versions of 
--      - ThreadNo  (TX)
--      - ChanNo (ChX)
--      - ...
class Show a => ShowShortcut a
  where
  showsShort :: a -> ShowS
  showShortcut :: a -> String

  showsShort = shows
  showShortcut x = showsShort x ""


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------

data Step = Step ThreadAction Object ObjContent Bool CodePosition
 
data CodePosition = CodePosition FilePath Int | NoPosition
  deriving (Eq,Show)

-- defining the type to differ the objects

data ThreadNo = ThreadNo Integer 
	      | GarbageCollector 
	      | UnknownThread
  deriving (Eq, Ord)
instance Enum ThreadNo where
  succ (ThreadNo i) = ThreadNo (i+1)
  pred (ThreadNo i) = ThreadNo (i-1)
  fromEnum (ThreadNo i) = fromInteger i
  toEnum i = ThreadNo (toInteger i)
instance Show ThreadNo where
  showsPrec _ (ThreadNo i) = showString "Thread" . shows i
  showsPrec _ GarbageCollector = showString "GC"
  showsPrec _ UnknownThread = showString "Thread(unknown)"
instance ShowShortcut ThreadNo where
  showsShort (ThreadNo i) = showString "T" . shows i
  showsShort GarbageCollector = showString "TGC"
  showsShort UnknownThread = showString "T()"


newtype MVarNo = MVarNo Integer
  deriving (Eq, Ord)
instance Enum MVarNo where
  succ (MVarNo i) = MVarNo (i+1)
  pred (MVarNo i) = MVarNo (i-1)
  fromEnum (MVarNo i) = fromInteger i
  toEnum i = MVarNo (toInteger i)
instance Show MVarNo where
  showsPrec _(MVarNo i) = showString "MVar" . shows i
instance ShowShortcut MVarNo where
  showsShort (MVarNo i) = showString "MV" . shows i


newtype ChanNo = ChanNo Integer
  deriving (Eq, Ord)
instance Enum ChanNo where
  succ (ChanNo i) = ChanNo (i+1)
  pred (ChanNo i) = ChanNo (i-1)
  fromEnum (ChanNo i) = fromInteger i
  toEnum i = ChanNo (toInteger i)
instance Show ChanNo where
  showsPrec _ (ChanNo i) = showString "Channel" . shows i
instance ShowShortcut ChanNo where
  showsShort (ChanNo i) = showString "Ch" . shows i


newtype QSemNo = QSemNo Integer
  deriving (Eq, Ord)
instance Enum QSemNo where
  succ (QSemNo i) = QSemNo (i+1)
  pred (QSemNo i) = QSemNo (i-1)
  fromEnum (QSemNo i) = fromInteger i
  toEnum i = QSemNo (toInteger i)
instance Show QSemNo where
  showsPrec _ (QSemNo i) = showString "QSem" . shows i
instance ShowShortcut QSemNo where
  showsShort (QSemNo i) = showString "QS" . shows i


newtype QSemNNo = QSemNNo Integer
  deriving (Eq, Ord)
instance Enum QSemNNo where
  succ (QSemNNo i) = QSemNNo (i+1)
  pred (QSemNNo i) = QSemNNo (i-1)
  fromEnum (QSemNNo i) = fromInteger i
  toEnum i = QSemNNo (toInteger i)
instance Show QSemNNo where
  showsPrec _ (QSemNNo i) = showString "QSemN" . shows i
instance ShowShortcut QSemNNo where
  showsShort (QSemNNo i) = showString "QSN" . shows i


newtype SampleVarNo = SampleVarNo Integer
  deriving (Eq, Ord)
instance Enum SampleVarNo where
  succ (SampleVarNo i) = SampleVarNo (i+1)
  pred (SampleVarNo i) = SampleVarNo (i-1)
  fromEnum (SampleVarNo i) = fromInteger i
  toEnum i = SampleVarNo (toInteger i)
instance Show SampleVarNo where
  showsPrec _ (SampleVarNo i) = showString "SampleVar" . shows i
instance ShowShortcut SampleVarNo where
  showsShort (SampleVarNo i) = showString "SV" . shows i



-- type Object composes all type together
--      this is needed to shorten the interface between GUI and CHD

data Object = NoObject
	    | ThreadObj ThreadNo 
	    | MVarObj MVarNo
	    | ChanObj ChanNo
	    | QSemObj QSemNo
	    | QSemNObj QSemNNo
	    | SampleVarObj SampleVarNo
	    | ArrowObj (Object, Object)
  deriving (Eq, Ord)
instance ColorFct Object where
  colorFct (MVarObj _) = "DarkBlue"
  colorFct (ChanObj _) = "SaddleBrown"
  colorFct (QSemObj _) = "DarkGreen"
  colorFct (QSemNObj _) = "green3"
  colorFct (SampleVarObj _) = "blue3"
  colorFct _ = "black"
instance Show Object where
  showsPrec _ (ThreadObj t) = shows t
  showsPrec _ (MVarObj m) = shows m
  showsPrec _ (ChanObj c) = shows c
  showsPrec _ (QSemObj q) = shows q
  showsPrec _ (QSemNObj q) = shows q
  showsPrec _ (SampleVarObj s) = shows s
  showsPrec _ (ArrowObj _) = showString ""
  showsPrec _ NoObject = showString "NoObject (ERROR)"


data ObjContent = NoContent
		| Quantity Int 
		| Label String
instance Show ObjContent where
  showsPrec _ (Quantity quan) = shows quan
  showsPrec _ (Label label) = (label ++)


-- just the type of state which a thread could be.

data ThreadStatus = TSRunning
		  | TSSuspend
		  | TSKilled
		  | TSFinished
		  | TSDelay
		  | TSDebug
		  | TSUnknown
  deriving (Show, Eq)
instance ColorFct ThreadStatus where
  colorFct TSRunning	       = "green3"
  colorFct TSSuspend	       = "red2"
  colorFct TSDebug	       = "yellow"
  colorFct TSDelay	       = "blue3"
  colorFct TSFinished	       = "white"
  colorFct _		       = "black"


-- just the type of action a thread could do.

data ThreadAction = TACreate
		  | TAWrite
		  | TARead
		  | TAFork
		  | TAKill
		  | TATry
		  | TASwap
		  | TARefresh
  deriving (Show, Eq)
instance ColorFct ThreadAction where
  colorFct TACreate	       = "DarkBlue"
  colorFct TAWrite	       = "green3"
  colorFct TARead	       = "red3"
  colorFct TAFork	       = "darkgrey"
  colorFct TAKill	       = "black"
  colorFct TASwap	       = "purple"
  colorFct TATry	       = "HotPink"
  --colorFct _		       = "white"


-- this is the new implementation of the break-counter
-- break everytime means the action would only run by the user
-- break notime means automatic continue for a thread
-- break waittime counts the automatic continues down until 
--                       everytime is reached.

data BreakCounter = EveryTime | WaitTime Int | NoTime
  deriving (Eq, Ord, Show)
instance Enum BreakCounter where
  succ (WaitTime i) = if (i == maxBound) 
    then NoTime 
    else WaitTime (succ i)
  succ x = x
  pred (WaitTime i) = if (i <= 1) 
    then EveryTime
    else WaitTime (pred i)
  pred x = x
  fromEnum NoTime = maxBound
  fromEnum (WaitTime i) = i
  fromEnum EveryTime = 0
  toEnum i = if (i == maxBound) 
    then NoTime
    else if (i <= 0) 
      then EveryTime 
      else (WaitTime i)


data ReturnMVar a = CHD (C.MVar a) 
		  | Gui a


data ReturnThread = Identity C.ThreadId 
		  | Number ThreadNo


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------


-- parseObject is needed to convert from string to object back.
--             used by Gui_doubleListbox and GuiInterface
--                     here the elements are strings in the listboxes...

parseObject :: String -> Object
parseObject x = 
  let (nameStr1,nameStr2) = breakCut ('>'==) x in
  if nameStr2==[]
     then let (name,numberStr) = break isDigit nameStr1 
	      number = read numberStr in
	  case name of
	       "Thread"    -> ThreadObj (ThreadNo number)
	       "MVar"	   -> MVarObj (MVarNo number)
	       "Channel"   -> ChanObj (ChanNo number)
	       "QSem"	   -> QSemObj (QSemNo number)
	       "QSemN"	   -> QSemNObj (QSemNNo number)
	       "SampleVar" -> SampleVarObj (SampleVarNo number)
	       -- _ -> NoObject  
     else (ArrowObj (parseObject nameStr1, parseObject nameStr2))

