-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   Concurrent Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------

module CHD.Control.Concurrent
  ( 
    CodePosition(..),
    module CHD.Control.Concurrent.MVar,
    module CHD.Control.Concurrent.Chan,
    module CHD.Control.Concurrent.QSem,
    module CHD.Control.Concurrent.QSemN,
    module CHD.Control.Concurrent.SampleVar,
    C.ThreadId,
    forkIO,		-- :: IO () -> IO ThreadId
    forkIOLine,		-- :: CodePosition -> IO () -> IO C.ThreadId
    forkIOLabel,	-- :: String -> IO () -> IO C.ThreadId
    forkIOLabelLine,	-- :: CodePosition -> String -> IO () -> IO C.ThreadId
    C.myThreadId,	-- :: IO ThreadId
    killThread,		-- :: ThreadId -> IO ()
    killThreadLine,	-- :: CodePosition -> ThreadId -> IO ()
    markThread,		-- :: ThreadId -> IO ()
    unmarkThread,		-- :: ThreadId -> IO ()
    --par,		-- :: a -> b -> b
    --seq,		-- :: a -> b -> b
    --fork,		-- :: a -> b -> b
    yield,		-- :: IO ()

    threadDelay,	-- :: Int -> IO ()
    threadWaitRead,	-- :: Int -> IO ()
    threadWaitWrite,	-- :: Int -> IO ()

    mergeIO,		-- :: [a]   -> [a] -> IO [a]
    nmergeIO,		-- :: [[a]] ->y IO [a]
    labelThread,	-- :: String -> IO String
    labelProgram,	-- :: String -> IO ()
    chdBreakThread	-- :: String -> IO ()
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import qualified Control.Concurrent as C

import CHD.Control.Concurrent.MVar hiding ( CodePosition(..) )
import CHD.Control.Concurrent.Chan hiding ( CodePosition(..) )
import CHD.Control.Concurrent.QSem hiding ( CodePosition(..) )
import CHD.Control.Concurrent.QSemN hiding ( CodePosition(..) )
import CHD.Control.Concurrent.SampleVar hiding ( CodePosition(..) )
import CHD.BaseTypes ( CodePosition(..) )
import CHD.DebugInterface
import CHD.DebugMsgChan
import CHD.DebugMain
import CHD.GuiMain
import CHD.GuiMsgChan
import CHD.BaseTypes


----------------------------------------------
-- not yet implemented:

import Control.Concurrent
  (
    --par,		-- :: a -> b -> b
    --seq,		-- :: a -> b -> b
    --fork,		-- :: a -> b -> b
    yield,		-- :: IO ()

    threadWaitRead,	-- :: Int -> IO ()
    threadWaitWrite,	-- :: Int -> IO ()

    mergeIO,		-- :: [a]   -> [a] -> IO [a]
    nmergeIO		-- :: [[a]] ->y IO [a]  
  )


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------

forkIO :: IO () -> IO C.ThreadId
forkIO = forkIOLine NoPosition


forkIOLine :: CodePosition -> IO () -> IO C.ThreadId
forkIOLine pos thread = do
  forkStop <- C.newEmptyMVar
  newId <- C.forkIO (do
      C.takeMVar forkStop
      debugStop1 <- sendDebugMsg ThreadStart
      C.takeMVar debugStop1
      thread
      sendDebugMsg ThreadEnd
      return ()
    )
  debugStop2 <- sendDebugMsg (ThreadForkSuspend (Identity newId) pos)
  C.takeMVar debugStop2
  debugStop3 <- sendDebugMsg (ThreadFork (Identity newId) (CHD forkStop) pos)
  C.takeMVar debugStop3
  return newId


forkIOLabel :: String -> IO () -> IO C.ThreadId
forkIOLabel name thread = 
  forkIO (do
    labelThread name
    thread
    )


forkIOLabelLine :: CodePosition -> String -> IO () -> IO C.ThreadId
forkIOLabelLine pos name thread = 
  forkIOLine pos (do
    labelThread name
    thread
    )


killThread :: C.ThreadId -> IO ()
killThread = killThreadLine NoPosition


killThreadLine :: CodePosition -> C.ThreadId -> IO ()
killThreadLine pos threadId = do
  debugStop1 <- sendDebugMsg (ThreadKillSuspend (Identity threadId) pos)
  C.takeMVar debugStop1
  C.killThread threadId
  debugStop2 <- sendDebugMsg (ThreadKill (Identity threadId) pos)
  C.takeMVar debugStop2


threadDelay :: Int -> IO ()
threadDelay time = do
  debugStop1 <- sendDebugMsg (ThreadDelaySuspend)
  C.takeMVar debugStop1
  C.threadDelay time
  debugStop2 <- sendDebugMsg (ThreadDelay)
  C.takeMVar debugStop2


markThread :: String -> C.ThreadId -> IO ()
markThread color threadId = do
  sendDebugMsg (ThreadMark (Identity threadId) color)
  return ()


unmarkThread :: C.ThreadId -> IO ()
unmarkThread threadId = do
  sendDebugMsg (ThreadUnmark (Identity threadId))
  return ()


labelThread :: String -> IO String
labelThread name = do
  oldName <- C.newEmptyMVar
  debugStop <- sendDebugMsg (ThreadLabel name oldName)
  C.takeMVar debugStop
  C.takeMVar oldName


labelProgram :: String -> IO ()
labelProgram name = do
  debugStop <- sendDebugMsg (ProgramLabel name)
  C.takeMVar debugStop
  return ()


chdBreakThread :: String -> IO ()
chdBreakThread breakCause = do
  debugStop <- sendDebugMsg (ThreadBreak breakCause)
  C.takeMVar debugStop
  return ()
