-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   QSem Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------

module CHD.Control.Concurrent.QSem
  (
    CodePosition(..),
    QSem(..),
    newQSem,		-- :: Int   -> IO QSem
    newQSemLine,	-- :: CodePosition -> Int -> IO QSem
    waitQSem,		-- :: QSem  -> IO ()
    waitQSemLine,	-- :: CodePosition -> QSem -> IO ()
    signalQSem,		-- :: QSem  -> IO ()
    signalQSemLine,	-- :: CodePosition -> QSem -> IO ()
    labelQSem,		-- :: QSem -> String -> IO String
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import qualified Control.Concurrent as C
import System.Mem.Weak

import CHD.DebugInterface
import CHD.BaseTypes


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------

data QSem = QSem QSemNo C.QSem      
instance Show QSem where
    showsPrec p (QSem number _) = shows number
--  deriving Eq


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------

newQSem :: Int -> IO QSem
newQSem = newQSemLine NoPosition


newQSemLine :: CodePosition -> Int -> IO QSem
newQSemLine pos x = do
  returnNewNoMVar <- C.newEmptyMVar
  debugStop1 <- sendDebugMsg (QSemNewSuspend (CHD returnNewNoMVar) x pos)
  C.takeMVar debugStop1
  qsem <- C.newQSem x
  qsemNo <- C.readMVar returnNewNoMVar
  debugStop2 <- sendDebugMsg (QSemNew qsemNo pos)
  C.takeMVar debugStop2
  addFinalizer qsem (do {sendDebugMsg (QSemDied qsemNo) ; return ()})
  return (QSem qsemNo qsem)


waitQSem :: QSem -> IO ()
waitQSem = waitQSemLine NoPosition


waitQSemLine :: CodePosition -> QSem  -> IO ()
waitQSemLine pos (QSem qsemNo qsem) = do
  debugStop1 <- sendDebugMsg (QSemWaitSuspend qsemNo pos) 
  C.takeMVar debugStop1
  C.waitQSem qsem
  debugStop2 <- sendDebugMsg (QSemWait qsemNo pos)
  C.takeMVar debugStop2


signalQSem :: QSem -> IO ()
signalQSem = signalQSemLine NoPosition

  
signalQSemLine :: CodePosition -> QSem  -> IO ()
signalQSemLine pos (QSem qsemNo qsem) = do
  debugStop1 <- sendDebugMsg (QSemSignalSuspend qsemNo pos)
  C.takeMVar debugStop1
  C.signalQSem qsem
  debugStop2 <- sendDebugMsg (QSemSignal qsemNo pos)
  C.takeMVar debugStop2
  

labelQSem :: QSem -> String -> IO String
labelQSem (QSem qsemNo qsem) name = do
  oldName <- C.newEmptyMVar
  debugStop <- sendDebugMsg (QSemLabel qsemNo name oldName)
  C.takeMVar debugStop
  C.takeMVar oldName
