-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   DebugMain Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------

module CHD.DebugMain
  (   
    startChd
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import CHD.GuiInterface
import CHD.GuiWindow
import CHD.DebugState
import CHD.BaseTypes
import CHD.BaseFunctions
import CHD.Environment
import CHD.PriorDoubleChannel

import qualified Control.Concurrent as C
import Data.FiniteMap


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------

startChd :: C.ThreadId -> IO ()
startChd myId = debugger (initDebuggerState myId)


-------------------------------------------------------------------------------
-- DEBUGGER message receiver (main)
--   receiving from Concurrent-functions as well as from Gui-Interface
-------------------------------------------------------------------------------

debugger :: DebuggerState -> IO ()
debugger state = do
  message <- readDblChan debugMsgChan
  case message of
    ---------------------------------------------------------------------------
    -- messages from Concurrent-functions (LESS-Channel)
    --   this is the main work for messages from the concurrent program
    ---------------------------------------------------------------------------
    Right (actorId, msg, stopper) -> do
      let actorNo = lookupThreadNo state actorId
      stopState <- stopThread state actorNo stopper
      transformState <- transformAndVisualize stopState actorNo msg
      breakState <- breakCheck transformState actorNo msg
      debugger breakState
    ---------------------------------------------------------------------------
    -- messages from Gui-Interface (PRIOR-Channel)
    --   just intercepting messages from the gui.
    --   Visualize was quite some work, since there could be the 
    --	   Visualize options-window open.
    ---------------------------------------------------------------------------
    Left msg -> do
      case msg of
	CHDQuit -> do
	  let ids = keysFM (threadMap state)
	  chdMessage("received Quit")
	  mapM_ (\threadId -> do
	          chdMessage("killing " ++ 
			     (show (lookupThreadNo state threadId)))
		  C.killThread threadId
--		  C.threadDelay 10000
		)
		ids
          myThreadId <- C.myThreadId
	  chdMessage("end")
	  C.killThread myThreadId
	Continue threadNo -> do
	  newState <- continueThread state threadNo 0
	  debugger newState
	AutoContinue threadNo unique -> do
	  newState <- continueThread state threadNo unique
	  debugger newState
	Visualize object switch -> do
	  let newState = stateSetVisualize state object switch
	  if switch
	    then do
	      case object of
	        ThreadObj tNo -> do
		  let threadState = lookupThread newState tNo 
		      status = maybe (threadStatus threadState) 
				     (\_ -> TSDebug) 
				     (threadStopper threadState)
		  sendGuiMsg $ GuiAdd object 
			  [ FillColor (colorFct status),
			    Name (threadLabel threadState) ]
		  case threadSuspend threadState of
		    Nothing -> return ()
		    Just (comObj, action) -> 
		      visualizeStep newState object
		_ -> do
		  let comObjState = lookupComObj newState object 
		      elements = map (\x -> showShortcut x) 
				     (comObjActors comObjState) 
		  sendGuiMsg $ GuiAdd object 
			  [ Elements elements,
			    Name (comObjLabel comObjState) ]
		  mapM_ 
		    (\threadNo -> 
		      case threadSuspend $ lookupThread newState threadNo of
		        Nothing -> return ()
			Just (comObj, action) ->
			  visualizeStep newState (ThreadObj threadNo))
		    (comObjSuspend comObjState)
	      sendGuiMsg $ GuiWinMessage VisualizationConfigWindow
		     $ InsertListbox EnableListbox [show object]
	      debugger newState
	    else do
	      sendPriorGuiInteract (GuiRemove object 0)
	      sendPriorGuiInteract $ InitGuiWin VisualizationConfigWindow
		       $ InsertListbox DisableListbox [show object]
	      continueState <- case object of
	        ThreadObj threadNo -> continueThread newState threadNo 0
		_ -> do
		  let suspend = comObjSuspend $ lookupComObj newState object
		  foldM
		    (\state threadNo -> continueThread state threadNo 0)
		    newState
		    suspend
	      debugger continueState
	BreakAction action time -> do
	  sendGuiMsg $ GuiWinMessage ActionBreakConfigWindow $
	    InsertListbox (case time of
	      EveryTime -> DisableListbox
	      _ -> EnableListbox) [show action]
	  debugger (stateSetActionBreak state action time)
	DisplayPosition treadName -> do
	  let actorObj = parseObject treadName
	      steps = lookupSteps state actorObj
              oldStep = if length steps > 1
			   then Just $ head (drop 1 steps)
			   else Nothing
              newStep = head steps
	  displayPosition actorObj oldStep newStep False
	  debugger state
	GetInfoGuiWin VisualizationConfigWindow -> do
	  let list = map (\object -> (object, lookupVisualize state object))
			 (lookupObjectList state)
	      (elist, dlist) = foldr (\(key,bool) (enable, disable) -> 
				       if bool 
				         then (key:enable, disable) 
					 else (enable, key:disable)) 
				     ([],[]) 
				     list
	  sendPriorGuiInteract (InitGuiWin VisualizationConfigWindow 
		     (InsertListbox DisableListbox (map show dlist)))
	  sendPriorGuiInteract (InitGuiWin VisualizationConfigWindow 
		     (InsertListbox EnableListbox (map show elist)))
	  debugger state
	GetInfoGuiWin ActionBreakConfigWindow -> do
	  let list = map (\action -> (action, lookupActionBreak state action))
			 [minBound..maxBound]
	      (elist, dlist) = foldr (\(key,time) (enable, disable) ->
				       if (time == NoTime)
				         then (key:enable, disable)
					 else (enable, key:disable))
				     ([],[])
				     list
	  sendPriorGuiInteract (InitGuiWin ActionBreakConfigWindow
		     (InsertListbox DisableListbox (map show dlist)))
	  sendPriorGuiInteract (InitGuiWin ActionBreakConfigWindow
		     (InsertListbox EnableListbox (map show elist)))
	  debugger state
	GetInfoGuiWin SourceCodeViewWindow -> do
	  let objList = filter (lookupVisualize state) 
			       (lookupThreadObjectList state)
	      stringList = map show objList
	      steps = lookupSteps state (last objList)
              oldStep = if length steps > 1
			   then Just (head $ drop 1 steps)
			   else Nothing
              newStep = head steps
	  sendPriorGuiInteract $ InitGuiWin SourceCodeViewWindow
		     $ InsertListbox SelectListbox stringList
	  displayPosition (last objList) oldStep newStep False
	  debugger state
	GetInfoGuiWin ProgressViewWindow -> do
	  let objList = filter (lookupVisualize state)
			       (lookupThreadObjectList state)
--	  mapM_ (\obj -> sendPriorGuiInteract $ InitGuiWin ProgressViewWindow
--		     $ InsertCanvas ProgressCanvas obj 
--				    (lookupProgress state obj)) objList
          return ()
	_ -> do
	  chdMessage("received unknown Message (prior):")
	  chdMessage("     '" ++ (show msg) ++ "'")
	  debugger state		


-------------------------------------------------------------------------------
-- operations THREAD-section
-------------------------------------------------------------------------------

-- this is invoked by a message from the GUI. 
-- it checks the identifier and continues the thread with the right number.

continueThread :: DebuggerState -> ThreadNo -> Int -> IO DebuggerState
continueThread state threadNo identifier = do
  let threadObj = ThreadObj threadNo
      threadState = lookupThread state threadNo
  case (threadStopper threadState) of
    Just (stopperId, stopper) -> 
      if ((identifier == 0) || (identifier == stopperId))
	then do 
	  C.putMVar stopper ()
	  if (lookupVisualize state threadObj)
	    then sendGuiMsg $ GuiPar threadObj
			[FillColor (colorFct
				 (threadStatus $ lookupThread state threadNo))]
	    else return ()
	  return (stateRemoveThreadStopper state threadNo)
	else do
	  return state
    _ -> return state


-- simply adds the mvar which a thread suspends on.

stopThread :: DebuggerState -> ThreadNo -> C.MVar () -> IO DebuggerState
stopThread state threadNo stopper = do
  return (stateAddThreadStopper state threadNo stopper)


{-
-- this is substituted due to the concept of sending a message to the gui to
--   return and say it is okay to continue
--   stopperId seems not to be needed anymore
--   verify?

autoContinueThread :: DebuggerState -> ThreadNo -> IO ()
autoContinueThread state threadNo = do
  maybe (return ())
	(\(stopperId, _) -> do
	  C.forkIO (do C.threadDelay ((autoContinueTime state) * 1000)
		       sendInteractMsg (AutoContinue threadNo stopperId))
	  return ())
	(threadStopper $ lookupThread state threadNo)
-}


-------------------------------------------------------------------------------
-- State Transformer & visualizer
-------------------------------------------------------------------------------

-- main work for the concurrent program: 
--   here every message is split up and checked what should be changed 
--     in the state
--   it controls either when the visualization-message should be send.

transformAndVisualize :: DebuggerState -> ThreadNo -> DebugMsg 
			 -> IO DebuggerState
transformAndVisualize startState actorNo msg = do
  let state = stateRemoveSuspend startState actorNo
      actorObj = ThreadObj actorNo
      visualizeActor = lookupVisualize state actorObj

-- submit color to Gui if needed.
  if visualizeActor 
    then do 
      let actionMsg = case msg of 
	    ThreadBreak cause -> show (reduceDebugMsg msg) ++ " (" ++ cause 
				 ++ ")"
	    _ -> show (reduceDebugMsg msg)
      sendGuiMsg (GuiPar actorObj [ -- FillColor (colorFct TSDebug),
				Action actionMsg ])
    else return ()

  case msg of

-- programlabel: submit it to Gui
    ProgramStart (Identity newId) -> do
      let (addState, threadNo) = stateAddThread state newId
	  threadObj = ThreadObj threadNo
	  step = Step TAFork threadObj NoContent False NoPosition
          threadState = (lookupThread state threadNo) {
	    threadStatus = TSRunning,
	    threadParent = actorNo,
	    threadBreak = NoTime
	    }
	  statState = stateSetThread addState threadNo threadState
	  newState' = stateAddStep statState actorObj step
      visualizeStep newState' actorObj
      return newState'
    ProgramLabel name -> do
      sendGuiMsg (GuiTitle name)
      return state
	-----------------------------------------------------------------------
	-- messages from CHD.ConcurrentMain
	-----------------------------------------------------------------------

-- fork: add new thread to state-data and submit to Gui
    ThreadForkSuspend (Identity newId) codePos -> do
      let (addState, threadNo) = stateAddThread state newId
	  threadObj = ThreadObj threadNo
	  step = Step TAFork threadObj NoContent True codePos
	  newState = stateSetSuspend addState actorNo threadObj TAFork
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return newState'
    ThreadFork (Identity threadId) (CHD mvar) codePos -> do
      let threadNo = lookupThreadNo state threadId
	  threadObj = ThreadObj threadNo
	  step = Step TAFork threadObj NoContent False codePos
	  threadState = (lookupThread state threadNo) {
	    threadStatus = TSRunning,
	    threadParent = actorNo,
	    threadBreak = NoTime
-- (if (autoContinueOption == 1) then NoTime else EveryTime) 
	    }
	  statState = stateSetThread state threadNo threadState
	  newState' = stateAddStep statState actorObj step
      visualizeStep newState' actorObj
--  sendGuiMsg $ GuiWinMessage ProgressViewWindow 
--	       $ InsertCanvas ProgressCanvas actorObj 
--			      [(codePos,threadObj,TAFork,False)]
      sendGuiMsg $ GuiWinMessage VisualizationConfigWindow 
	       $ InsertListbox EnableListbox [show threadObj]
      C.putMVar mvar ()
      return newState'
    ThreadKillSuspend (Identity threadId) codePos -> do
      let threadNo = lookupThreadNo state threadId
	  threadObj = ThreadObj threadNo
	  step = Step TAKill threadObj NoContent True codePos
	  newState' = stateAddStep state actorObj step
      visualizeStep newState' actorObj
      return newState'
    ThreadKill (Identity threadId) codePos -> do
      let threadNo = lookupThreadNo state threadId
	  threadObj = ThreadObj threadNo
	  step = Step TAKill threadObj NoContent False codePos
	  newState' = stateAddStep state actorObj step
      visualizeStep newState' actorObj
      if (lookupVisualize newState' threadObj) 
        then do
	  sendGuiMsg $ GuiWinMessage VisualizationConfigWindow 
		 $ RemoveListbox EnableListbox [show threadObj]
	  sendGuiMsg (GuiDel threadObj)
	else sendGuiMsg $ GuiWinMessage VisualizationConfigWindow 
		    $ RemoveListbox DisableListbox [show threadObj]
--      let (newState', threadNo) = stateRemoveThread newState threadId
      return (stateSetThreadStatus newState' threadNo TSKilled)
    ThreadStart ->
      return state
    ThreadEnd -> do
      if visualizeActor 
        then do
	  sendGuiMsg $ GuiWinMessage VisualizationConfigWindow 
		 $ RemoveListbox EnableListbox [show actorObj]
	  sendGuiMsg (GuiDel actorObj)
	else sendGuiMsg $ GuiWinMessage VisualizationConfigWindow 
		    $ RemoveListbox DisableListbox [show actorObj]
--      let (newState, threadNo) = stateRemoveThread state actorId
      return (stateSetThreadStatus state actorNo TSFinished)
    ThreadDelaySuspend -> do
      return (stateSetThreadStatus state actorNo TSDelay)
    ThreadDelay -> do
      return (stateSetThreadStatus state actorNo TSRunning)
    ThreadLabel name oldName -> do
      C.putMVar oldName (lookupLabel state actorObj)
      let label = (if (name == "") 
            then (show actorNo) 
	    else ((showShortcut actorNo) ++ ":" ++ name))
      if visualizeActor
        then sendGuiMsg $ GuiPar actorObj [Name label]
	else return ()
      return $ stateSetLabel state actorObj label 
    ThreadMark (Identity threadId) color -> do    
      let threadNo = lookupThreadNo state threadId
	  threadObj = ThreadObj threadNo
      sendGuiMsg $ GuiPar threadObj [Thickness 2,OutlineColor color]
      return state
    ThreadUnmark (Identity threadId) -> do    
      let threadNo = lookupThreadNo state threadId
	  threadObj = ThreadObj threadNo
      sendGuiMsg $ GuiPar threadObj [Thickness 1,OutlineColor "black"]
      return state


----------------------------------------------------------------------
-- messages from CHD.ConcurrentMVar
----------------------------------------------------------------------

    MVarNewEmptySuspend (CHD returnNewNoMVar) codePos -> do
      let (addState, mvarNo) = stateAddMVar state
	  mvarObj = MVarObj mvarNo
	  step = Step TACreate mvarObj NoContent True codePos 
	  newState = stateSetSuspend addState actorNo mvarObj TACreate
	  newState' = stateAddStep newState actorObj step
      C.putMVar returnNewNoMVar mvarNo
      visualizeStep newState' actorObj
      return newState'
    MVarNewEmpty mvarNo codePos -> do
      let mvarObj = MVarObj mvarNo
	  step = Step TACreate mvarObj NoContent False codePos 
	  newState' = stateAddStep state actorObj step
      visualizeStep newState' actorObj
      sendGuiMsg $ GuiWinMessage VisualizationConfigWindow
		 $ InsertListbox EnableListbox [show mvarObj]          
      return newState'
    MVarNewSuspend (CHD returnNewNoMVar) label codePos -> do
      let (addState, mvarNo) = stateAddMVar state
	  mvarObj = MVarObj mvarNo
	  step = Step TACreate mvarObj (Label label) True codePos
	  newState = stateSetSuspend addState actorNo mvarObj TACreate
	  newState' = stateAddStep newState actorObj step
      C.putMVar returnNewNoMVar mvarNo
      visualizeStep newState' actorObj
      return newState'
    MVarNew mvarNo label codePos -> do
      let mvarObj = MVarObj mvarNo
	  step = Step TACreate mvarObj (Label label) False codePos 
	  newState = stateMVarSetContents state mvarNo actorNo label
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      sendGuiMsg $ GuiWinMessage VisualizationConfigWindow
	     $ InsertListbox EnableListbox [show mvarObj]
      return newState'
    MVarTakeSuspend mvarNo codePos -> do
      let mvarObj = MVarObj mvarNo
	  step = Step TARead mvarObj NoContent True codePos
	  newState = stateSetSuspend state actorNo mvarObj TARead
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return (stateSetThreadStatus newState' actorNo TSSuspend)
    MVarTake mvarNo codePos -> do
      let mvarObj = MVarObj mvarNo
	  step = Step TARead  mvarObj NoContent False codePos
	  newState = stateMVarRemoveContents state mvarNo
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return (stateSetThreadStatus newState' actorNo TSRunning)
    MVarReadSuspend mvarNo codePos -> do
      let mvarObj = MVarObj mvarNo
	  step = Step TARead mvarObj NoContent True codePos
	  newState = stateSetSuspend state actorNo mvarObj TARead
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return (stateSetThreadStatus newState' actorNo TSSuspend)
    MVarRead mvarNo codePos -> do
      let mvarObj = MVarObj mvarNo
	  step = Step TARead mvarObj NoContent False codePos
	  newState' = stateAddStep state actorObj step
      visualizeStep newState' actorObj
      return (stateSetThreadStatus newState' actorNo TSRunning)
    MVarPutSuspend mvarNo label codePos -> do
      let mvarObj = MVarObj mvarNo
	  step = Step TAWrite mvarObj (Label label) True codePos
	  newState = stateSetSuspend state actorNo mvarObj TAWrite
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return (stateSetThreadStatus newState' actorNo TSSuspend)
    MVarPut mvarNo label codePos -> do
      let mvarObj = MVarObj mvarNo
	  step = Step TAWrite mvarObj (Label label) False codePos
	  newState = stateMVarSetContents state mvarNo actorNo label
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return (stateSetThreadStatus newState' actorNo TSRunning)
    MVarSwapSuspend mvarNo codePos -> do
      let mvarObj = MVarObj mvarNo
	  step = Step TASwap mvarObj NoContent True codePos
	  newState = stateSetSuspend state actorNo mvarObj TASwap
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return (stateSetThreadStatus newState' actorNo TSSuspend)
    MVarSwap mvarNo label codePos -> do
      let mvarObj = MVarObj mvarNo
	  step = Step TASwap mvarObj (Label label) False codePos
	  newState = stateMVarSetContents state mvarNo actorNo label
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return newState' 
    MVarTryTakeSuspend mvarNo codePos -> do
      let mvarObj = MVarObj mvarNo
	  step = Step TATry mvarObj NoContent True codePos
	  newState = stateSetSuspend state actorNo mvarObj TATry
	  newState' = stateAddStep newState actorObj step		   
      visualizeStep newState' actorObj
      return newState'
    MVarTryPutSuspend mvarNo codePos -> do
      let mvarObj = MVarObj mvarNo
	  step = Step TATry mvarObj NoContent True codePos
	  newState = stateSetSuspend state actorNo mvarObj TATry
	  newState' = stateAddStep newState actorObj step   
      visualizeStep newState' actorObj
      return newState'
    MVarTry mvarNo codePos -> do
      let mvarObj = MVarObj mvarNo
	  step = Step TATry mvarObj NoContent False codePos
	  newState' = stateAddStep state actorObj step
      visualizeStep newState' actorObj
      return newState'
    MVarDied mvarNo -> do
      let mvarObj = MVarObj mvarNo          
      if lookupVisualize state mvarObj
        then do
	  sendGuiMsg $ GuiWinMessage VisualizationConfigWindow 
		 $ RemoveListbox EnableListbox [show mvarObj]
	  sendGuiMsg (GuiDel mvarObj)
	else sendGuiMsg $ GuiWinMessage VisualizationConfigWindow 
		    $ RemoveListbox DisableListbox [show mvarObj]
      return (stateRemoveComObj state mvarObj)
    MVarLabel mvarNo name oldName -> do
      let mvarObj = MVarObj mvarNo 
      C.putMVar oldName (lookupLabel state mvarObj)
      let label = (if (name == "") 
            then (show mvarNo)
            else ((showShortcut mvarNo) ++ ":" ++ name))
      if lookupVisualize state mvarObj
        then sendGuiMsg $ GuiAdd mvarObj [Name label]
	else return ()
      return $ stateSetLabel state mvarObj label


-----------------------------------------------------------------------
-- messages from CHD.ConcurrentChannel
-----------------------------------------------------------------------
    
    ChanNewSuspend (CHD chanNoMVar) codePos -> do
      let (addState, chanNo) = stateAddChannel state
	  step = Step TACreate NoObject NoContent True codePos
	  chanObj = ChanObj chanNo
	  newState = stateSetSuspend addState actorNo chanObj TACreate
	  newState' = stateAddStep newState actorObj step
      C.putMVar chanNoMVar chanNo
      visualizeStep newState' actorObj
      sendGuiMsg $ GuiWinMessage VisualizationConfigWindow
	     $ InsertListbox EnableListbox [show chanObj]
      return newState'
    ChanNew chanNo codePos -> do
      let chanObj = ChanObj chanNo
	  step = Step TACreate chanObj NoContent False codePos
	  newState' = stateAddStep state actorObj step
      visualizeStep newState' actorObj
      sendGuiMsg $ GuiWinMessage VisualizationConfigWindow
	     $ InsertListbox EnableListbox [show chanObj]
      return newState'
    ChanWriteSuspend chanNo codePos -> do
      let chanObj = ChanObj chanNo
	  step = Step TAWrite chanObj NoContent True codePos
	  newState = stateSetSuspend state actorNo chanObj TAWrite
	  newState' = stateAddStep newState actorObj step		   
      visualizeStep newState' actorObj
      return newState'
    ChanWrite chanNo label codePos -> do
      let chanObj = ChanObj chanNo
	  step = Step TAWrite chanObj (Label label) False codePos
	  newState = stateChannelAddMessage state chanNo actorNo label
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return newState'
    ChanReadSuspend chanNo codePos -> do
      let chanObj = ChanObj chanNo
	  step = Step TARead chanObj NoContent True codePos
	  newState = stateSetSuspend state actorNo chanObj TARead
	  newState' = stateAddStep newState actorObj step	   
      visualizeStep newState' actorObj
      return (stateSetThreadStatus newState' actorNo TSSuspend)
    ChanRead chanNo codePos -> do
      let chanObj = ChanObj chanNo
	  step = Step TARead chanObj NoContent False codePos
	  newState = stateChannelRemoveMessage state chanNo
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj 
      return (stateSetThreadStatus newState' actorNo TSRunning)
    ChanDied chanNo -> do
      let chanObj = ChanObj chanNo
      if lookupVisualize state chanObj
        then do
	  sendGuiMsg $ GuiWinMessage VisualizationConfigWindow 
		 $ RemoveListbox EnableListbox [show chanObj]
	  sendGuiMsg (GuiDel chanObj)
	else sendGuiMsg $ GuiWinMessage VisualizationConfigWindow 
		    $ RemoveListbox DisableListbox [show chanObj]
      return (stateRemoveComObj state (ChanObj chanNo))
    ChanLabel chanNo name oldName -> do
      let chanObj = ChanObj chanNo
      C.putMVar oldName (lookupLabel state chanObj)
      let label = (if (name == "")
            then (show chanNo)
	    else ((showShortcut chanNo) ++ ":" ++ name))
      if lookupVisualize state chanObj
        then sendGuiMsg $ GuiAdd chanObj [Name label]
	else return ()
      return $ stateSetLabel state chanObj label


-----------------------------------------------------------------------
-- messages from CHD.ConcurrentQSem
-----------------------------------------------------------------------
    
    QSemNewSuspend (CHD returnNewNoMVar) quan codePos -> do
      let (addState, qsemNo) = stateAddQSem state quan
	  qsemObj = QSemObj qsemNo
	  step = Step TACreate qsemObj (Quantity quan) True codePos
	  newState = stateSetSuspend addState actorNo qsemObj TACreate
	  newState' = stateAddStep newState actorObj step
      C.putMVar returnNewNoMVar qsemNo
      visualizeStep newState' actorObj
      return newState'
    QSemNew qsemNo codePos -> do
      let qsemObj = QSemObj qsemNo
	  step = Step TACreate qsemObj NoContent False codePos
	  newState' = stateAddStep state actorObj step
      visualizeStep newState' actorObj
      sendGuiMsg $ GuiWinMessage VisualizationConfigWindow
	     $ InsertListbox EnableListbox [show qsemObj]
      return newState'
    QSemWaitSuspend qsemNo codePos -> do
      let qsemObj = QSemObj qsemNo
	  step = Step TARead qsemObj NoContent True codePos
	  newState = stateSetSuspend state actorNo qsemObj TARead
	  newState' = stateAddStep newState actorObj step		   
      visualizeStep newState' actorObj
      return (stateSetThreadStatus newState' actorNo TSSuspend)
    QSemWait qsemNo codePos -> do
      let qsemObj = QSemObj qsemNo
	  step = Step TARead qsemObj NoContent False codePos 
	  newState = stateDecQuantity state qsemObj 1
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return (stateSetThreadStatus newState' actorNo TSRunning)
    QSemSignalSuspend qsemNo codePos -> do
      let qsemObj = QSemObj qsemNo
	  step = Step TAWrite qsemObj NoContent True codePos
	  newState = stateSetSuspend state actorNo qsemObj TAWrite
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return newState'
    QSemSignal qsemNo codePos -> do
      let qsemObj = QSemObj qsemNo
	  step = Step TAWrite qsemObj NoContent False codePos
	  newState = stateIncQuantity state qsemObj 1
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return newState'
    QSemLabel qsemNo name oldName -> do
      let qsemObj = QSemObj qsemNo 
      C.putMVar oldName (lookupLabel state qsemObj)
      let label = (if (name == "") 
            then (show qsemNo)
            else ((showShortcut qsemNo) ++ ":" ++ name))
      if lookupVisualize state qsemObj
        then sendGuiMsg $ GuiAdd qsemObj [Name label]
	else return ()
      return $ stateSetLabel state qsemObj label


-----------------------------------------------------------------------
-- messages from CHD.ConcurrentQSemN
-----------------------------------------------------------------------

    QSemNNewSuspend (CHD returnNewNoMVar) quan codePos -> do
      let (addState, qsemnNo) = stateAddQSemN state quan
	  step = Step TACreate qsemnObj (Quantity quan) True codePos
	  qsemnObj = QSemNObj qsemnNo
	  newState = stateSetSuspend addState actorNo qsemnObj TACreate
	  newState' = stateAddStep newState actorObj step
      C.putMVar returnNewNoMVar qsemnNo
      visualizeStep newState' actorObj
      sendGuiMsg $ GuiWinMessage VisualizationConfigWindow
	     $ InsertListbox EnableListbox [show qsemnObj]
      return newState'
    QSemNNew qsemnNo quan codePos -> do
      let qsemnObj = QSemNObj qsemnNo
	  step = Step TACreate qsemnObj (Quantity quan) False codePos
	  newState' = stateAddStep state actorObj step		   
      visualizeStep newState' actorObj
      sendGuiMsg $ GuiWinMessage VisualizationConfigWindow
	     $ InsertListbox EnableListbox [show qsemnObj]
      return newState'
    QSemNWaitSuspend qsemnNo quan codePos -> do
      let qsemnObj = QSemNObj qsemnNo
	  step = Step TARead qsemnObj (Quantity quan) True codePos
	  newState = stateSetSuspend state actorNo qsemnObj TARead
	  newState' = stateAddStep newState actorObj step	   
      visualizeStep newState' actorObj
      return (stateSetThreadStatus newState' actorNo TSSuspend)
    QSemNWait qsemnNo quan codePos -> do
      let qsemnObj = QSemNObj qsemnNo
	  step = Step TARead qsemnObj (Quantity quan) False codePos
          newState = stateDecQuantity state qsemnObj quan
	  newState' = stateAddStep newState actorObj step	   
      visualizeStep newState' actorObj
      return (stateSetThreadStatus newState' actorNo TSRunning)
    QSemNSignalSuspend qsemnNo quan codePos -> do
      let qsemnObj = QSemNObj qsemnNo
	  step = Step TAWrite qsemnObj (Quantity quan) True codePos
	  newState = stateSetSuspend state actorNo qsemnObj TAWrite
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return newState'
    QSemNSignal qsemnNo quan codePos -> do
      let qsemnObj = QSemNObj qsemnNo
	  step = Step TAWrite qsemnObj (Quantity quan) False codePos
	  newState = stateIncQuantity state qsemnObj quan
	  newState' = stateAddStep newState actorObj step		   
      visualizeStep newState' actorObj
      return newState'
    QSemNLabel qsemnNo name oldName -> do
      let qsemnObj = QSemNObj qsemnNo 
      C.putMVar oldName (lookupLabel state qsemnObj)
      let label = (if (name == "") 
            then (show qsemnNo)
            else ((showShortcut qsemnNo) ++ ":" ++ name))
      if lookupVisualize state qsemnObj
        then sendGuiMsg $ GuiAdd qsemnObj [Name label]
	else return ()
      return $ stateSetLabel state qsemnObj label
   

-----------------------------------------------------------------------
-- messages from CHD.ConcurrentSampleVar
-----------------------------------------------------------------------
    
    SampleVarNewEmptySuspend (CHD returnNewNoMVar) codePos -> do
      let (addState, samplevarNo) = stateAddSampleVar state
	  samplevarObj = SampleVarObj samplevarNo
	  step = Step TACreate samplevarObj NoContent True codePos
	  newState = stateSetSuspend addState actorNo samplevarObj TACreate
	  newState' = stateAddStep newState actorObj step
      C.putMVar returnNewNoMVar samplevarNo
      visualizeStep newState' actorObj
      return newState'
    SampleVarNewEmpty samplevarNo codePos -> do
      let samplevarObj = SampleVarObj samplevarNo
	  step = Step TACreate samplevarObj NoContent False codePos
	  newState' = stateAddStep state actorObj step
      visualizeStep newState' actorObj
      sendGuiMsg $ GuiWinMessage VisualizationConfigWindow
	     $ InsertListbox EnableListbox [show samplevarObj]
      return newState'
    SampleVarNewSuspend (CHD returnNewNoMVar) codePos -> do
      let (addState, samplevarNo) = stateAddSampleVar state
	  samplevarObj = SampleVarObj samplevarNo
	  step = Step TACreate samplevarObj NoContent True codePos
	  newState = stateSetSuspend addState actorNo samplevarObj TACreate
	  newState' = stateAddStep newState actorObj step
      C.putMVar returnNewNoMVar samplevarNo
      visualizeStep newState actorObj
      return newState'
    SampleVarNew samplevarNo label codePos -> do
      let samplevarObj = SampleVarObj samplevarNo
	  step = Step TACreate samplevarObj (Label label) False codePos
	  newState = stateSampleVarSetContents state samplevarNo actorNo label
	  newState' = stateAddStep newState actorObj step	   
      visualizeStep newState actorObj
      sendGuiMsg $ GuiWinMessage VisualizationConfigWindow
	     $ InsertListbox EnableListbox [show samplevarObj]
      return newState'
    SampleVarWriteSuspend samplevarNo codePos -> do
      let samplevarObj = SampleVarObj samplevarNo
	  step = Step TAWrite samplevarObj NoContent True codePos
	  newState = stateSetSuspend state actorNo samplevarObj TAWrite
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState actorObj
      return newState'
    SampleVarWrite samplevarNo label codePos -> do
      let samplevarObj = (SampleVarObj samplevarNo)
	  step = Step TAWrite samplevarObj (Label label) False codePos
	  newState = stateSampleVarSetContents state samplevarNo actorNo label
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return newState'
    SampleVarReadSuspend samplevarNo codePos -> do
      let samplevarObj = SampleVarObj samplevarNo
	  step = Step TARead samplevarObj NoContent True codePos
	  newState = stateSetSuspend state actorNo samplevarObj TARead
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return (stateSetThreadStatus newState' actorNo TSSuspend)
    SampleVarRead samplevarNo codePos -> do
      let samplevarObj = SampleVarObj samplevarNo
	  step = Step TARead samplevarObj NoContent False codePos
	  newState = stateSampleVarRemoveContents state samplevarNo
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return (stateSetThreadStatus newState' actorNo TSRunning)
    SampleVarEmptySuspend samplevarNo codePos -> do
      let samplevarObj = SampleVarObj samplevarNo
	  step = Step TAWrite samplevarObj NoContent True codePos
	  newState = stateSetSuspend state actorNo samplevarObj TAWrite
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return newState'
    SampleVarEmpty samplevarNo codePos -> do
      let samplevarObj = SampleVarObj samplevarNo
	  step = Step TAWrite samplevarObj NoContent False codePos
	  newState = stateSampleVarRemoveContents state samplevarNo
	  newState' = stateAddStep newState actorObj step
      visualizeStep newState' actorObj
      return newState'
    SampleVarDied samplevarNo -> do
      let samplevarObj = SampleVarObj samplevarNo 
      if lookupVisualize state samplevarObj
        then do
	  sendGuiMsg $ GuiWinMessage VisualizationConfigWindow 
		 $ RemoveListbox EnableListbox [show samplevarObj]
	  sendGuiMsg (GuiDel samplevarObj)
	else sendGuiMsg $ GuiWinMessage VisualizationConfigWindow 
		    $ RemoveListbox DisableListbox [show samplevarObj]
      return (stateRemoveComObj state samplevarObj)
    SampleVarLabel samplevarNo name oldName -> do
      let samplevarObj = SampleVarObj samplevarNo 
      C.putMVar oldName (lookupLabel state samplevarObj)
      let label = (if (name == "") 
            then (show samplevarNo)
            else ((showShortcut samplevarNo) ++ ":" ++ name))
      if lookupVisualize state samplevarObj
        then sendGuiMsg $ GuiAdd samplevarObj [Name label]
	else return ()
      return $ stateSetLabel state samplevarObj label   

    _ -> return state


-------------------------------------------------------------------------------
-- Visualize Action
-------------------------------------------------------------------------------

displayPosition :: Object -> Maybe Step -> Step -> Bool -> IO ()
displayPosition actorObj oldStep (Step action comObj _ suspend codePos) 
		insert = do
  case oldStep of
       Nothing -> return ()
       Just (Step _ _ _ _ NoPosition) -> return ()
       Just (Step _ _ _ _ (CodePosition _ lastLine)) -> 
	      	   sendGuiMsg $ GuiWinMessage SourceCodeViewWindow
			      $ UnmarkEdit DisplayEdit lastLine
  case (action,suspend,insert) of
       (TAFork,False,True) ->
		   sendGuiMsg $ GuiWinMessage SourceCodeViewWindow
			     $ InsertListbox SelectListbox [show comObj]
       (TAKill,False,True) ->
		   sendGuiMsg $ GuiWinMessage SourceCodeViewWindow
			     $ RemoveListbox SelectListbox [show comObj]
       _ -> return ()
  case codePos of
       CodePosition file line ->
	   do
	   sendGuiMsg $ GuiWinMessage SourceCodeViewWindow
		      $ Select SelectListbox file (show actorObj)
	   sendGuiMsg $ GuiWinMessage SourceCodeViewWindow
		      $ MarkEdit DisplayEdit line (colorFct action) "white"
       NoPosition -> return ()


-- checks what should be visualized and what message should be send to GUI

-- visualizeStep
visualizeStep :: DebuggerState -> Object -> IO ()
visualizeStep state actorObj =
  let steps = lookupSteps state actorObj
      oldStep = if length steps > 1
		   then Just (head $ drop 1 steps)
		   else Nothing
      newStep@(Step action comObj content suspend _) = head steps
      visualizeComObj = lookupVisualize state comObj
      visualizeActorObj = lookupVisualize state actorObj
      arrow = case action of
        TARead   -> ArrowObj (comObj, actorObj)
	TAWrite  -> ArrowObj (actorObj, comObj)
	TAFork	 -> ArrowObj (actorObj, comObj)
	_	 -> ArrowObj (actorObj, comObj)
      parameter = case (action, actorObj) of
        (TACreate, ThreadObj actorNo) -> [Parent actorNo]
	(TAFork, ThreadObj actorNo)   -> [Parent actorNo]
	_			      -> []
      color = colorFct comObj
      contentFct content =
	case content of
	     Label l	    -> []
	     Quantity (x+1) -> [FillColor color,
				OutlineColor color]
	     _		    -> [FillColor "DarkRed",
				OutlineColor "DarkRed"]
      actors = lookupActors state comObj
      contents = lookupContents state comObj
      colors = foldr (++) [] $ map contentFct contents
      elements = case comObj of
	QSemObj  _ -> map show contents
	QSemNObj _ -> map show contents
	_ -> case contents of
		  [] -> map (\x -> showShortcut x) actors
		  _  -> zipWith (\x y-> y ++ " : " ++ show x) contents
				(map (\x -> showShortcut x) actors) 
      arrowName = case content of
        NoContent -> []
	Label "" -> []
	Label label -> [Name label]
	Quantity quan -> [Name (show quan)] in
  do
  if visualizeComObj
    then do
      displayPosition actorObj oldStep newStep True
      if suspend
        then return () -- sendGuiMsg (GuiAdd comObj parameter) : is eh leer.
	else if action==TAFork
	     then sendGuiMsg (GuiAdd comObj parameter)
	     else sendGuiMsg (GuiAdd comObj ((Elements elements:colors)
					     ++ parameter))	     
      if visualizeActorObj
        then if suspend
	     then if action==TAFork || action==TACreate
		  then return ()
		  else sendGuiMsg (GuiAdd arrow ([FillColor (colorFct action),
						 Thickness 1] ++ arrowName))
	     else do
	          sendGuiMsg (GuiAdd arrow ([FillColor (colorFct action),
					    Thickness 2] ++ arrowName))
	          sendGuiMsg (GuiDel arrow)
	else return ()
    else return ()


-------------------------------------------------------------------------------
-- Break Check
-------------------------------------------------------------------------------

-- checks like described in diploma thesis wether a thread should 
--    automaticly continue or not.

breakCheck :: DebuggerState -> ThreadNo -> DebugMsg -> IO DebuggerState
breakCheck state actorNo msg = do
  element <- secondObject state msg
  let (visualizeElement, breakElement) = case element of
        NoObject -> (True, NoTime)
	_ -> (lookupVisualize state element, lookupBreak state element)
      actorObj = ThreadObj actorNo
      visualizeActor = lookupVisualize state actorObj
      breakActor = lookupBreak state actorObj
      breakAction = lookupActionBreak state (reduceDebugMsg msg)
      continue = and [ (breakActor > EveryTime),
		       (breakAction > EveryTime),
		       (breakElement > EveryTime) ]
--      newState = stateSetBreak state actorObj (pred breakActor)
  if (continue || (not visualizeActor) || (not visualizeElement))
    then sendGuiMsg (ThreadReady actorNo)
    else sendGuiMsg (GuiPar actorObj [ FillColor (colorFct TSDebug) ])
  {- case msg of
    ThreadFork (Identity newId) _ -> do
      let threadNo = lookupThreadNo state newId
          threadObj = ThreadObj threadNo
	  visualizeElement = lookupVisualize state threadObj
	  breakElement = lookupBreak state threadObj
	  breakAction = lookupActionBreak state ThreadStartA
	  continueElement = and [ (breakElement > EveryTime),
	  			  (breakAction > EveryTime) ]
      if (continueElement || (not visualizeElement))
        then sendGuiMsg (ThreadReady threadNo)
	else sendGuiMsg (GuiPar threadObj [ FillColor (colorFct TSDelay) ])
    _ -> return () -}
  return state


-------------------------------------------------------------------------------
-- second Object of DebugMsg
-------------------------------------------------------------------------------

-- just returns the second object for the check.

secondObject :: DebuggerState -> DebugMsg -> IO Object
secondObject state msg =
  case msg of
    ThreadForkSuspend (Identity threadId) _ -> 
      return (ThreadObj (lookupThreadNo state threadId))
    ThreadKillSuspend (Identity threadId) _ -> 
      return (ThreadObj (lookupThreadNo state threadId))
    ThreadKill (Identity threadId) _ -> 
      return (ThreadObj (lookupThreadNo state threadId))
    MVarNewEmptySuspend (CHD mvarNoMVar) _ -> do
      mvarNo <- C.readMVar mvarNoMVar
      return (MVarObj mvarNo)
    MVarNewEmpty mvarNo _ -> do
      return (MVarObj mvarNo)
    MVarNew mvarNo _ _ -> do
      return (MVarObj mvarNo)
    MVarTakeSuspend mvarNo _ -> return (MVarObj mvarNo)
    MVarTake mvarNo _ -> return (MVarObj mvarNo)
    MVarReadSuspend mvarNo _ -> return (MVarObj mvarNo)
    MVarRead mvarNo _ -> return (MVarObj mvarNo)
    MVarPutSuspend mvarNo _ _ -> return (MVarObj mvarNo)
    MVarPut mvarNo _ _ -> return (MVarObj mvarNo)
    MVarSwapSuspend mvarNo _ -> return (MVarObj mvarNo)
    MVarSwap mvarNo _ _ -> return (MVarObj mvarNo)
    MVarTryTakeSuspend mvarNo _ -> return (MVarObj mvarNo)
    MVarTryPutSuspend mvarNo _ -> return (MVarObj mvarNo)
    MVarTry mvarNo _ -> return (MVarObj mvarNo)
    MVarDied mvarNo -> return (MVarObj mvarNo)
    MVarLabel mvarNo _ _ -> return (MVarObj mvarNo)

    ChanNewSuspend (CHD chanNoMVar) _ -> do
      chanNo <- C.readMVar chanNoMVar
      return (ChanObj chanNo)
    ChanNew chanNo _ -> do
      return (ChanObj chanNo)
    ChanWriteSuspend chanNo _ -> return (ChanObj chanNo) 
    ChanWrite chanNo _ _ -> return (ChanObj chanNo) 
    ChanReadSuspend chanNo _ -> return (ChanObj chanNo) 
    ChanRead chanNo _ -> return (ChanObj chanNo) 
    ChanUnGetSuspend chanNo -> return (ChanObj chanNo) 
    ChanUnGet chanNo -> return (ChanObj chanNo) 
    ChanDied chanNo -> return (ChanObj chanNo) 
    ChanLabel chanNo _ _ -> return (ChanObj chanNo) 

    QSemNewSuspend (CHD qsemNoMVar) _ _ -> do
      qsemNo <- C.readMVar qsemNoMVar
      return (QSemObj qsemNo)
    QSemNew qsemNo _ -> do
      return (QSemObj qsemNo)  
    QSemWaitSuspend qsemNo _ -> return (QSemObj qsemNo)
    QSemWait qsemNo _ -> return (QSemObj qsemNo)
    QSemSignalSuspend qsemNo _ -> return (QSemObj qsemNo)
    QSemSignal qsemNo _ -> return (QSemObj qsemNo)
    QSemDied qsemNo -> return (QSemObj qsemNo)
    QSemLabel qsemNo _ _ -> return (QSemObj qsemNo)

    QSemNNewSuspend (CHD qsemnNoMVar) _ _ -> do
      qsemnNo <- C.readMVar qsemnNoMVar
      return (QSemNObj qsemnNo)
    QSemNNew qsemnNo _ _ -> do
      return (QSemNObj qsemnNo)
    QSemNWaitSuspend qsemnNo _ _ -> return (QSemNObj qsemnNo)
    QSemNWait qsemnNo _ _ -> return (QSemNObj qsemnNo)
    QSemNSignalSuspend qsemnNo _ _ -> return (QSemNObj qsemnNo)
    QSemNSignal qsemnNo _ _ -> return (QSemNObj qsemnNo)
    QSemNDied qsemnNo -> return (QSemNObj qsemnNo)
    QSemNLabel qsemnNo _ _ -> return (QSemNObj qsemnNo)

    SampleVarNewEmptySuspend (CHD samplevarNoMVar) _ -> do
      samplevarNo <- C.readMVar samplevarNoMVar
      return (SampleVarObj samplevarNo)
    SampleVarNewEmpty samplevarNo _ -> do
      return (SampleVarObj samplevarNo)
    SampleVarNewSuspend (CHD samplevarNoMVar) _ -> do
      samplevarNo <- C.readMVar samplevarNoMVar
      return (SampleVarObj samplevarNo)
    SampleVarNew samplevarNo _ _ -> do
      return (SampleVarObj samplevarNo)
    SampleVarWriteSuspend samplevarNo _ -> return (SampleVarObj samplevarNo)
    SampleVarWrite samplevarNo _ _ -> return (SampleVarObj samplevarNo)
    SampleVarReadSuspend samplevarNo _ -> return (SampleVarObj samplevarNo)
    SampleVarRead samplevarNo _ -> return (SampleVarObj samplevarNo)
    SampleVarEmptySuspend samplevarNo _ -> return (SampleVarObj samplevarNo)
    SampleVarEmpty samplevarNo _ -> return (SampleVarObj samplevarNo)
    SampleVarDied samplevarNo -> return (SampleVarObj samplevarNo)
    SampleVarLabel samplevarNo _ _ -> return (SampleVarObj samplevarNo)
    
    _ -> return NoObject



    
