-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   DebugMsgChan Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------

module CHD.DebugMsgChan
  (
    debugMsgChan,
    DebugMsg(..),
    DebugMsgAction(..),
    CHDInteractMsg(..), 
    reduceDebugMsg,
    interactCHD,
    msgCHD	 
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import qualified Control.Concurrent as C
import System.IO.Unsafe

import CHD.GuiWindow
import CHD.BaseTypes
import CHD.PriorDoubleChannel


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------

type DebugMsgChan = DblChan CHDInteractMsg DebugMessage


data CHDInteractMsg = 
    CHDQuit			-- Quitting Gui-Interface
  | CHDStop			-- Stop CHD (switching to Prio-Mode)
  | CHDContinue			-- Continue CHD (switching to DblChan)
  | Continue ThreadNo		-- Continue Thread
  | AutoContinue ThreadNo Int	-- AutoContinue with correct Identifier
  | Visualize Object Bool	-- set Visualization for Object
  | BreakAction DebugMsgAction BreakCounter -- set Break for Object
  | DisplayPosition String
  | GetInfoGuiWin GuiWindow
  deriving Show


-- complete message from the program to CHD

type DebugMessage = (C.ThreadId, DebugMsg, C.MVar ())


-- these two are needed to implement a protokoll-feature. 
-- this not happend yet!
-- one type cannot be written as string, the other does.

data DebugMsgAction = 
      ThreadBreakA		-- top Position because not setable

    | ProgramLabelA
    | ProgramStartA

    | ThreadForkSuspendA
    | ThreadForkA
    | ThreadKillSuspendA
    | ThreadKillA
    | ThreadStartA
    | ThreadEndA
    | ThreadLabelA
    | ThreadMarkA
    | ThreadUnmarkA
    | ThreadDelaySuspendA
    | ThreadDelayA
    
    | MVarNewEmptySuspendA
    | MVarNewEmptyA
    | MVarNewSuspendA
    | MVarNewA
    | MVarTakeSuspendA
    | MVarTakeA
    | MVarReadSuspendA
    | MVarReadA
    | MVarPutSuspendA
    | MVarPutA 
    | MVarSwapSuspendA
    | MVarSwapA
    | MVarTryTakeSuspendA
    | MVarTryPutSuspendA
    | MVarTryA
    | MVarDiedA
    | MVarLabelA

    | ChanNewSuspendA
    | ChanNewA
    | ChanWriteSuspendA
    | ChanWriteA
    | ChanReadSuspendA
    | ChanReadA
    | ChanUnGetSuspendA
    | ChanUnGetA
    | ChanDiedA
    | ChanLabelA

    | QSemNewSuspendA
    | QSemNewA
    | QSemWaitSuspendA
    | QSemWaitA
    | QSemSignalSuspendA
    | QSemSignalA
    | QSemDiedA
    | QSemLabelA

    | QSemNNewSuspendA
    | QSemNNewA
    | QSemNWaitSuspendA
    | QSemNWaitA
    | QSemNSignalSuspendA
    | QSemNSignalA
    | QSemNDiedA
    | QSemNLabelA

    | SampleVarNewEmptySuspendA
    | SampleVarNewEmptyA
    | SampleVarNewSuspendA
    | SampleVarNewA
    | SampleVarEmptySuspendA
    | SampleVarEmptyA
    | SampleVarReadSuspendA
    | SampleVarReadA
    | SampleVarWriteSuspendA
    | SampleVarWriteA
    | SampleVarDiedA 
    | SampleVarLabelA
  deriving (Show, Eq, Ord, Read, Enum)
instance Bounded DebugMsgAction where
  minBound = ProgramLabelA
  maxBound = SampleVarLabelA


-- main message-type.
--   implement new messages here first!

data DebugMsg = 
      ProgramLabel String
    | ProgramStart ReturnThread

    | ThreadStart
    | ThreadForkSuspend ReturnThread CodePosition
    | ThreadFork ReturnThread (ReturnMVar ()) CodePosition
    | ThreadKillSuspend ReturnThread CodePosition
    | ThreadKill ReturnThread CodePosition
    | ThreadEnd
    | ThreadLabel String (C.MVar String)
    | ThreadMark ReturnThread String
    | ThreadUnmark ReturnThread
    | ThreadDelaySuspend
    | ThreadDelay
    | ThreadBreak String
    
    | MVarNewEmptySuspend (ReturnMVar MVarNo) CodePosition
    | MVarNewEmpty MVarNo CodePosition
    | MVarNewSuspend (ReturnMVar MVarNo) String CodePosition
    | MVarNew MVarNo String CodePosition
    | MVarTakeSuspend MVarNo CodePosition	-- will suspend on MVar
    | MVarTake MVarNo CodePosition		-- took from MVar
    | MVarReadSuspend MVarNo CodePosition	-- will suspend read on MVar
    | MVarRead MVarNo CodePosition		-- read from MVar
    | MVarPutSuspend MVarNo String CodePosition	-- will put in MVar
    | MVarPut MVarNo String CodePosition	-- done put in MVar
    | MVarSwapSuspend MVarNo CodePosition	-- will suspend swap on MVar
    | MVarSwap MVarNo String CodePosition
    | MVarTryTakeSuspend MVarNo CodePosition
    | MVarTryPutSuspend MVarNo CodePosition
    | MVarTry MVarNo CodePosition
    | MVarDied MVarNo				-- MVar is garbage collected
    | MVarLabel MVarNo String (C.MVar String)	-- label MVar

    | ChanNewSuspend (ReturnMVar ChanNo) CodePosition
    | ChanNew ChanNo CodePosition
    | ChanWriteSuspend ChanNo CodePosition
    | ChanWrite ChanNo String CodePosition
    | ChanReadSuspend ChanNo CodePosition
    | ChanRead ChanNo CodePosition
    | ChanUnGetSuspend ChanNo
    | ChanUnGet ChanNo
    | ChanDied ChanNo
    | ChanLabel ChanNo String (C.MVar String)

    | QSemNewSuspend (ReturnMVar QSemNo) Int CodePosition
    | QSemNew QSemNo CodePosition
    | QSemDied QSemNo
    | QSemWaitSuspend QSemNo CodePosition
    | QSemWait QSemNo CodePosition
    | QSemSignalSuspend QSemNo CodePosition
    | QSemSignal QSemNo CodePosition
    | QSemLabel QSemNo String (C.MVar String)

    | QSemNNewSuspend (ReturnMVar QSemNNo) Int CodePosition
    | QSemNNew QSemNNo Int CodePosition
    | QSemNDied QSemNNo
    | QSemNWaitSuspend QSemNNo Int CodePosition
    | QSemNWait QSemNNo Int CodePosition
    | QSemNSignalSuspend QSemNNo Int CodePosition
    | QSemNSignal QSemNNo Int CodePosition
    | QSemNLabel QSemNNo String (C.MVar String)

    | SampleVarNewEmptySuspend (ReturnMVar SampleVarNo) CodePosition
    | SampleVarNewEmpty SampleVarNo CodePosition
    | SampleVarNewSuspend (ReturnMVar SampleVarNo) CodePosition
    | SampleVarNew SampleVarNo String CodePosition
    | SampleVarEmptySuspend SampleVarNo CodePosition
    | SampleVarEmpty SampleVarNo CodePosition
    | SampleVarReadSuspend SampleVarNo CodePosition
    | SampleVarRead SampleVarNo CodePosition
    | SampleVarWriteSuspend SampleVarNo CodePosition
    | SampleVarWrite SampleVarNo String CodePosition
    | SampleVarDied SampleVarNo
    | SampleVarLabel SampleVarNo String (C.MVar String)

instance Show DebugMsg where
  showsPrec _ debugMsg =
    case debugMsg of 
      ProgramLabel name		    -> showString "label Program: " .
				       shows name
      ProgramStart _		    -> showString "start Program"


      ThreadForkSuspend _ _	    -> showString "ForkSuspend"
      ThreadFork _ _ _		    -> showString "Fork"
      ThreadKillSuspend thread _    -> showString "KillSuspend " .
				       showsThread thread
      ThreadKill thread	_	    -> showString "Kill " .
				       showsThread thread
      ThreadStart		    -> showString "Start"
      ThreadEnd			    -> showString "End"
      ThreadLabel name _	    -> showString "Label " . shows name
      ThreadMark thread _	    -> showString "Mark " .
				       showsThread thread
      ThreadUnmark thread	    -> showString "Unmark " .
				       showsThread thread
      ThreadDelaySuspend	    -> showString "DelaySuspend"
      ThreadDelay		    -> showString "Delay"
      ThreadBreak cause		    -> showString "CHD-Break (". shows cause .
				       showString ")"


      MVarNewEmptySuspend no _	    -> showString "NewEmptySuspend " .
				       showsNumber no
      MVarNewEmpty no _		    -> showString "NewEmpty " . shows no
      MVarNewSuspend no label _	    -> showString "NewSuspend " .
				       showsNumber no . 
				       showString (" with " ++ label)
      MVarNew no label _	    -> showString "New " . shows no . 
				       showString (" with " ++ label)
      MVarTakeSuspend no _	    -> showString "TakeSuspend " . shows no
      MVarTake no _		    -> showString "Take " . shows no
      MVarReadSuspend no _	    -> showString "ReadSuspend " . shows no
      MVarRead no _		    -> showString "Read " . shows no
      MVarPutSuspend no label _	    -> showString "PutSuspend " . shows no .
				       showString (" with " ++ label)
      MVarPut no label _	    -> showString "Put " . shows no . 
				       showString (" with " ++ label)
      MVarSwapSuspend no _	    -> showString "SwapSuspend " . shows no
      MVarSwap no label _	    -> showString "Swap " . shows no .
				       showString (" with " ++ label)
      MVarTryTakeSuspend no _	    -> showString "TryTakeSuspend " . 
				       shows no
      MVarTryPutSuspend no _	    -> showString "TryPutSuspend " . shows no
      MVarTry no _ 		    -> showString "Try " . shows no
      MVarDied no		    -> showString "removed MVar " . shows no .
				       showString " from Memory"
      MVarLabel no name	_	    -> showString "Label " . shows no . 
				       shows (" with " ++ name)


      ChanNewSuspend no	_	    -> showString "NewSuspend " . 
				       showsNumber no
      ChanNew no _		    -> showString "New " . shows no
      ChanWriteSuspend no _	    -> showString "WriteSuspend " . shows no
      ChanWrite no label _	    -> showString "Write " . shows no .
				       shows (" with " ++ label)
      ChanReadSuspend no _	    -> showString "ReadSuspend " . shows no
      ChanRead no _		    -> showString "Read " . shows no
      ChanUnGetSuspend no	    -> showString "UngetSuspend " . shows no
      ChanUnGet no		    -> showString "Unget " . shows no
      ChanDied no		    -> showString "Died " .
				       shows no . showString " from Memory"
      ChanLabel no name _	    -> showString "Label " . shows no .
				       shows (" with " ++ name)

			
      QSemNewSuspend no	_ _         -> showString "NewSuspend " .
				       showsNumber no	
      QSemNew no _		    -> showString "New " .
				       shows no	
      QSemDied no		    -> showString "Died " . shows no 
      QSemWaitSuspend no _	    -> showString "WaitSuspend " . shows no
      QSemWait no _		    -> showString "Wait " . shows no
      QSemSignalSuspend no _	    -> showString "SignalSuspend " .
				       shows no
      QSemSignal no _		    -> showString "Signal " . shows no
      QSemLabel no name	_	    -> showString "Label " . shows no .
				       shows (" with " ++ name)


      QSemNNewSuspend no quantity _ -> showString "NewSuspend " . 
				       showsNumber no .
				       shows " with " . shows quantity
      QSemNNew no quan _            -> showString "New " . shows no .
				       shows (" with " ++ show quan)
      QSemNDied no		    -> showString "Died " . shows no
      QSemNWaitSuspend no quan _    -> showString "WaitSuspend " . shows no .
				       shows (" with " ++ show quan)
      QSemNWait no quan _	    -> showString "Wait " . shows no .
				       shows (" with " ++ show quan)
      QSemNSignalSuspend no _ _	    -> showString "SignalSuspend " .
				       shows no
      QSemNSignal no quan _	    -> showString "Signal " . shows no .
				       shows (" with " ++ show quan)
      QSemNLabel no name _          -> showString "Label " . shows no .
				       shows (" with " ++ name)


      SampleVarNewEmptySuspend no _ -> showString "NewEmptySuspend " .
				       showsNumber no
      SampleVarNewEmpty no _	    -> showString "NewEmpty " . 
				       shows no
      SampleVarNew no label _	    -> showString "New " . shows no .
				       shows (" with " ++ label)
      SampleVarEmptySuspend no _    -> showString "EmptySuspend " . shows no
      SampleVarEmpty no _	    -> showString "Empty " . shows no
      SampleVarReadSuspend no _     -> showString "ReadSuspend " . shows no
      SampleVarRead no _	    -> showString "Read " . shows no
      SampleVarWriteSuspend no _    -> showString "WriteSuspend " .
				       shows no
      SampleVarWrite no label _     -> showString "Write " . shows no .
				       shows (" with " ++ label)
      SampleVarDied no		    -> showString "Died " . shows no
      SampleVarLabel no name _	    -> showString "Label " .shows no .
				       shows name
      msg -> error (show msg)

    where 
    showsThread thread = 
      case thread of
        Number threadNo -> shows threadNo
        Identity _ -> showString "new Thread"

    showsNumber number =
      case number of
        CHD _ -> showString "(Number)"
        Gui num -> shows num


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------

reduceDebugMsg :: DebugMsg -> DebugMsgAction
reduceDebugMsg msg = 
  case msg of
    ProgramLabel _ -> ProgramLabelA
    ProgramStart _ -> ProgramStartA

    ThreadForkSuspend _ _ -> ThreadForkSuspendA
    ThreadFork _ _ _ -> ThreadForkA
    ThreadKillSuspend _ _ -> ThreadKillSuspendA
    ThreadKill _ _ -> ThreadKillA
    ThreadStart -> ThreadStartA
    ThreadEnd -> ThreadEndA
    ThreadLabel _ _ -> ThreadLabelA
    ThreadMark _ _ -> ThreadMarkA
    ThreadUnmark _ -> ThreadUnmarkA
    ThreadDelaySuspend -> ThreadDelaySuspendA
    ThreadDelay -> ThreadDelayA
    ThreadBreak _ -> ThreadBreakA

    MVarNewEmptySuspend _ _ -> MVarNewEmptySuspendA
    MVarNewEmpty _ _ -> MVarNewEmptyA
    MVarNewSuspend _ _ _ -> MVarNewSuspendA
    MVarNew _ _ _ -> MVarNewA
    MVarTakeSuspend _ _ -> MVarTakeSuspendA
    MVarTake _ _ -> MVarTakeA
    MVarReadSuspend _ _ -> MVarReadSuspendA
    MVarRead _ _ -> MVarReadA
    MVarPutSuspend _ _ _ -> MVarPutSuspendA
    MVarPut _ _ _ -> MVarPutA
    MVarSwapSuspend _ _ -> MVarSwapSuspendA
    MVarSwap _ _ _ -> MVarSwapA
    MVarTryPutSuspend _ _ -> MVarTryPutSuspendA
    MVarTryTakeSuspend _ _ -> MVarTryTakeSuspendA
    MVarTry _ _ -> MVarTryA
    MVarDied _  -> MVarDiedA
    MVarLabel _ _ _ -> MVarLabelA

    ChanNewSuspend _ _ -> ChanNewSuspendA
    ChanNew _ _ -> ChanNewA
    ChanWriteSuspend _ _ -> ChanWriteSuspendA
    ChanWrite _ _ _ -> ChanWriteA
    ChanReadSuspend _ _ -> ChanReadSuspendA
    ChanRead _ _ -> ChanReadA
    ChanUnGetSuspend _ -> ChanUnGetSuspendA
    ChanUnGet _ -> ChanUnGetA
    ChanDied _ -> ChanDiedA
    ChanLabel _ _ _ -> ChanLabelA

    QSemNewSuspend _ _ _ -> QSemNewSuspendA 
    QSemNew _ _ -> QSemNewA 
    QSemWaitSuspend _ _ -> QSemWaitSuspendA
    QSemWait _ _ -> QSemWaitA
    QSemSignalSuspend _ _ -> QSemSignalSuspendA
    QSemSignal _ _ -> QSemSignalA
    QSemDied _ -> QSemDiedA
    QSemLabel _ _ _ -> QSemLabelA

    QSemNNewSuspend _ _ _ -> QSemNNewSuspendA
    QSemNNew _ _ _ -> QSemNNewA 
    QSemNWaitSuspend _ _ _ -> QSemNWaitSuspendA
    QSemNWait _ _ _ -> QSemNWaitA
    QSemNSignalSuspend _ _ _ -> QSemNSignalSuspendA
    QSemNSignal _ _ _ -> QSemNSignalA
    QSemNDied _ -> QSemNDiedA
    QSemNLabel _ _ _ -> QSemNLabelA

    SampleVarNewEmptySuspend _ _ -> SampleVarNewEmptySuspendA
    SampleVarNewEmpty _ _ -> SampleVarNewEmptyA
    SampleVarNewSuspend _ _ -> SampleVarNewSuspendA
    SampleVarNew _ _ _ -> SampleVarNewA
    SampleVarEmptySuspend _ _ -> SampleVarEmptySuspendA
    SampleVarEmpty _ _ -> SampleVarEmptyA
    SampleVarReadSuspend _ _ -> SampleVarReadSuspendA
    SampleVarRead _ _ -> SampleVarReadA
    SampleVarWriteSuspend _ _ -> SampleVarWriteSuspendA
    SampleVarWrite _ _ _ -> SampleVarWriteA    
    SampleVarDied _ -> SampleVarDiedA
    SampleVarLabel _ _ _ -> SampleVarLabelA


-------------------------------------------------------------------------------
-- DEBUGGER message channel constant
--   first time a Concurrent-function is called,
--   debugger will be started
--   (term will be reduced once)
-------------------------------------------------------------------------------

debugMsgChan :: DebugMsgChan
debugMsgChan = unsafePerformIO ( do
  dbgChan <- newDblChan
  myId <- C.myThreadId
  debugStop <- C.newEmptyMVar
  writeDblChanLess dbgChan (myId, ProgramStart (Identity myId), debugStop)
  return dbgChan
  )


-------------------------------------------------------------------------------
-- DEBUG message sender
--   create stopper to wait for ackknowledgement from user
-------------------------------------------------------------------------------

msgCHD :: DebugMsg -> IO (C.MVar ())
msgCHD message = do
  debugStop <- C.newEmptyMVar
  myId <- C.myThreadId
  writeDblChanLess debugMsgChan (myId, message, debugStop)
  return debugStop


interactCHD :: CHDInteractMsg -> IO ()
interactCHD message = do
  writeDblChanPrio debugMsgChan message
