-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   GuiState Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------

-- description shortcut: GS = GuiState

-- allinall a implementation of the record-type of the GUI-state
-- most functions are lookup and add/remove-functions
-- the calculations of the canvas-positions are not as simple.

-- some functions have to be GUI-actions to interact with the canvas
--      like binding a popupMenu.

module CHD.GuiState
  (
    CanvasObjectState(..),
    GuiState(..),
    initGuiState,		-- :: GS
    lookupObjectState,		-- :: GS -> Object -> CanvasObjectState
    setObjectState,		-- :: GS -> Object -> CanvasObjectState -> GS
    stateAddGuiWindow,		-- :: GS -> ConfigWindow -> OperateConfWin 
				--	 -> GS
    execGuiWindow,		-- :: GS -> ConfigWindow -> ConfWinMsg 
				--	 -> GUI ()
    stateRemGuiWindow,		-- :: GS -> ConfigWindow -> GS
    stateAddObject,		-- :: GS -> Object -> GUI GS
    stateParameterObject,	-- :: GS -> Object -> Parameter -> GUI GS
    stateRemoveObject,		-- :: GS -> Object -> GUI GS
    displayElements,
    maybeObjectState,
    initCanvasObjectState,
    arrowCoords,
    designThread,
    getWidthCommElem,
    getHeightCommElem,
    fontCommElem,
    CommElem,
    removeCommElem,
    checkScrollRegion,
    maxCommElem,
    getNewCoord,
    guiMoveObject,		-- :: GS -> Object -> Coord -> GUI GS
    removeCanvasObject,		-- :: GS -> Object -> Int -> GUI GS
    linkBefore			-- :: GS -> Object -> Maybe Object -> GUI GS
  )
    where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import Data.FiniteMap
import Tcl

import CHD.BaseTypes
import CHD.BaseFunctions
import CHD.GuiMsgChan
import CHD.DebugMsgChan
import CHD.GuiWindow
import CHD.Environment

-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------

type CommElem = (CWidget CRec, CWidget CTex)


data CanvasObjectState = CanvasObjectState {
  coordinate	       :: Coord,
  labelText	       :: CWidget CTex,
  threadOval	       :: CWidget COva,
  threadAction	       :: CWidget CTex,
  arrowLine	       :: CWidget CLin,
  arrowText	       :: CWidget CTex,
  commElem	       :: [CommElem],
  endText	       :: Maybe (CWidget CTex),
  nextCO, prevCO       :: Maybe Object,
  arrows	       :: [Object],
  delIdentifier	       :: Int
  }


-- Canvas Config State
data CanvasPreferences = CanvasPreferences {
  sizeThread		 :: Int,
  pos0Thread		 :: Coord,
  maxCommElem		 :: Int,
  heightCommVarElem	 :: Int,
  widthCommVarElem	 :: Int,
  heightCommSemElem	 :: Int,
  widthCommSemElem	 :: Int,
  fontCommElem		 :: String,
  pos0Comm		 :: Coord
  }
 

data GuiState = GuiState {
  windowMain		 :: Window,
  guiWindows		 :: FiniteMap GuiWindow OperateGuiWin,
  -- protocol		 :: [GuiMsg],			-- !!!
  windowProtocol	 :: Maybe Window,
  editProtocol		 :: Maybe Edit,
  canvasMain		 :: Canvas,

  objectDB		 :: FiniteMap Object CanvasObjectState,

  layoutOperation	 :: GuiState -> Object 
			    -> (Coord, Maybe Object, GuiState),

  lastThread, lastComm	 :: Maybe Object,
  preference		 :: CanvasPreferences,

  removeArrowDelay	 :: Int,
  removeElementDelay	 :: Int
  }


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------


defaultCanvasPreferences :: CanvasPreferences
defaultCanvasPreferences = CanvasPreferences {
    sizeThread		 = 20,
    pos0Thread		 = (150,20),
    maxCommElem		 = 4,
    heightCommVarElem	 = 14,
    widthCommVarElem	 = 100,
    heightCommSemElem	 = 14,
    widthCommSemElem	 = 25,
    fontCommElem	 = "-adobe-helvetica-medium-r-normal--10-100-75-75-p-56-iso8859-1",
    pos0Comm		 = (330,20)
  }


initGuiState :: GuiState
initGuiState = GuiState {
  windowMain		      = undefined,
  guiWindows		      = emptyFM,
  -- protocol		      = [],             -- !!!
  windowProtocol	      = Nothing,
  editProtocol		      = Nothing,
  canvasMain		      = undefined,

  objectDB		      = emptyFM,

  layoutOperation	      = layoutStandard,

  lastThread		      = Nothing, 
  lastComm		      = Nothing,
  preference		      = defaultCanvasPreferences,

  removeArrowDelay	      = removeArrowDelayOption,
  removeElementDelay	      = removeElementDelayOption
}


initCanvasObjectState :: CanvasObjectState
initCanvasObjectState = CanvasObjectState {
  coordinate	      = (0,0),
  labelText	      = undefined,
  threadOval	      = undefined,
  threadAction	      = undefined,
  arrowLine	      = undefined,
  arrowText	      = undefined,
  commElem	      = [],
  endText	      = Nothing,
  
  nextCO	      = Nothing,
  prevCO	      = Nothing,
  arrows	      = [],
  delIdentifier	      = 1
  }


-------------------------------------------------------------------------------
-- layout FUNCTIONS
-------------------------------------------------------------------------------


getHeightCommElem :: GuiState -> Object -> Int
getHeightCommElem guiState object = 
  case object of
       QSemObj _ -> heightCommSemElem (preference guiState)
       QSemNObj _ -> heightCommSemElem (preference guiState)
       _ -> heightCommVarElem (preference guiState)	


getWidthCommElem ::  GuiState -> Object -> Int
getWidthCommElem guiState object = 
  case object of
       QSemObj _ -> widthCommSemElem (preference guiState)
       QSemNObj _ -> widthCommSemElem (preference guiState)		
       _ -> widthCommVarElem (preference guiState)	


checkScrollRegion :: GuiState -> Rect -> GUI ()
checkScrollRegion state ((x1,y1),(x2,y2)) = do
  let canvas = canvasMain state
  scrollRegionString <- cget canvas scrollregion
  let [sx1,sy1,sx2,sy2] = map read (words scrollRegionString)
      scrollRegion = ((min x1 sx1, min y1 sy1),(max x2 sx2, max y2 sy2))
  cset canvas [scrollregion scrollRegion]


layoutStandard :: GuiState -> Object -> (Coord, Maybe Object, GuiState)
layoutStandard guiState object = 
  let hThread = sizeThread $ preference guiState
      hCommElem = getHeightCommElem guiState object
      posThread = pos0Thread $ preference guiState
      posComm = pos0Comm $ preference guiState in
  case object of
    ThreadObj _ -> 
      case lastThread guiState of
        Nothing -> (posThread, Nothing, guiState { lastThread = Just object })
	Just last -> 
	  let cOld = coordinate (lookupObjectState guiState last) 
	      cNew = vectorAdd cOld (0,20+hThread) in
	  (cNew, Just last, guiState { lastThread = Just object })
    ArrowObj _ -> 
      ((0,0), Nothing, guiState)
    _ -> 
      case lastComm guiState of
        Nothing -> (posComm, Nothing, guiState { lastComm = Just object })
	Just last -> 
	  let cOld = coordinate (lookupObjectState guiState last) 
	      cNew = vectorAdd cOld (0,20+hCommElem) in
	  (cNew, Just last, guiState { lastComm = Just object })


getNewCoord :: GuiState -> Object -> (Coord, Maybe Object, GuiState)
getNewCoord guiState = (layoutOperation guiState) guiState


-------------------------------------------------------------------------------
-- ObjectState FUNCTIONS
-------------------------------------------------------------------------------


maybeObjectState :: GuiState -> Object -> Maybe CanvasObjectState
maybeObjectState state object =
  lookupFM (objectDB state) object


lookupObjectState :: GuiState -> Object -> CanvasObjectState
lookupObjectState state object = 
  lookupWithDefaultFM (objectDB state)
		      initCanvasObjectState
		      object


setObjectState :: GuiState -> Object -> CanvasObjectState -> GuiState
setObjectState state object objectState =
  state { objectDB = addToFM (objectDB state) object objectState }


removeObjectState :: GuiState -> Object -> GuiState
removeObjectState state object =
  state { objectDB = delFromFM (objectDB state) object }


-------------------------------------------------------------------------------
-- state GUIWINDOW
-------------------------------------------------------------------------------

stateAddGuiWindow :: GuiState -> GuiWindow -> OperateGuiWin -> GuiState
stateAddGuiWindow state key operate =
  state { guiWindows = addToFM (guiWindows state) key operate }


lookupGuiWindow :: GuiState -> GuiWindow -> Maybe OperateGuiWin
lookupGuiWindow state key =
  lookupFM (guiWindows state) key


execGuiWindow :: GuiState -> GuiWindow -> GuiWinMsg -> GUI ()
execGuiWindow state key message =
  maybe (return ())
	(\operate -> operate message)
	(lookupFM (guiWindows state) key)


stateRemGuiWindow :: GuiState -> GuiWindow -> GuiState
stateRemGuiWindow state key =
  state { guiWindows = delFromFM (guiWindows state) key }


-------------------------------------------------------------------------------
-- state modify FUNCTIONS
-------------------------------------------------------------------------------

stateAddObject :: GuiState -> Object -> GUI GuiState
stateAddObject guiState object = 
  let popup = (\coord -> interactGui $ PopupMenu object [] coord)
      moveup = interactGui $ MoveUp object in 
  case maybeObjectState guiState object of
    Nothing -> case object of
      ThreadObj tNo -> do
        state <- designThread guiState object
	let objectState = lookupObjectState state object
	    t = labelText objectState
	    a = threadAction objectState
	    o = threadOval objectState
	    continue = proc $ interactCHD (Continue tNo) -- !!!
        cset t [text (show tNo)]
	bind t "<1>" continue  -- !!!
	bind a "<1>" continue  -- !!!
	bind o "<1>" continue  -- !!!
	bind t "<2>" moveup
	bind a "<2>" moveup
	bind o "<2>" moveup
	bindXY t "<3>" popup
	bindXY a "<3>" popup
	bindXY o "<3>" popup
        return state
      ArrowObj _ -> do
        state <- designArrow guiState object
        return state
      _ -> do
        state <- designComm guiState object
	let objectState = lookupObjectState state object
	    t = labelText objectState
	    continue = interactGui $ ContinueSuspended object
        cset t [text (show object)]
	bind t "<1>" continue
	bind t "<2>" moveup
	bindXY t "<3>" popup
        return state
    Just objectState -> do
      let newObjectState = objectState { delIdentifier = 
				           succ (delIdentifier objectState) }
      return ( setObjectState guiState object newObjectState )


stateParameterObject :: GuiState -> Object -> Parameter -> GUI GuiState
stateParameterObject state object parameter = do
  case maybeObjectState state object of
    Just objectState -> case (object, parameter) of
      (ArrowObj _, Name name) -> do
        cset (arrowText objectState) [text (take 20 name)]
	return state
      (_, Name name) -> do
        cset (labelText objectState) [text (take 20 name)]
	return state
      (ThreadObj _, Action name) -> do
        cset (threadAction objectState) [text name]
	return state
      (ArrowObj _, FillColor color) -> do
        cset (arrowLine objectState) [fill color]
	return state
      (ThreadObj _, FillColor color) -> do
        cset (threadOval objectState) [fill color]
	return state
      (ChanObj _, FillColor color) -> 
        let list = commElem (lookupObjectState state object) 
	    func = (\(r,_) y -> (cset r [fill color]) >> y) in
	do
	foldr func (return ()) list
	cset (labelText objectState) [fill color]
	return state
      (_, FillColor color) ->
        let [(r, _)] = commElem (lookupObjectState state object) in
	do
	cset r [fill color]
	return state	
      (ThreadObj _, OutlineColor color) -> do
        cset (threadOval objectState) [outline color]
	return state
      (ChanObj _, OutlineColor color) -> 
        let list = commElem (lookupObjectState state object) 
	    func = (\(r,_) y -> (cset r [outline color]) >> y) in
	do
	foldr func (return ()) list
	cset (labelText objectState) [fill color]
	return state
      (_, OutlineColor color) ->
        let [(r, _)] = commElem (lookupObjectState state object) in
	do
	cset r [outline color]
	cset (labelText objectState) [fill color]
	return state
      (ArrowObj _, Thickness x) -> do
	cset (arrowLine objectState) [width x]
	return state
      (ThreadObj threadObj, Thickness x) -> do
	cset (threadOval (lookupObjectState state (ThreadObj threadObj))) 
	     [width x]
	return state	
      (MVarObj _, Elements list) -> do
        newState <- displayElements state object list
	return newState
      (ChanObj _, Elements list) -> do
        newState <- displayElements state object list
	return newState
      (QSemObj _, Elements list) -> do
        newState <- displayElements state object list
	return newState
      (QSemNObj _, Elements list) -> do
        newState <- displayElements state object list
	return newState
      (SampleVarObj _, Elements list) -> do
        newState <- displayElements state object list
	return newState	

      _ -> return state
    Nothing -> return state



stateRemoveObject :: GuiState -> Object -> GUI GuiState
stateRemoveObject guiState object = 
  case maybeObjectState guiState object of
    Just objectState -> do
      let removeMsg = interactGui (GuiRemove object 
					     (delIdentifier objectState))
      case object of
	ThreadObj _ -> do
	  let label = labelText objectState
	      oval = threadOval objectState
	      action = threadAction objectState
	  cset oval [outline "red", width 2]
	  cset label [fill "red"]
	  cset action [fill "red"]
	  bind oval "<1>" removeMsg
	  bind label "<1>" removeMsg
	  bind action "<1>" removeMsg
	  bind oval "<2>" removeMsg
	  bind label "<2>" removeMsg
	  bind action "<2>" removeMsg
	  bind oval "<3>" removeMsg
	  bind label "<3>" removeMsg
	  bind action "<3>" removeMsg
	  return ()
	ArrowObj _ -> do
	  bind (arrowLine objectState) "<1>" removeMsg
	  return ()
	_ -> do
	  let label = labelText objectState
	  cset label [fill "red"]
	  bind label "<1>" removeMsg
	  bind label "<2>" removeMsg
	  bind label "<3>" removeMsg
	  mapM_ (\(rect, text) -> do 
		  cset rect [outline "red"]
		  bind rect "<1>" removeMsg
		  bind text "<1>" removeMsg 
		  bind rect "<2>" removeMsg
		  bind text "<2>" removeMsg 
		  bind rect "<3>" removeMsg
		  bind text "<3>" removeMsg) 
		(commElem objectState)
	  maybe (return ()) 
		(\object -> (do bind object "<1>" removeMsg
				bind object "<2>" removeMsg
				bind object "<3>" removeMsg
				return ())) 
		(endText objectState)
	  return ()
      case object of
        ArrowObj _ -> after (removeArrowDelay guiState) removeMsg
	_ -> after (removeElementDelay guiState) removeMsg
      return guiState
    Nothing -> return guiState


designArrow :: GuiState -> Object -> GUI GuiState
designArrow state object@(ArrowObj (obj1, obj2)) = do
  let canvas = canvasMain state
      coords = arrowCoords state object
  l <- cline canvas coords []
  t <- ctext canvas (arrowTextCoords coords) [fill (colorFct object)]
  let objectState = initCanvasObjectState { arrowLine = l, arrowText = t }
  state1 <- case maybeObjectState state obj1 of
    Nothing -> return state
    Just obj1State ->
      let newObj1State = obj1State { arrows = object : (arrows obj1State) } in
      return ( setObjectState state obj1 newObj1State )
  state2 <- case maybeObjectState state1 obj2 of
    Nothing -> return state1
    Just obj2State ->
      let newObj2State = obj2State { arrows = object : (arrows obj2State) } in
      return ( setObjectState state1 obj2 newObj2State )
  return (setObjectState state2 object objectState)
designArrow _ _ = error "error in designArrow"


designComm :: GuiState -> Object -> GUI GuiState
designComm guiState object = do
  let (c@(x,y),last,state) = getNewCoord guiState object
      canvas = canvasMain state
  checkScrollRegion state ((x,y-20),(x+130,y))
  t <- ctext canvas c [anchor SW, fill (colorFct object)]
  let state1 = case last of
        Nothing -> state
        Just obj -> 
	  let objStat = lookupObjectState state obj in
	  setObjectState state obj (objStat { nextCO = Just object })
  let objectState =  initCanvasObjectState {
		       arrows = [],
		       prevCO = last,
		       nextCO = Nothing,
		       coordinate = c,
		       labelText = t,
		       commElem = [],
		       endText = Nothing }
      newState = setObjectState state1 object objectState
  displayElements newState object []


displayElements :: GuiState -> Object -> [String] -> GUI GuiState
displayElements state object list = do
  let popup = (\coord -> interactGui $ PopupMenu object list coord)
      continue = interactGui $ ContinueSuspended object
      moveup = interactGui $ MoveUp object
      objectState = lookupObjectState state object
      color = colorFct object
      canvas = canvasMain state
      maxElem = maxCommElem $ preference state
      shortList = map (take 15) list
      l = max 1 (length shortList)
      difference = l - maxElem
      c@(_,y) = coordinate objectState
      height = getHeightCommElem state object
      width = getWidthCommElem state object
      endTextCoord = vectorAdd (coordinate objectState) 
			       (maxElem*(width+2)+2,(div height 2)+1)
      (elem, trash) = splitAt l (commElem objectState)
  checkScrollRegion state (c,((fst endTextCoord)+20,y+height+10))
  removeCommElem trash
  newElem <- case (elem, shortList) of
    ([], []) -> do
      r <- crectangle canvas 
		      c 
		      (vectorAdd c (width,height)) 
		      [outline color, fill "white"] 
      t <- ctext canvas 
		 (vectorAdd c ((div width 2),(div height 2)+1)) 
		 [ anchor C, justify CenterJ,
		   font (fontCommElem $ preference state) ]
      bind r "<1>" continue
      bind t "<1>" continue
      bind r "<2>" moveup
      bind t "<2>" moveup
      bindXY r "<3>" popup
      bindXY t "<3>" popup
      return [(r,t)]
    ([(r, t)], []) -> do
      cset r [outline color, fill "white"]
      cset t [text ""]
      return elem
    _ -> do
      (_, newElemList) <- foldM
        (\(coord, newList) (maybeElem, textElem) -> do
	  elem <- case maybeElem of
	    Just (r,t) -> do
	      cset r [outline color, fill color]
	      cset t [text textElem, fill "white"]
	      bindXY r "<3>" popup	-- ??
	      bindXY t "<3>" popup	-- ??
	      return (r,t)
	    Nothing -> do
	      r <- crectangle canvas 
		     coord 
		     (vectorAdd coord (width,height)) 
		     [outline color, fill color] 
	      t <- ctext canvas 
		     (vectorAdd coord ((div width 2),(div height 2)+1)) 
		     [ anchor C, justify CenterJ, text textElem, fill "white",
		       font (fontCommElem $ preference state) ]
	      bind r "<1>" continue
	      bind t "<1>" continue
	      bind r "<2>" moveup
	      bind t "<2>" moveup
	      bindXY r "<3>" popup
	      bindXY t "<3>" popup
	      return (r,t)
	  return (vectorAdd coord (width+2,0),newList ++ [elem]) )
        (c,[])
	(zipMaybe elem (take maxElem shortList))
      return newElemList
  newEndText <- if (l > maxElem)
    then maybe (do t <- ctext canvas 
			      endTextCoord 
			      [ anchor W, 
			        text ("+" ++ (show difference)), 
			        fill "red",
				font (fontCommElem $ preference state) ]
		   bind t "<1>" continue
		   bind t "<2>" moveup
		   bindXY t "<3>" popup
		   return (Just t))
	       (\t -> do cset t 
			      [text ("+" ++ (show difference))]
			 return (Just t))
	       (endText objectState)
    else maybe (return Nothing)
	       (\t -> do removeObject t
			 return Nothing)
	       (endText objectState)
  let newObjectState = objectState { commElem = newElem,
				     endText = newEndText }
  return ( setObjectState state object newObjectState )
  where
  refreshElements :: GuiState -> Coord -> String -> [String] -> [CommElem] 
		     -> [CommElem] -> GUI [CommElem]
  refreshElements _ _ _ [] _ out = return out
  refreshElements state c color (t1:shortList) [] out = do
    let canvas = canvasMain state
	height = getHeightCommElem state object
	width = getWidthCommElem state object
    r <- crectangle canvas 
		    c 
		    (vectorAdd c (width,height)) 
		    [outline "white", fill color] 
    t <- ctext canvas 
	       (vectorAdd c ((div width 2),(div height 2)+1)) 
	       [ anchor C, justify CenterJ, text t1, fill "white",
		 font (fontCommElem $ preference state) ]
    refreshElements state 
		    (vectorAdd c (width,0)) 
		    color 
		    shortList 
		    [] 
		    (out ++ [(r,t)]) 
  refreshElements state c color (t1:shortList) ((r,t):elem) out = do
    cset r [outline "white", fill color] 
    cset t [text t1, fill "white"]
    refreshElements state 
		    (vectorAdd c (
		    (getWidthCommElem state object),0)) 
		    color
		    shortList
		    elem 
		    (out ++ [(r,t)]) 


-------------------------------------------------------------------------------
-- design FUNCTIONS
-------------------------------------------------------------------------------


arrowEndLine :: Float -> Float -> [Vector] -> [Vector]
arrowEndLine len alpha list@(_:_:_) =
  let (before, [line, end]) = splitAt ((length list) - 2) list
      beta = vectorDirection end line
      left = vectorCreate len (beta+alpha) 
      right = vectorCreate len (beta-alpha) 
      mid = vectorCreate (2*len/3) beta in
  before ++ [line, 
	     (vectorAdd end mid), 
	     (vectorAdd end left), 
	     end, 
	     (vectorAdd end right), 
	     (vectorAdd end mid)]
arrowEndLine _ _ list = list


arrowCoords :: GuiState -> Object -> [Coord]
arrowCoords state (ArrowObj (obj1, obj2)) = 
  let c1 = case (obj1, maybeObjectState state obj1) of
	     (ThreadObj _, Just objectState) -> 
	       let c = coordinate objectState
		   size = div (sizeThread $ preference state) 2 in
	       [vectorAdd c (5, size), vectorAdd c (30, size)]
	     (ArrowObj _, _) -> []
	     (_, Nothing) -> []
	     (_, Just objectState) ->
	       let c = coordinate objectState
		   size = getHeightCommElem state obj1
		   in -- ACHTUNG
	       [vectorAdd c (-5, size), vectorAdd c (-30, size)]
      c2 = case (obj2, maybeObjectState state obj2) of
             (ThreadObj _, Just objectState) -> 
	       let c = coordinate objectState
		   size = div (sizeThread $ preference state) 2 in
	       (case (obj1, lookupFM (objectDB state) obj1) of
	         (ThreadObj _, Just objectState0) -> 
		   let c0 = coordinate objectState0
		       diff = round ((vectorLength (vectorSub c c0)) / 2)
		       additional = vectorAdd c0 (50, size + diff) in
		   [additional]
		 _ -> []) ++
	         [vectorAdd c (30, size), vectorAdd c (5, size)]
	     (ArrowObj _, _) -> []
	     (_, Nothing) -> []
	     (_, Just objectState) ->
	       let c = coordinate objectState in
	       [vectorAdd c (-30, 0), vectorAdd c (-5, 0)] in
  arrowEndLine 16 (pi/12) (c1 ++ c2)


arrowTextCoords :: [Coord] -> Coord
arrowTextCoords coords =
  let sumCoords = foldr vectorAdd (0,0) coords
      noCoords = toEnum (length coords)
      vLength = (vectorLength sumCoords) / noCoords
      vAngle = vectorAngle sumCoords in
  vectorAdd (vectorCreate vLength vAngle) (0,-5)


designThread :: GuiState -> Object -> GUI GuiState
designThread guiState object = do
  let (c@(x,y),last,state) = getNewCoord guiState object
      canvas = canvasMain state
      size = sizeThread $ preference state    
  checkScrollRegion state ((x-size-130,y-20),(x,y+size+10))
  t <- ctext canvas c [anchor SE, fill (colorFct object)]
  a <- ctext canvas (vectorAdd c (-size-3,div size 2)) 
    [ anchor E, justify RightJ, font $ fontCommElem $ preference guiState ]
  o <- coval canvas c (vectorAdd c (-size,size)) [fill "white"]
  let newState = case last of
        Nothing -> state
	Just obj -> 
	  let objStat = lookupObjectState state obj in
	  setObjectState state obj (objStat { nextCO = Just object })
  let objectState = initCanvasObjectState {
		      arrows = [],
		      prevCO = last,
		      nextCO = Nothing,
		      coordinate = c,
		      labelText = t,
		      threadAction = a,
		      threadOval = o }
  return $ setObjectState newState object objectState






-------------------------------------------------------------------------------
-- move FUNCTIONS
-------------------------------------------------------------------------------


guiMoveObject :: GuiState -> Object -> Coord -> GUI GuiState
guiMoveObject state object cNew = do
  let objectState = lookupObjectState state object
      cOld = coordinate objectState
      cDiff = vectorSub cNew cOld
  (newObjectState, cNext) <- case object of
    ArrowObj _ -> return (objectState, cNew)
    ThreadObj _ -> do
      let hThread = sizeThread $ preference state
      moveObject (labelText objectState) cDiff
      moveObject (threadAction objectState) cDiff
      moveObject (threadOval objectState) cDiff
      return (objectState { coordinate = cNew }, vectorAdd cNew (0,20+hThread))
    _ -> do
      let hComObj = getHeightCommElem state object
      mapM_ (\(r,t) -> do
	      moveObject r cDiff
	      moveObject t cDiff)
	    (commElem objectState)
      maybe (return ())
	    (\object -> moveObject object cDiff)
	    (endText objectState)
      moveObject (labelText objectState) cDiff
      return (objectState { coordinate = cNew }, vectorAdd cNew (0,20+hComObj))
  let newState = setObjectState state object newObjectState
  mapM_ (\arrow -> do
	  let arrowCOS = lookupObjectState newState arrow 
	  setCoords (arrowLine arrowCOS) (arrowCoords newState arrow))
	(arrows newObjectState)
  case (nextCO newObjectState) of
    Nothing -> return newState
    Just nextObject -> guiMoveObject newState nextObject cNext


-------------------------------------------------------------------------------
-- remove FUNCTIONS
-------------------------------------------------------------------------------


removeCommElem :: [CommElem] -> GUI ()
removeCommElem = 
  mapM_ (\(rect, text) -> do 
	  removeObject rect
	  removeObject text) 


removeCanvasObject :: GuiState -> Object -> Int -> GUI GuiState
removeCanvasObject guiState object identifier = 
  case maybeObjectState guiState object of
    Just objectState -> 
      if ((identifier == 0) || (identifier == delIdentifier objectState))
        then do
	  delState <-
	    foldM (\state arrow -> do
		    let arrowState = lookupObjectState state arrow
		    removeCanvasObject state arrow (delIdentifier arrowState))
		  guiState
		  (arrows objectState)
	  state <- case object of
	    ThreadObj _ -> do
	      removeObject (labelText objectState)
	      removeObject (threadAction objectState)
	      removeObject (threadOval objectState)
	      return delState
	    ArrowObj (obj1, obj2) -> do
	      removeObject (arrowLine objectState)
	      removeObject (arrowText objectState)
	      state1 <- case maybeObjectState delState obj1 of
	        Just obj1State -> do
		  let newObj1State = obj1State { arrows = 
				       filter (object /=) (arrows obj1State) }
		  return (setObjectState delState obj1 newObj1State)
		Nothing -> return delState
	      state2 <- case maybeObjectState state1 obj2 of
	        Just obj2State -> do
		  let newObj2State = obj2State { arrows = 
				       filter (object /=) (arrows obj2State) }
		  return (setObjectState state1 obj2 newObj2State)
		Nothing -> return state1
      	      return state2
	    _ -> do
	      removeObject (labelText objectState)
	      maybe (return ()) 
		    (\object -> removeObject object) 
		    (endText objectState)
	      removeCommElem (commElem objectState)
	      return delState
	  newState <- case object of
	    ArrowObj _ -> return state
	    _ -> do
	      let state2 = removeLink state object 
	      case nextCO objectState of
		Nothing -> return state2
		Just obj -> guiMoveObject state2 obj (coordinate objectState)
	  return ( removeObjectState newState object )
      else return guiState
    Nothing -> return guiState


-------------------------------------------------------------------------------
-- link FUNCTIONS
-------------------------------------------------------------------------------


linkBefore :: GuiState -> Object -> Maybe Object -> GuiState
linkBefore linkedState object link = 
  let state = removeLink linkedState object
      lastObject = case object of 
	ThreadObj _ -> lastThread state
	_ -> lastComm state
      (prevObj, nextObj) = findLinkPosition state link lastObject
      newObjectState = (lookupObjectState state object)
		         { prevCO = prevObj,
			   nextCO = nextObj }
      prevState = setLinkNext state prevObj (Just object)
      nextState = setLinkPrev prevState nextObj (Just object)
      final = setObjectState nextState object newObjectState 
  in
  if (lastObject == prevObj)
    then case object of
	   ThreadObj _ -> final { lastThread = (Just object) }
	   _ -> final { lastComm = (Just object) }
    else final


findLinkPosition :: GuiState -> Maybe Object -> Maybe Object 
		      -> (Maybe Object, Maybe Object)
findLinkPosition state link position = 
  if ((link == position) || (position == Nothing))
    then (position, Nothing)
    else 
      let prevObj = maybe Nothing
			  (\object -> prevCO (lookupObjectState state object))
			  position in
      if (link == prevObj)
        then (link, position)
	else findLinkPosition state link prevObj


removeLink :: GuiState -> Object -> GuiState
removeLink state object = 
  let objectState = lookupObjectState state object
      maybePrevObj = prevCO objectState
      maybeNextObj = nextCO objectState
      prevState = setLinkNext state maybePrevObj maybeNextObj 
      nextState = setLinkPrev prevState maybeNextObj maybePrevObj
      newObjectState = objectState 
		         { prevCO = Nothing,
			   nextCO = Nothing }
      final = setObjectState nextState object newObjectState
  in
  case object of
    ThreadObj _ -> if ((lastThread final) == (Just object))
		     then final { lastThread = maybePrevObj }
		     else final
    _ -> if ((lastComm final) == (Just object))
	   then final { lastComm = maybePrevObj }
	   else final


setLinkPrev :: GuiState -> Maybe Object -> Maybe Object -> GuiState
setLinkPrev state maybeObject link =
  maybe state
	(\object -> let objectState = (lookupObjectState state object)
				        { prevCO = link } in
		    setObjectState state object objectState )
	maybeObject


setLinkNext :: GuiState -> Maybe Object -> Maybe Object -> GuiState
setLinkNext state maybeObject link =
  maybe state
	(\object -> let objectState = (lookupObjectState state object)
				        { nextCO = link } in
		    setObjectState state object objectState )
	maybeObject



