-------------------------------------------------------------------------------
-- PriorDoubleChannel
-- 
-- type DblChan a b 
--
-- extension of "type Chan (Either a b)" from module Concurrent 
--   where messages of "type a" have priority
--
-- by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------

module CHD.PriorDoubleChannel
    ( 
      DblChan,			-- type seems to be "Chan (Either a b)"

      newDblChan,		-- :: IO (DblChan a b)			
				-- create Channel

      readDblChan,		-- :: DblChan a b -> IO (Either a b)
				-- read from Channel

      readDblChanPrio,		-- :: DblChan a b -> IO a
				-- read(&suspend) from Prior-Channel 
    
      readDblChanLess,		-- :: DblChan a b -> IO b
				-- read(&suspend) from Less-Channel

      writeDblChanPrio,		-- :: DblChan a b -> a -> IO ()
				-- write to Left (priority) Channel

      writeDblChanLess,		-- :: DblChan a b -> b -> IO ()
				-- write to Right (lesser) Channel

      isEmptyDblChan		-- :: DblChan a b -> IO Bool
				-- check for contents
    )
    where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import Control.Concurrent


-- CONCEPT:
-- DblChan is Quadtrupel where
--   1.Element is MVar to suspend on, when Channels are empty
--   2.Element is Channel of type a  -- (left)
--   3.Element is Channel of type b  -- (right)
--   4.Element is MVar to make operations atomar and 
--                     to signal where to write next data
--                                     (to MVar or Channel)

-- READ will only suspend on the mvar
-- so WRITE must write the mvar instead of the channels 
--   if READ is suspended
--   otherwise WRITE should write the channels

-- READ will check for data 
--   1. channel a  -- (left)
--   2. cahnnel b  -- (right)
--   no data? -> suspend on mvar 


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------


data InputLvl = Suspend | PrioLess

type DblChan a b = (MVar (Either a b),
	            Chan a,
		    Chan b,
		    MVar InputLvl)


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------


newDblChan :: IO (DblChan a b)
newDblChan = do
    suspend <- newEmptyMVar
    prioChan <- newChan
    lessChan <- newChan
    atomar <- newMVar PrioLess
    return (suspend, prioChan, lessChan, atomar)


readDblChan :: DblChan a b -> IO (Either a b)
readDblChan (suspend, prioChan, lessChan, atomar) = do
    takeMVar atomar
    prioCheck <- isEmptyChan prioChan
    if prioCheck then do
        lessCheck <- isEmptyChan lessChan
	if lessCheck then do
	    putMVar atomar Suspend
	    takeMVar suspend
	  else do
	    value <- readChan lessChan
	    putMVar atomar PrioLess
	    return (Right value)
      else do
	value <- readChan prioChan
	putMVar atomar PrioLess
	return (Left value)


readDblChanPrio :: DblChan a b -> IO a
readDblChanPrio (_, prioChan, _, atomar) = do
    takeMVar atomar
    putMVar atomar PrioLess
    readChan prioChan


readDblChanLess :: DblChan a b -> IO b
readDblChanLess (_, _, lessChan, atomar) = do
    takeMVar atomar
    putMVar atomar PrioLess
    readChan lessChan


writeDblChanPrio :: DblChan a b -> a -> IO ()
writeDblChanPrio (suspend, prioChan, _, atomar) value = do
    inputLvl <- takeMVar atomar
    case inputLvl of
        Suspend -> putMVar suspend (Left value)
	PrioLess -> writeChan prioChan value
    putMVar atomar PrioLess


writeDblChanLess :: DblChan a b -> b -> IO ()
writeDblChanLess (suspend, _, lessChan, atomar) value = do
    inputLvl <- takeMVar atomar
    case inputLvl of
        Suspend -> putMVar suspend (Right value)
	PrioLess -> writeChan lessChan value
    putMVar atomar PrioLess


isEmptyDblChan :: DblChan a b -> IO Bool
isEmptyDblChan (_, prioChan, lessChan, _) = do
    prioCheck <- isEmptyChan prioChan
    lessCheck <- isEmptyChan lessChan
    return (prioCheck && lessCheck)
