#include <stdbool.h>
#include <globus_common.h>
#include <globus_io.h>

void globus_if_error_abort(globus_result_t, char *, int);
#define GLOBUS_IF_ERROR_ABORT(result) globus_if_error_abort(result, __FILE__, __LINE__)

#define NTIMES 2

globus_io_attr_t    g_attribute;

globus_io_attr_t g_attribute;
globus_io_secure_authorization_data_t g_auth_data;
globus_mutex_t      g_mutex;
globus_cond_t       g_cond;

int g_callback_count = 0;

#define BUFFER_SIZE (32 * 1024)
const char g_message[] = "Hello FOO";

globus_byte_t g_buffer[BUFFER_SIZE];

void init_stack(void);
void fin_stack(void);
void init_attr(void);

void listener_run(unsigned short *);
void connecter_run(unsigned short);

void callback_inc();
void callback_dec();
void callback_wait();
void accept_cb(void *, globus_io_handle_t *, globus_result_t);
void read_cb(globus_xio_handle_t, globus_result_t, globus_byte_t *, globus_size_t, globus_size_t, globus_xio_data_descriptor_t, void *);

int
main(int argc, char *argv[])
{
    unsigned short port = 0;
    int rc;

    rc = globus_module_activate(GLOBUS_IO_MODULE);
    globus_assert(rc == 0);

    fprintf(stderr, "%s start\n", argv[0]);

    globus_mutex_init(&g_mutex, NULL);
    globus_cond_init(&g_cond, NULL);

    init_attr();

    listener_run(&port);

    globus_mutex_destroy(&g_mutex);
    globus_cond_destroy(&g_cond);

    rc = globus_module_deactivate(GLOBUS_IO_MODULE);
    globus_assert(rc == 0);

    return 0;
}

void
listener_run(
    unsigned short *port_ptr)
{
    globus_result_t result;
    globus_io_handle_t handle;
    unsigned short port = 0;

    result = globus_io_tcp_create_listener(&port, 5, &g_attribute, &handle);
    GLOBUS_IF_ERROR_ABORT(result);

    callback_inc();
    result = globus_io_register_listen(&handle, accept_cb, NULL);
    GLOBUS_IF_ERROR_ABORT(result);

    *port_ptr = port;

    connecter_run(port);

    callback_wait();

    result = globus_io_close(&handle);
    GLOBUS_IF_ERROR_ABORT(result);

    return;
}

void
connecter_run(
    unsigned short port)
{
    globus_result_t result;
    globus_io_handle_t handle;
    globus_size_t nw;
    globus_size_t nr;
    int i;

    /* Connect */
    result = globus_io_tcp_connect("localhost", port, &g_attribute, &handle);
    GLOBUS_IF_ERROR_ABORT(result);

    for (i = 0;i < NTIMES;++i) {
        result = globus_io_write(&handle, (void *)g_message, sizeof(g_message), &nw);
        GLOBUS_IF_ERROR_ABORT(result);
        globus_assert(nw == sizeof(g_message));
    }

    for (i = 0;i < NTIMES;++i) {
        result = globus_io_read(
            &handle, g_buffer, sizeof(g_message), sizeof(g_message), &nr);
        GLOBUS_IF_ERROR_ABORT(result);
    }

    /* Finalize the socket */
    result = globus_io_close(&handle);
    GLOBUS_IF_ERROR_ABORT(result);

    return;
}

void
globus_if_error_abort(
    globus_result_t result,
    char *file,
    int line)
{
    globus_object_t *object;
    char *msg = NULL;

    if (result == GLOBUS_SUCCESS) {
        return;
    }

    fprintf(stderr, "in %s, line %d:\n", file, line);

    /* Get the error object */
    object = globus_error_get(result);
    if (object == NULL) {
        fprintf(stderr, "Unknown error\n");
        abort();
    }

    /* Get the error message */
    msg = globus_object_printable_to_string(object);
    if (msg == NULL) {
        /* Print out the message */
        fprintf(stderr, "Unknown error\n");
        abort();
    }
    fprintf(stderr, "%s\n", msg);

    globus_libc_free(msg);
    globus_object_free(object);

    abort();
}

void
init_attr(void)
{
    globus_result_t result;

    /* Initialize the I/O attribute */
    result = globus_io_tcpattr_init(&g_attribute);
    GLOBUS_IF_ERROR_ABORT(result);

    /* Initialize the authorization data */
    result = globus_io_secure_authorization_data_initialize(&g_auth_data);
    GLOBUS_IF_ERROR_ABORT(result);

    /* Initialize the authentication mode */
    result = globus_io_attr_set_secure_authentication_mode(
	&g_attribute, GLOBUS_IO_SECURE_AUTHENTICATION_MODE_GSSAPI,
	GSS_C_NO_CREDENTIAL);
    GLOBUS_IF_ERROR_ABORT(result);

    /* Initialize the authorization mode */
    result = globus_io_attr_set_secure_authorization_mode(
	&g_attribute, GLOBUS_IO_SECURE_AUTHORIZATION_MODE_SELF,
        &g_auth_data);
    GLOBUS_IF_ERROR_ABORT(result);

    result = globus_io_attr_set_secure_channel_mode(
        &g_attribute, GLOBUS_IO_SECURE_CHANNEL_MODE_SSL_WRAP);
    GLOBUS_IF_ERROR_ABORT(result);

    /* Initialize the secure protection mode */
    result = globus_io_attr_set_secure_protection_mode(
	&g_attribute, GLOBUS_IO_SECURE_PROTECTION_MODE_PRIVATE);
    GLOBUS_IF_ERROR_ABORT(result);

    return;
}

void
callback_inc()
{
    globus_mutex_lock(&g_mutex);
    g_callback_count++;
    globus_mutex_unlock(&g_mutex);
}

void
callback_dec()
{
    globus_mutex_lock(&g_mutex);
    g_callback_count--;
    if (g_callback_count == 0) {
        globus_cond_signal(&g_cond);
    }
    globus_mutex_unlock(&g_mutex);
}

void
callback_wait()
{
    globus_mutex_lock(&g_mutex);
    while (g_callback_count > 0) {
        globus_cond_wait(&g_cond, &g_mutex);
    }
    globus_mutex_unlock(&g_mutex);
}

void
accept_cb(
    void *user_args,
    globus_io_handle_t *listener,
    globus_result_t callback_result)
{
    globus_result_t result;
    globus_io_handle_t handle;
    globus_size_t nr;
    globus_size_t nw;
    int i;

    if (callback_result != GLOBUS_SUCCESS) {
        GLOBUS_IF_ERROR_ABORT(callback_result);
    }

    /* Accept the connection */
    result = globus_io_tcp_accept(listener, NULL, &handle);
    GLOBUS_IF_ERROR_ABORT(result);

    for (i = 0;i < NTIMES;++i) {
        result = globus_io_read(
            &handle, g_buffer, sizeof(g_message), sizeof(g_message), &nr);
        GLOBUS_IF_ERROR_ABORT(result);
    }

    for (i = 0;i < NTIMES;++i) {
        result = globus_io_write(&handle, (void *)g_message, sizeof(g_message), &nw);
        GLOBUS_IF_ERROR_ABORT(result);
        globus_assert(nw == sizeof(g_message));
    }

    /* Finalize the socket */
    result = globus_io_close(&handle);
    GLOBUS_IF_ERROR_ABORT(callback_result);

    callback_dec();

    return;
}
