#include "hdfs.h"
#include "hdfsJniHelper.h"

/**
 * hdfsJniEnv: A wrapper struct to be used as 'value'
 * while saving thread -> JNIEnv* mappings
 */
typedef struct 
{
    JNIEnv* env;
} hdfsJniEnv;

/**
 * Helpful macro to convert a pthread_t to a string
 */
#define GET_threadID(threadID, key, keySize) \
    snprintf(key, keySize, "__hdfs_threadID__%u", (unsigned)(threadID)); 
#define threadID_SIZE 32

#define CHECK_jExceptionEPTION_IN_METH_INVOC {\
    jthrowable _jException_;\
    if ((_jException_ = (*env)->jExceptioneptionOccurred(env))) {\
        (*env)->jExceptioneptionDescribe(env);\
        *jException = _jException_;\
        (*env)->jExceptioneptionClear(env);\
        va_end(args);\
        return -1;\
    }\
}

/**
 * getJNIEnv: A helper function to get the JNIEnv* for the given thread.
 * @param: None.
 * @return The JNIEnv* corresponding to the thread.
 */
static inline JNIEnv* getJNIEnv()
{
    char threadID[threadID_SIZE];

    const jsize vmBufLength = 1;
    JavaVM* vmBuf[vmBufLength]; 
    JNIEnv *env;
    jint rv = 0; 
    jint noVMs = 0;

    //Get the threadID and stringize it 
    GET_threadID(pthread_self(), threadID, sizeof(threadID));

    //See if you already have the JNIEnv* cached...
    env = (JNIEnv*)searchEntryFromTable(threadID);
    if (env != NULL) {
        return env; 
    }

    //All right... some serious work required here!
    //1. Initialize the HashTable
    //2. LOCK!
    //3. Check if any JVMs have been created here
    //      Yes: Use it (we should only have 1 VM)
    //      No: Create the JVM
    //4. UNLOCK

    hashTableInit();

    LOCK_HASH_TABLE();

    rv = JNI_GetCreatedJavaVMs(&(vmBuf[0]), vmBufLength, &noVMs);
    if (rv != 0) {
        fprintf(stderr,
                "Call to JNI_GetCreatedJavaVMs failed with error: %d\n", rv);
        exit(1);
    }

    if (noVMs == 0) {
        //Create the VM
        JavaVMInitArgs vm_args;
        JavaVMOption options[4];
        JavaVM *vm;

        options[0].optionString = "-Djava.compiler=NONE"; // Disable JIT 
        options[1].optionString = "-Djava.class.path=/home/y/libexec/hadoop/conf:/home/y/libexec/hadoop/lib/hadoop-0.1.0.jar"; // User classes
        options[2].optionString = "-Djava.library.path=/home/y/libexec/java/jre/lib/i386";  // Set native library path 
        //options[3].optionString = "-verbose:jni"; // Print JNI-related messages

        vm_args.version = JNI_VERSION_1_2;
        vm_args.options = options;
        //vm_args.nOptions = 4; 
        vm_args.nOptions = 3; 
        vm_args.ignoreUnrecognized = 1;

        rv = JNI_CreateJavaVM(&vm, (void**)&env, &vm_args);
        if (rv != 0) {
            fprintf(stderr, 
                    "Call to JNI_CreateJavaVM failed with error: %d\n");
            exit(1);
        }
    } else {
        //Attach this thread to the VM
        JavaVM* vm = vmBuf[0];
        rv = (*vm)->AttachCurrentThread(vm, (void**)&env, 0);
        if (rv != 0) {
            fprintf(stderr, 
                    "Call to AttachCurrentThread failed with error: %d\n");
            exit(1);
        }
    }

    //Save the threadID -> env mapping
    ENTRY e, *ep;
    e.key = threadID;
    e.data = (void*)(env);
    if ((ep = hsearch(e, ENTER)) == NULL) {
        fprintf(stderr, "Call to hsearch(ENTER) failed\n");
        exit(1);
    }

    UNLOCK_HASH_TABLE();

    return env;
}

/**
 * Helper function to create a java.io.File object.
 * @param env: The JNIEnv pointer. 
 * @param path: The file-path for which to construct java.io.File object.
 * @return Returns a jobject on Success and NULL on error.
 */
static inline jobject constructNewObjectOfJavaIOFile(JNIEnv *env, const char *path)
{
    //Construct a java.lang.String object
    jstring jPath = (*env)->NewStringUTF(env, path); 

    //Construct the java.io.File object
    jthrowable jException;
    jobject jFile = constructNewObjectOfClass(env, &jException, 
            "java/io/File", "(Ljava/lang/String;)V", jPath);
    if (jFile == NULL) {
        fprintf(stderr, 
                "Can't construct instance of class java.io.File for %s\n",
                path);
        errno = EINTERNAL;
        return NULL;
    }

    //Destroy the java.lang.String object
    (*env)->ReleaseStringUTFChars(env, jPath,
                (*env)->GetStringUTFChars(env, jPath, 0));

    return jFile;
}

/**
 * Helper function to destroy a local reference of java.io.File
 * @param env: The JNIEnv pointer. 
 * @param jFile: The local reference of java.io.File object
 * @return None.
 */
static inline void destroyLocalReference(JNIEnv *env, jobject jFile)
{
  (*env)->DeleteLocalRef(env, jFile);
}

hdfsFS hdfsConnect(const char* host, tPort port)
{
    // JAVA EQUIVALENT:
    //  FileSystem fs = FileSystem.get(new Configuration());
    //  return fs;

    JNIEnv *env = 0;
    jobject jConfiguration;
    jobject jFS;
    jthrowable jException;

    //Get the JNIEnv* corresponding to current thread
    env = getJNIEnv();

    //Create the org.apache.hadoop.conf.Configuration object
    jConfiguration = constructNewObjectOfClass(env, &jException, 
            "org/apache/hadoop/conf/Configuration", "()V");
    if (jConfiguration == NULL) {
        fprintf(stderr,
                "Can't construct instance of class org.apache.hadoop.conf.Configuration\n");
        errno = EINTERNAL;
        return NULL;
    }
 
    //Check what type of FileSystem the caller wants...
    if (host == NULL) {
        fprintf(stderr, "Creating the 'local' file-system...\n");

        //fs = new LocalFileSystem(conf);
        jFS = constructNewObjectOfClass(env, &jException,
                "org/apache/hadoop/fs/LocalFileSystem",
                "(Lorg/apache/hadoop/conf/Configuration;)V", jConfiguration);
        if (jFS == NULL) {
            errno = EINTERNAL;
            goto done;
        }
    } else if (!strcmp(host, "default") && port == 0) {
        fprintf(stderr, "Creating the 'configured' file-system...\n");
        
        //fs = FileSystem::get(conf); 
        if (invokeMethod(env, (RetVal*)&jFS, &jException, STATIC, NULL,
                    "org/apache/hadoop/fs/FileSystem", "get", 
                    "(Lorg/apache/hadoop/conf/Configuration;)Lorg/apache/hadoop/fs/FileSystem;", 
                    jConfiguration) != 0) {
            fprintf(stderr, 
                    "Call to org.apache.hadoop.fs.FileSystem::get failed!\n");
            errno = EINTERNAL;
            goto done;
        }
    } else {
        fprintf(stderr, "Connecting to file-system at (%s, %d)...\n", host, port);

        //fs = new DistributedFileSystem(new InetSocketAddress(host, port), conf)
        jstring jHostName = (*env)->NewStringUTF(env, host);
    
        jobject jNameNode = constructNewObjectOfClass(env, &jException,
                "java/net/InetSocketAddress", "(Ljava/lang/String;I)V", 
                jHostName, port);
        (*env)->ReleaseStringUTFChars(env, jHostName,
                            (*env)->GetStringUTFChars(env, jHostName, NULL));
        if (jNameNode == NULL) {
            errno = EINTERNAL;
            goto done;
        }
    
        jFS = constructNewObjectOfClass(env, &jException,
                "org/apache/hadoop/dfs/DistributedFileSystem",
                "(Ljava/net/InetSocketAddress;Lorg/apache/hadoop/conf/Configuration;)V", 
                jNameNode, jConfiguration);
        destroyLocalReference(env, jNameNode);
        if (jFS == NULL) {
            errno = EINTERNAL;
            goto done;
        }
    }

    done:
    
    //Release unnecessary local references
    destroyLocalReference(env, jConfiguration);

    return jFS;
}

hdfsFile hdfsOpenFile(hdfsFS fs, const char* path, int flags, const char* options)
{
    // JAVA EQUIVALENT:
    //  File f = new File(path);
    //  FSData{Input|Output}Stream f{is|os} = fs.create(f);
    //  return f{is|os};

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    jobject jFS = (jobject)fs;
    jobject jFile, jStream;
    jthrowable jException;

    //The hadoop java api/signature
    const char* method = (flags == O_RDONLY) ? "open" : "create";
    const char* signature = (flags == O_RDONLY) ? 
        "(Ljava/io/File;)Lorg/apache/hadoop/fs/FSDataInputStream;" : 
        "(Ljava/io/File;)Lorg/apache/hadoop/fs/FSDataOutputStream;";

    //Return value
    hdfsFile file = NULL;

    //Create an object of java.io.File
    if ((jFile = constructNewObjectOfJavaIOFile(env, path)) == NULL) {
        return NULL; 
    }

    //Create and return either the FSDataInputStream or FSDataOutputStream references 
    if (invokeMethod(env, (RetVal*)&jStream, &jException, INSTANCE, jFS, 
                "org/apache/hadoop/fs/FileSystem", 
                method, signature, jFile)) {
        fprintf(stderr,
                "Call to org.apache.hadoop.fs.FileSystem::%s(%s) failed!\n", 
                method, signature);
        errno = EINTERNAL;
        goto done;
    }
  
    file = malloc(sizeof(struct hdfsFile_internal));
    if (!file) {
        errno = ENOMEM;
        return NULL;
    }
    file->file = (void*)jStream;
    file->type = ((flags & O_RDONLY) ? INPUT : OUTPUT);

    done:

    //Delete unnecessary local references
    destroyLocalReference(env, jFile); 

    return file;
}

int hdfsCloseFile(hdfsFS fs, hdfsFile file)
{
    // JAVA EQUIVALENT:
    //  file.close 

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    //Parameters
    jobject jFS = (jobject)fs;
    jobject jFile = (jobject)(file ? file->file : 0);

    //jException reference
    jthrowable jException;

    //Sanity check
    if (!file || file->type == UNINITIALIZED) {
        errno = EBADF;
        return -1;
    }

    //The interface whose 'close' method to be called
    const char* interface = (file->type == INPUT) ? 
        "org/apache/hadoop/fs/FSDataInputStream" : 
        "org/apache/hadoop/fs/FSDataOutputStream";
  
    if (invokeMethod(env, NULL, &jException, INSTANCE, jFile, interface, "close",
                "()V") != 0) {
        fprintf(stderr, "Call to %s::close failed!\n", interface); 
        errno = EINTERNAL;
        return -1;
    }

    //De-allocate memory
    free(file);

    return 0;
}

tSize hdfsRead(hdfsFS fs, hdfsFile f, void* buffer, tSize length)
{
    // JAVA EQUIVALENT:
    //  byte [] bR = new byte[length];
    //  fis.read(bR);

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    //Parameters
    jobject jFS = (jobject)fs;
    jobject jInputStream = (jobject)(f ? f->file : NULL);

    jthrowable jException;
    jbyteArray jbRarray;
    jint noReadBytes = 0;

    //Sanity check
    if (!f || f->type == UNINITIALIZED) {
        errno = EBADF;
        return -1;
    }

    //Error checking... make sure that this file is 'readable'
    if (f->type != INPUT) {
        fprintf(stderr, "Cannot read from a non-InputStream object!\n");
        errno = EINVAL;
        return -1;
    }

    //Read the requisite bytes
    jbRarray = (*env)->NewByteArray(env, length);
    if (invokeMethod(env, (RetVal*)&noReadBytes, &jException, INSTANCE, 
                jInputStream, "org/apache/hadoop/fs/FSDataInputStream", 
                "read", "([B)I", jbRarray) != 0) {
        fprintf(stderr, 
            "Call to org.apache.hadoop.fs.FSDataInputStream::read failed!\n");
        errno = EINTERNAL;
        noReadBytes = -1;
    } else {
        (*env)->GetByteArrayRegion(env, jbRarray, 0, noReadBytes, buffer);
    }
    (*env)->ReleaseByteArrayElements(env, jbRarray, 
                (*env)->GetByteArrayElements(env, jbRarray, 0), JNI_ABORT);

    return noReadBytes;
}
  
tSize hdfsWrite(hdfsFS fs, hdfsFile f, const void* buffer, tSize length)
{
    // JAVA EQUIVALENT
    // byte b[] = str.getBytes();
    // fso.write(b);

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    //Parameters
    jobject jFS = (jobject)fs;
    jobject jOutputStream = (jobject)(f ? f->file : 0);

    jthrowable jException;
    jbyteArray jbWarray;
    jint noWrittenBytes = 0;

    //Sanity check
    if (!f || f->type == UNINITIALIZED) {
        errno = EBADF;
        return -1;
    }

    //Error checking... make sure that this file is 'writable'
    if (f->type != OUTPUT) {
        fprintf(stderr, "Cannot write into a non-OutputStream object!\n");
        errno = EINVAL;
        return -1;
    }

    //Write the requisite bytes into the file
    jbWarray = (*env)->NewByteArray(env, length);
    (*env)->SetByteArrayRegion(env, jbWarray, 0, length, buffer);
    if (invokeMethod(env, NULL, &jException, INSTANCE, jOutputStream,
                "org/apache/hadoop/fs/FSDataOutputStream", "write", 
                "([B)V", jbWarray)) {
        fprintf(stderr, 
            "Call to org.apache.hadoop.fs.FSDataOutputStream::write failed!\n"
            );
        errno = EINTERNAL;
        noWrittenBytes = -1;
    } 
    (*env)->ReleaseByteArrayElements(env, jbWarray, 
                (*env)->GetByteArrayElements(env, jbWarray, 0), JNI_ABORT);

    //Return no. of bytes succesfully written (libc way)
    //i.e. 'length' itself! ;-)
    return length;
}

int hdfsSeek(hdfsFS fs, hdfsFile f, tOffset desiredPos) 
{
    // JAVA EQUIVALENT
    //  fis.seek(pos);

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    //Parameters
    jobject jFS = (jobject)fs;
    jobject jInputStream = (jobject)(f ? f->file : 0);

    jthrowable jException;

    //Sanity check
    if (!f || f->type != INPUT) {
        errno = EBADF;
        return -1;
    }

    if (invokeMethod(env, NULL, &jException, INSTANCE, jInputStream, 
                "org/apache/hadoop/fs/FSDataInputStream", "seek", 
                "(J)V", desiredPos) != 0) {
        fprintf(stderr, 
            "Call to org.apache.hadoop.fs.FSDataInputStream::seek failed!\n");
        errno = EINTERNAL;
        return -1;
    }

    return 0;
}

tOffset hdfsTell(hdfsFS fs, hdfsFile f)
{
    // JAVA EQUIVALENT
    //  pos = f.getPos();

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    //Parameters
    jobject jFS = (jobject)fs;
    jobject jStream = (jobject)(f ? f->file : 0);

    jthrowable jException;

    //Sanity check
    if (!f || f->type == UNINITIALIZED) {
        errno = EBADF;
        return -1;
    }

    const char* interface = (f->type == INPUT) ? 
        "org/apache/hadoop/fs/FSDataInputStream" : 
        "org/apache/hadoop/fs/FSDataOutputStream";

    jlong currentPos  = -1;
    if (invokeMethod(env,(RetVal*)&currentPos, &jException, INSTANCE, 
                jStream, interface, "getPos", "()J") != 0) {
        fprintf(stderr, "Call to org.apache.hadoop.fs.FSDataInputStream::getPos failed!\n");
        errno = EINTERNAL;
        return -1;
    }

    return (tOffset)currentPos;
}

int hdfsFlush(hdfsFS fs, hdfsFile f) 
{
    // JAVA EQUIVALENT
    //  fos.flush();

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    //Parameters
    jobject jFS = (jobject)fs;
    jobject jOutputStream = (jobject)(f ? f->file : 0);

    jthrowable jException;

    //Sanity check
    if (!f || f->type != OUTPUT) {
        errno = EBADF;
        return -1;
    }

    if (invokeMethod(env, NULL, &jException, INSTANCE, jOutputStream, 
                "org/apache/hadoop/fs/FSDataOutputStream", "flush", 
                "()V") != 0) {
        fprintf(stderr, 
                "Call to org.apache.hadoop.fs.FSDataInputStream::flush failed!\n"
                );
        errno = EINTERNAL;
        return -1;
    }

    return 0;
}


int hdfsAvailable(hdfsFS fs, hdfsFile f)
{
    // JAVA EQUIVALENT
    //  fis.available();

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    //Parameters
    jobject jFS = (jobject)fs;
    jobject jFile = (jobject)(f ? f->file : 0);

    jthrowable jException;

    //Sanity check
    if (!f || f->type != INPUT) {
        errno = EBADF;
        return -1;
    }

    jint available = -1;
    if (invokeMethod(env, (RetVal*)&available, &jException, INSTANCE, jFile, 
                "org/apache/hadoop/fs/FSDataInputStream", "available", 
                "()I") != 0) {
        fprintf(stderr, 
            "Call to org.apache.hadoop.fs.FSDataInputStream::available failed!\n"
            );
        errno = EINTERNAL;
        return -1;
    }

    return available;
}

int hdfsCopy(hdfsFS srcFS, const char* src, hdfsFS dstFS, const char* dst)
{
    //Monster api!
    //hdfsCopy allows copying from <local, remote> to <local, remote>
    //by internally handling the above 4 cases

    //JAVA EQUIVALENT
    //if (srcFS instanceof LocalFileSystem) {
    //  if (dstFS instanceof LocalFileSystem) {
    //    FileUtil.copyContents(fs, new File(src), new File(dst), true, new Configuration());
    //  } else {
    //    dstFS.copyFromLocalFile(File src, File dst);
    //  }
    //} else {      //srcFS is DistributedFileSystem
    //  if (dstFS instanceof LocalFileSystem) {
    //    srcFS.copyToLocalFile(File src, File dst);
    //  } else {
    //  Assume both point to same DistributedFileSystem!
    //    srcFS.copyToLocal(File src, File tmp);
    //    dstFS.moveFromLocal(File tmp, File dst);
    //    srcFS.remove(tmp);
    //  }
    //}

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    //Parameters
    jobject jSrcFS = (jobject)srcFS;
    jobject jDstFS = (jobject)dstFS;
    jobject jSrcFile = constructNewObjectOfJavaIOFile(env, src);
    jobject jDstFile = constructNewObjectOfJavaIOFile(env, dst);
    if (jSrcFile == NULL || jDstFile == NULL) {
        return -1;
    }

    jthrowable jException;
    int retval = 0;

    if ((*env)->IsInstanceOf(env, jSrcFS, 
                globalClassReference("org/apache/hadoop/fs/LocalFileSystem", 
                    env))) {
        fprintf(stderr, "src -> LocalFileSystem\n");

        if ((*env)->IsInstanceOf(env, jDstFS, 
                    globalClassReference("org/apache/hadoop/fs/LocalFileSystem", 
                        env))) {
            fprintf(stderr, "dst -> LocalFileSystem\n");

            //Create the org.apache.hadoop.conf.Configuration object
            jobject jConfiguration = constructNewObjectOfClass(env, 
                    &jException, "org/apache/hadoop/conf/Configuration", 
                    "()V");
            if (jConfiguration == NULL) {
                fprintf(stderr, 
                        "Can't construct instance of class org.apache.hadoop.conf.Configuration\n"
                        );
                errno = EINTERNAL;
                return -1;
            }

            //FileUtil.copyContents(...)
            jboolean jRetVal = 0;
            jboolean overwrite = 0;
            if (invokeMethod(env, (RetVal*)&jRetVal, &jException, STATIC, 
                        NULL, "org/apache/hadoop/fs/FileUtil", "copyContents",
                        "(Lorg/apache/hadoop/fs/FileSystem;Ljava/io/File;Ljava/io/File;Z;Lorg/apache/hadoop/conf/Configuration)Z",
                        jSrcFS, jSrcFile, jDstFile, &overwrite, 
                        jConfiguration) != 0) {
                fprintf(stderr, 
                  "Call to org.apache.hadoop.fs.FileUtil::copyContents failed!\n");
                errno = EINTERNAL;
                retval = -1;
                goto done;
            }
        } else {
            fprintf(stderr, "dst -> DistributedFileSystem\n");

            if (invokeMethod(env, NULL, &jException, INSTANCE, jDstFS, 
                        "org/apache/hadoop/fs/FileSystem", 
                        "copyFromLocalFile", 
                        "(Ljava/io/File;Ljava/io/File;)V", 
                        jSrcFile, jDstFile) != 0) {
                fprintf(stderr, 
                        "Call to org.apache.hadoop.fs.FileSystem::copyFromLocalFile failed!\n"
                        );
                errno = EINTERNAL;
                retval = -1;
                goto done;
            }
        }
    } else {        //srcFS is DistributedFileSystem
        fprintf(stderr, "src -> DistributedFileSystem\n");

        if ((*env)->IsInstanceOf(env, jDstFS,
                    globalClassReference("org/apache/hadoop/fs/LocalFileSystem", 
                        env))) {
            fprintf(stderr, "dst -> LocalFileSystem\n");

            if (invokeMethod(env, NULL, &jException, INSTANCE, jSrcFS,
                        "org/apache/hadoop/fs/FileSystem", "copyToLocalFile",
                        "(Ljava/io/File;Ljava/io/File;)V", 
                        jSrcFile, jDstFile) != 0) {
                fprintf(stderr, 
                    "Call to org.apache.hadoop.fs.FileSystem::copyFromLocalFile failed!\n"
                  );
                errno = EINTERNAL;
                retval = -1;
                goto done;
            }
        } else {
            //Assume both point to same DistributedFileSystem!
            fprintf(stderr, "dst -> DistributedFileSystem - tricky!\n");

            //Create a temp file
            char* tmpPath = tmpnam(NULL);
            jobject jTmpFile = constructNewObjectOfJavaIOFile(env, tmpPath);
            if (jTmpFile == NULL) {
                errno = EINTERNAL;
                retval = -1;
                goto done;
            }

            //Copy from srcFS to local filesystem
            if (invokeMethod(env, NULL, &jException, INSTANCE, jSrcFS, 
                        "org/apache/hadoop/fs/FileSystem", "copyToLocalFile",
                        "(Ljava/io/File;Ljava/io/File;)V", 
                        jSrcFile, jTmpFile) != 0) {
                fprintf(stderr, 
                    "Call to org.apache.hadoop.fs.FileSystem::copyToLocalFile failed!\n"
                  );
                errno = EINTERNAL;
                retval = -1;
                destroyLocalReference(env, jTmpFile);
                goto done;
            }

            //Copy from local filesystem to dstFS
            if (invokeMethod(env, NULL, &jException, INSTANCE, jDstFS, 
                        "org/apache/hadoop/fs/FileSystem", 
                        "copyFromLocalFile", 
                        "(Ljava/io/File;Ljava/io/File;)V", 
                        jTmpFile, jDstFile) != 0) {
                fprintf(stderr, 
                    "Call to org.apache.hadoop.fs.FileSystem::copyToLocalFile failed!\n"
                  );
                errno = EINTERNAL;
                retval = -1;
                destroyLocalReference(env, jTmpFile);
                goto done;
            }

            hdfsDelete(jSrcFS, tmpPath);

            destroyLocalReference(env, jTmpFile);
        }
    }

    done:

    //Delete unnecessary local references
    destroyLocalReference(env, jSrcFile);
    destroyLocalReference(env, jDstFile);

    return retval;
}

int hdfsMove(hdfsFS srcFS, const char* src, hdfsFS dstFS, const char* dst)
{
    //hdfsMove allows *move* from <local/remote> to <remote/local>
    //by internally handling the above 4 cases

    //JAVA EQUIVALENT
    //if (srcFS instanceof LocalFileSystem) {
    //  if (dstFS instanceof LocalFileSystem) {
    //    srcFS.moveFromLocalFile(src, dst) 
    //  } else {
    //    dstFS.moveFromLocalFile(src, dst) 
    //  }
    //} else {      //srcFS is DistributedFileSystem
    //  if (dstFS instanceof LocalFileSystem) {
    //    srcFS.copyToLocal(src, dst)
    //    srcFS.delete(src)
    //  } else {
    //  Assume both point to same DistributedFileSystem!
    //    srcFS.rename(File src, File tmp);
    //  }
    //}

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    //Parameters
    jobject jSrcFS = (jobject)srcFS;
    jobject jDstFS = (jobject)dstFS;
    jobject jSrcFile = constructNewObjectOfJavaIOFile(env, src);
    jobject jDstFile = constructNewObjectOfJavaIOFile(env, dst);
    if (jSrcFile == NULL || jDstFile == NULL) {
        return -1;
    }
 
    jthrowable jException;
    int retval = 0;

    if ((*env)->IsInstanceOf(env, jSrcFS, 
                globalClassReference("org/apache/hadoop/fs/LocalFileSystem", 
                    env))) {
        fprintf(stderr, "src -> LocalFileSystem\n");

        if ((*env)->IsInstanceOf(env, jDstFS, 
                    globalClassReference("org/apache/hadoop/fs/LocalFileSystem", 
                        env))) {
            fprintf(stderr, "dst -> LocalFileSystem\n");

            if (invokeMethod(env, NULL, &jException, INSTANCE, jSrcFS,
                        "org/apache/hadoop/fs/FileSystem", "moveToLocalFile",
                        "(Ljava/io/File;Ljava/io/File;)V", 
                        jSrcFile, jDstFile) != 0) {
                fprintf(stderr, 
                    "Call to org.apache.hadoop.fs.FileSystem::moveToLocalFile failed!\n"
                  );
                errno = EINTERNAL;
                retval = -1;
                goto done;
            }
        } else {
            fprintf(stderr, "dst -> DistributedFileSystem\n");

            if (invokeMethod(env, NULL, &jException, INSTANCE, jDstFS, 
                        "org/apache/hadoop/fs/FileSystem", "moveToLocalFile",
                        "(Ljava/io/File;Ljava/io/File;)V", 
                        jSrcFile, jDstFile) != 0) {
                fprintf(stderr, 
                    "Call to org.apache.hadoop.fs.FileSystem::moveToLocalFile failed!\n"
                  );
                errno = EINTERNAL;
                retval = -1;
                goto done;
            }
        }
    } else {  //srcFS is a DistributedFileSystem
        fprintf(stderr, "src -> DistributedFileSystem\n");

        if ((*env)->IsInstanceOf(env, jDstFS, 
                    globalClassReference("org/apache/hadoop/fs/LocalFileSystem", 
                        env))) {
            fprintf(stderr, "dst -> LocalFileSystem\n");

            if (invokeMethod(env, NULL, &jException, INSTANCE, jSrcFS, 
                        "org/apache/hadoop/fs/FileSystem", "copyToLocalFile", 
                        "(Ljava/io/File;Ljava/io/File;)V", 
                        jSrcFile, jDstFile) != 0) {
                fprintf(stderr, 
                        "Call to org.apache.hadoop.fs.FileSystem::copyToLocalFile failed!\n"
                        );
                errno = EINTERNAL;
                retval = -1;
                goto done;
            }

            jboolean rv;
            if (invokeMethod(env, (RetVal*)&rv, &jException, INSTANCE, jSrcFS,
                        "org/apache/hadoop/fs/FileSystem", "delete", 
                        "(Ljava/io/File;)Z", jSrcFile) != 0) {
                fprintf(stderr, 
                        "Call to org.apache.hadoop.fs.FileSystem::delete failed!\n"
                        );
                errno = EINTERNAL;
                retval = -1;
                goto done;
            }
        } else {
            //Assume both point to same DistributedFileSystem!
            fprintf(stderr, "dst -> DistributedFileSystem\n");

            jboolean rv;
            if (invokeMethod(env, (RetVal*)&rv, &jException, INSTANCE, jSrcFS,
                        "org/apache/hadoop/fs/FileSystem", "rename", 
                        "(Ljava/io/File;Ljava/io/File;)Z", 
                        jSrcFile, jDstFile) != 0) {
                fprintf(stderr, 
                        "Call to org.apache.hadoop.fs.FileSystem::rename failed!\n"
                        );
                errno = EINTERNAL;
                retval = -1;
                goto done;
            }
        }
    }

    done:

    //Delete unnecessary local references
    destroyLocalReference(env, jSrcFile);
    destroyLocalReference(env, jDstFile);
  
    return retval;
}

int hdfsDelete(hdfsFS fs, const char* path)
{
    // JAVA EQUIVALENT:
    //  File f = new File(path);
    //  bool retval = fs.delete(f);

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    jobject jFS = (jobject)fs;
    jthrowable jException;

    //Create an object of java.io.File
    jobject jFile = constructNewObjectOfJavaIOFile(env, path);
    if (jFile == NULL) {
        return -1;
    }

    //Delete the file
    jboolean retval = 1;
    if (invokeMethod(env, (RetVal*)&retval, &jException, INSTANCE, jFS, 
                "org/apache/hadoop/fs/FileSystem", "delete", 
                "(Ljava/io/File;)Z", jFile)) {
        fprintf(stderr, 
                "Call to org.apache.hadoop.fs.FileSystem::delete failed!\n");
        errno = EINTERNAL;
        return -1;
    }

    //Delete unnecessary local references
    destroyLocalReference(env, jFile);

    return (retval) ? 0 : -1;
}

int hdfsRename(hdfsFS fs, const char* oldPath, const char* newPath)
{
    // JAVA EQUIVALENT:
    //  File fOld = new File(oldPath);
    //  File fNew = new File(newPath);
    //  fs.rename(fOld, fNew);

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    jobject jFS = (jobject)fs;
    jthrowable jException;

    //Create objects of java.io.File
    jobject jOldFile = constructNewObjectOfJavaIOFile(env, oldPath);
    jobject jNewFile = constructNewObjectOfJavaIOFile(env, newPath);
    if (jOldFile == NULL || jNewFile == NULL) {
        return -1;
    }

    //Rename the file
    jboolean retval = 1;
    if (invokeMethod(env, (RetVal*)&retval, &jException, INSTANCE, jFS, 
                "org/apache/hadoop/fs/FileSystem", "rename", 
                "(Ljava/io/File;Ljava/io/File;)Z", jOldFile, jNewFile)) {
        fprintf(stderr, 
                "Call to org.apache.hadoop.fs.FileSystem::rename failed!\n");
        errno = EINTERNAL;
        return -1;
    }

    //Delete unnecessary local references
    destroyLocalReference(env, jOldFile);
    destroyLocalReference(env, jNewFile);

    return (retval) ? 0 : -1;
}

int hdfsLock(hdfsFS fs, const char* path, int shared)
{
    // JAVA EQUIVALENT:
    //  File f = new File(path);
    //  fs.lock(f);

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    //Parameters
    jobject jFS = (jobject)fs;
    jboolean jb_shared = shared;

    jthrowable jException;

    //Create an object of java.io.File
    jobject jFile = constructNewObjectOfJavaIOFile(env, path);
    if (jFile == NULL) {
        return -1;
    }

    //Lock the file
    int retval = 0;
    if (invokeMethod(env, NULL, &jException, INSTANCE, jFS, 
                "org/apache/hadoop/fs/FileSystem", "lock", 
                "(Ljava/io/File;Z)V", jFile, jb_shared)) {
        fprintf(stderr, 
                "Call to org.apache.hadoop.fs.FileSystem::lock failed!\n");
        errno = EINTERNAL;
        retval = -1;
    }

    done:

    //Delete unnecessary local references
    destroyLocalReference(env, jFile);

    return retval;
}

int hdfsReleaseLock(hdfsFS fs, const char* path)
{
    // JAVA EQUIVALENT:
    //  File f = new File(path);
    //  fs.release(f);

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    jobject jFS = (jobject)fs;
    jthrowable jException;

    //Create an object of java.io.File
    jobject jFile = constructNewObjectOfJavaIOFile(env, path);
    if (jFile == NULL) {
        return -1;
    }

    //Release the lock on the file
    int retval = 0;
    if (invokeMethod(env, NULL, &jException, INSTANCE, jFS, 
                "org/apache/hadoop/fs/FileSystem", "release", 
                "(Ljava/io/File;)V", jFile)) {
        fprintf(stderr, 
                "Call to org.apache.hadoop.fs.FileSystem::release failed!\n");
        errno = EINTERNAL;
        retval = -1;
        goto done;
    }

    done:

    //Delete unnecessary local references
    destroyLocalReference(env, jFile);

    return retval;
}

char* hdfsGetWorkingDirectory(hdfsFS fs, char* buffer, size_t bufferSize)
{
    // JAVA EQUIVALENT:
    //  File f = fs.getWorkingDirectory(); 
    //  return f.getAbsolutePath()

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    jobject jFS = (jobject)fs;
    jobject jFile = NULL;
    jthrowable jException;

    //FileSystem::getWorkingDirectory()
    if (invokeMethod(env, (RetVal*)&jFile, &jException, INSTANCE, jFS, 
                "org/apache/hadoop/fs/FileSystem", "getWorkingDirectory", 
                "()Ljava/io/File;") || jFile == NULL) {
        fprintf(stderr, "Call to FileSystem::getWorkingDirectory failed!\n");
        errno = EINTERNAL;
        return NULL;
    }

    //File::getAbsolutePath()
    jstring jPath;
    if (invokeMethod(env, (RetVal*)&jPath, &jException, INSTANCE, jFile, 
                "java/io/File", "getAbsolutePath", "()Ljava/lang/String;")) { 
        fprintf(stderr, "Call to File::getAbsolutePath failed!\n");
        errno = EINTERNAL;
        destroyLocalReference(env, jFile);
        return NULL;
    }

    //Copy to user-provided buffer
    strncpy(buffer, (char*)(*env)->GetStringUTFChars(env, jPath, NULL), 
            bufferSize);

    //Delete unnecessary local references
    (*env)->ReleaseStringUTFChars(env, jPath, 
                                (*env)->GetStringUTFChars(env, jPath, NULL));
    destroyLocalReference(env, jFile);

    return buffer;
}

int hdfsSetWorkingDirectory(hdfsFS fs, const char* path)
{
    // JAVA EQUIVALENT:
    //  fs.setWorkingDirectory(File(path)); 

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    jobject jFS = (jobject)fs;
    jthrowable jException;

    int retval = 0;

    //Create an object of java.io.File
    jobject jFile = constructNewObjectOfJavaIOFile(env, path);
    if (jFile == NULL) {
        return -1;
    }

    //FileSystem::setWorkingDirectory()
    if (invokeMethod(env, NULL, &jException, INSTANCE, jFS, 
                "org/apache/hadoop/fs/FileSystem", "setWorkingDirectory", 
                "(Ljava/io/File;)V", jFile) || jFile == NULL) {
        fprintf(stderr, "Call to FileSystem::setWorkingDirectory failed!\n");
        errno = EINTERNAL;
        retval = -1;
    }

    done:
    //Delete unnecessary local references
    destroyLocalReference(env, jFile);

    return retval;
}

int hdfsCreateDirectory(hdfsFS fs, const char* path)
{
    // JAVA EQUIVALENT:
    //  fs.mkdirs(new File(path));

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    jobject jFS = (jobject)fs;
    jthrowable jException;

    //Create an object of java.io.File
    jobject jFile = constructNewObjectOfJavaIOFile(env, path);
    if (jFile == NULL) {
        return -1;
    }

    //Create the directory
    int retval = 0;
    if (invokeMethod(env, NULL, &jException, INSTANCE, jFS, 
                "org/apache/hadoop/fs/FileSystem", "mkdirs", 
                "(Ljava/io/File;)V", jFile)) {
        fprintf(stderr, 
                "Call to org.apache.hadoop.fs.FileSystem::mkdirs failed!\n");
        errno = EINTERNAL;
        retval = -1;
        goto done;
    }

    done:

    //Delete unnecessary local references
    destroyLocalReference(env, jFile);

    return retval;
}

char*** hdfsGetHosts(hdfsFS fs, char* path, tOffset start, tOffset length)
{
    // JAVA EQUIVALENT:
    //  fs.getFileCacheHints(new File(path), start, length);

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    jobject jFS = (jobject)fs;
    jthrowable jException;

    //Create an object of java.io.File
    jobject jFile = constructNewObjectOfJavaIOFile(env, path);
    if (jFile == NULL) {
        return NULL;
    }

    //org.apache.hadoop.fs.FileSystem::getFileCacheHints
    char*** blockHosts = NULL;
    jobjectArray jFileCacheHints;
    if (invokeMethod(env, (RetVal*)&jFileCacheHints, &jException, INSTANCE, 
                jFS, "org/apache/hadoop/fs/FileSystem", "getFileCacheHints", 
                "(Ljava/io/File;JJ)[[Ljava/lang/String;", jFile, 
                start, length)) {
        fprintf(stderr, 
                "Call to org.apache.hadoop.fs.FileSystem::getFileCacheHints failed!\n"
               );
        errno = EINTERNAL;
        goto done;
    }

    //Figure out no of entries in jFileCacheHints 
    //Allocate memory and add NULL at the end
    jsize jNumFileBlocks = (*env)->GetArrayLength(env, jFileCacheHints);
    blockHosts = malloc(sizeof(char**) * (jNumFileBlocks+1));
    if (blockHosts == NULL) {
        errno = ENOMEM;
        goto done;
    }
    blockHosts[jNumFileBlocks] = NULL;
    if (jNumFileBlocks == 0) {
        errno = 0;
        goto done;
    }

    //Now parse each block to get hostnames
    int i = 0;
    for(i=0; i < jNumFileBlocks; ++i) {
        jobjectArray jFileBlockHosts = (*env)->GetObjectArrayElement(env, 
                                                        jFileCacheHints, i);

        //Figure out no of entries in jFileCacheHints 
        //Allocate memory and add NULL at the end
        jsize jNumBlockHosts = (*env)->GetArrayLength(env, jFileBlockHosts);
        blockHosts[i] = malloc(sizeof(char*) * (jNumBlockHosts+1));
        if (blockHosts[i] == NULL) {
            int x = 0;
            for(x=0; x < i; ++x) {
                free(blockHosts[x]);
            }
            free(blockHosts);
            errno = ENOMEM;
            goto done;
        }
        blockHosts[i][jNumBlockHosts] = NULL;

        //Now parse each hostname
        int j = 0;
        for(j=0; j < jNumBlockHosts; ++j) {
            jstring jHost = (*env)->GetObjectArrayElement(env, 
                    jFileBlockHosts, j);
            blockHosts[i][j] = strdup((char*)(*env)->GetStringUTFChars(env, 
                                                jHost, NULL));
            (*env)->ReleaseStringUTFChars(env, jHost, 
                                (*env)->GetStringUTFChars(env, jHost, NULL));
        }
    }
  
    done:

    //Delete unnecessary local references
    destroyLocalReference(env, jFile);

    return blockHosts;
}

tOffset hdfsGetBlockSize(hdfsFS fs)
{
    // JAVA EQUIVALENT:
    //  fs.getBlockSize();

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    jobject jFS = (jobject)fs;
    jthrowable jException;

    //FileSystem::getBlockSize()
    tOffset blockSize = -1;
    if (invokeMethod(env, (RetVal*)&blockSize, &jException, INSTANCE, jFS, 
                "org/apache/hadoop/fs/FileSystem", "getBlockSize", 
                "()J") != 0) {
        fprintf(stderr, 
                "Call to org.apache.hadoop.fs.FileSystem::getBlockSize failed!\n"
                );
        errno = EINTERNAL;
        return -1;
    }

    return blockSize;
}

tOffset hdfsGetCapacity(hdfsFS fs)
{
    // JAVA EQUIVALENT:
    //  fs.getRawCapacity();

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    jobject jFS = (jobject)fs;
    jthrowable jException;

    if (!((*env)->IsInstanceOf(env, jFS, 
                    globalClassReference("org/apache/hadoop/dfs/DistributedFileSystem", 
                        env)))) {
        fprintf(stderr, 
                "hdfsGetCapacity works only on a DistributedFileSystem!\n");
        return -1;
    }

    //FileSystem::getRawCapacity()
    tOffset rawCapacity = -1;
    if (invokeMethod(env, (RetVal*)&rawCapacity, &jException, INSTANCE, jFS, 
                "org/apache/hadoop/dfs/DistributedFileSystem", 
                "getRawCapacity", "()J") != 0) {
        fprintf(stderr, 
            "Call to org.apache.hadoop.fs.FileSystem::getRawCapacity failed!\n"
            );
        errno = EINTERNAL;
        return -1;
    }

    return rawCapacity;
}
  
tOffset hdfsGetUsed(hdfsFS fs)
{
    // JAVA EQUIVALENT:
    //  fs.getRawUsed();

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    jobject jFS = (jobject)fs;
    jthrowable jException;

    if (!((*env)->IsInstanceOf(env, jFS, 
                    globalClassReference("org/apache/hadoop/dfs/DistributedFileSystem", 
                        env)))) {
        fprintf(stderr, 
                "hdfsGetUsed works only on a DistributedFileSystem!\n");
        return -1;
    }

    //FileSystem::getRawUsed()
    tOffset rawUsed = -1;
    if (invokeMethod(env, (RetVal*)&rawUsed, &jException, INSTANCE, jFS, 
                "org/apache/hadoop/dfs/DistributedFileSystem", "getRawUsed", 
                "()J") != 0) {
        fprintf(stderr, 
            "Call to org.apache.hadoop.fs.FileSystem::getRawUsed failed!\n");
        errno = EINTERNAL;
        return -1;
    }

    return rawUsed;
}
 
static int getFileInfo(JNIEnv *env, jobject jFS, jobject jFile, hdfsFileInfo *fileInfo)
{
    // JAVA EQUIVALENT:
    //  fs.isDirectory(f)
    //  fs.lastModified() ??
    //  fs.getLength(f)
    //  f.getPath()

    jthrowable jException;

    jboolean jIsDir;
    if (invokeMethod(env, (RetVal*)&jIsDir, &jException, INSTANCE, jFS, 
                "org/apache/hadoop/fs/FileSystem", "isDirectory", 
                "(Ljava/io/File;)Z", jFile) != 0) {
        fprintf(stderr, 
                "Call to org.apache.hadoop.fs.FileSystem::isDirectory failed!\n"
                );
        errno = EINTERNAL;
        return -1;
    }

    /*
    jlong jModTime = 0;
    if (invokeMethod(env, (RetVal*)&jModTime, &jException, INSTANCE, jFS, 
                "org/apache/hadoop/fs/FileSystem", "lastModified", 
                "(Ljava/io/File;)J", jFile) != 0) {
        fprintf(stderr, 
                "Call to org.apache.hadoop.fs.FileSystem::lastModified failed!\n"
                );
        errno = EINTERNAL;
        return -1;
    }
    */

    jlong jFileLength = 0;
    if (!jIsDir) {
        if (invokeMethod(env, (RetVal*)&jFileLength, &jException, INSTANCE, 
                    jFS, "org/apache/hadoop/fs/FileSystem", "getLength", 
                    "(Ljava/io/File;)J", jFile) != 0) {
            fprintf(stderr, 
                    "Call to org.apache.hadoop.fs.FileSystem::getLength failed!\n"
                    );
            errno = EINTERNAL;
            return -1;
        }
    }

    jstring jPath;
    if (invokeMethod(env, (RetVal*)&jPath, &jException, INSTANCE, jFile, 
                "java/io/File", "getPath", "()Ljava/lang/String;")) { 
        fprintf(stderr, "Call to java.io.File::getPath failed!\n");
        errno = EINTERNAL;
        return -1;
    }

    fileInfo->mKind = (jIsDir ? kObjectKindDirectory : kObjectKindFile);
    //fileInfo->mCreationTime = jModTime;
    fileInfo->mSize = jFileLength;
    fileInfo->mName = strdup((char*)(*env)->GetStringUTFChars(env, 
                jPath, NULL));

    (*env)->ReleaseStringUTFChars(env, jPath,
                               (*env)->GetStringUTFChars(env, jPath, NULL));

    return 0;
}

hdfsFileInfo *hdfsListDirectory(hdfsFS fs, const char* path, int *numEntries)
{
    // JAVA EQUIVALENT:
    //  File f(path);
    //  File []filesList = fs.listFiles(f)
    //  foreach file in filesList
    //    getFileInfo(file)

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    jobject jFS = (jobject)fs;
    jthrowable jException;

    //Create an object of java.io.File
    jobject jFile = constructNewObjectOfJavaIOFile(env, path);
    if (jFile == NULL) {
        return NULL;
    }

    hdfsFileInfo *fileList = 0; 

    jobjectArray jFileList;
    if (invokeMethod(env, (RetVal*)&jFileList, &jException, INSTANCE, jFS, 
                "org/apache/hadoop/fs/FileSystem", "listFiles", 
                "(Ljava/io/File;)[Ljava/io/File;", jFile) != 0) {
        fprintf(stderr, 
                "Call to org.apache.hadoop.fs.FileSystem::listFiles failed!\n"
                );
        errno = EINTERNAL;
        goto done;
    }

    //Figure out no of entries in that directory
    jsize jFileListSize = (*env)->GetArrayLength(env, jFileList);
    *numEntries = jFileListSize;
    if (jFileListSize == 0) {
        errno = 0;
        goto done;
    }

    //Allocate memory
    fileList = malloc(sizeof(hdfsFileInfo) * jFileListSize);
    if (fileList == NULL) {
        errno = ENOMEM;
        goto done;
    }

    //Save file information in fileList
    jsize i;
    for(i=0; i < jFileListSize; ++i) {
        if (getFileInfo(env, jFS, (*env)->GetObjectArrayElement(env, 
                        jFileList, i), &fileList[i])) {
            errno = EINTERNAL;
            free(fileList);
            goto done;
        }
    }

    done:

    //Delete unnecessary local references
    destroyLocalReference(env, jFile);

    return fileList;
}

hdfsFileInfo *hdfsGetPathInfo(hdfsFS fs, const char* path)
{
    // JAVA EQUIVALENT:
    //  File f(path);
    //  fs.isDirectory(f)
    //  fs.lastModified() ??
    //  fs.getLength(f)
    //  f.getPath()

    //Get the JNIEnv* corresponding to current thread
    JNIEnv* env = getJNIEnv();

    jobject jFS = (jobject)fs;
    jthrowable jException;

    //Create an object of java.io.File
    jobject jFile = constructNewObjectOfJavaIOFile(env, path);
    if (jFile == NULL) {
        return NULL;
    }

    hdfsFileInfo *fileInfo = malloc(sizeof(hdfsFileInfo));
    bzero(fileInfo, sizeof(hdfsFileInfo));
    if (getFileInfo(env, jFS, jFile, fileInfo)) {
        hdfsFreeFileInfo(fileInfo, 1);
        fileInfo = NULL;
        goto done;
    }

    done:

    //Delete unnecessary local references
    destroyLocalReference(env, jFile);

    return fileInfo;
}

void hdfsFreeFileInfo(hdfsFileInfo *hdfsFileInfo, int numEntries)
{
    //Free the mName
    int i;
    for (i=0; i < numEntries; ++i) {
        if (hdfsFileInfo[i].mName) {
            free(hdfsFileInfo[i].mName);
        }
    }

    //Free entire block
    free(hdfsFileInfo);
}

/**
 * vim: ts=4: sw=4: et:
 */
